/*
 * Copyright (C) 2002 Stefan Holst
 * Copyright (C) 2025 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA.
 *
 * odk abstracts from xine and other software / hardware interfaces
 */

/*
#define LOG
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <xine.h>

#include "common.h"
#include "../videowin.h"
#include "../actions.h"
#include "../mediamark.h"
#include "../common/config_wrapper.h"

#include "utils.h"
#include "oxine.h"
#include "odk.h"

#undef ODK_COLOR_TEST

#define V_WIDTH 800
#define V_HEIGHT 600

struct  odk_s {
  gGui_t     *gui;
  xine_t     *xine;
  xine_stream_t *stream;

  xitk_t *xitk;

  xine_osd_t *osd;
  int size[2];
  double vscale, hscale;

  int (*event_handler)(void *data, oxine_event_t *ev);
  void *event_handler_data;

  odk_palette_t *palette;

  int unscaled_osd, is_freetype;
};

static uint32_t odk_swap_color (uint32_t _cbcry) {
  /* see <xine/alphablend.h:clut_t. */
  const union { uint32_t v; uint8_t cb_cr_y_foo[4]; } test = { 1 };
  if (test.cb_cr_y_foo[0])
    return _cbcry;
  else
    return (_cbcry >> 24) | ((_cbcry & 0x00ff0000) >> 8) | ((_cbcry & 0x0000ff00) << 8) | (_cbcry << 24);
}

void odk_palette_set (odk_t *odk, odk_palette_t *p) {
  odk->palette = p;
  xine_osd_set_palette (odk->osd, odk->palette->yuv, odk->palette->o);
}

/** generates a gradient on palette out of given values. */

static void _odk_palette_fill_text (odk_palette_t *p, int indx) {
  uint32_t bg, fg, bo;
  int m20, m25, m40, m50, m60, m75, m80;
  uint32_t a1, b1, a2, b2, v;
  /* some weird fixed point math normalized to 1.0 == 256. */
#define odk_m(_m,_s) (_m+((_s*(((_m*_m)>>8)-256))>>8))
  m20 = odk_m (2 * 256 * 20 / 100, p->sharpness) >> 1;
  m25 = odk_m (2 * 256 * 25 / 100, p->sharpness) >> 1;
  m40 = odk_m (2 * 256 * 40 / 100, p->sharpness) >> 1;
  m50 = 256 * 50 / 100;
  m60 = 256 - m40;
  m75 = 256 - m25;
  m80 = 256 - m20;
#ifdef ODK_COLOR_TEST
  printf ("gui.oxine.odk.palette.add_text (%d) = %u %u %u %u %u %u %u.\n", s,
    ((unsigned int)m20 * 100 + 128) >> 8,
    ((unsigned int)m25 * 100 + 128) >> 8,
    ((unsigned int)m40 * 100 + 128) >> 8,
    ((unsigned int)m50 * 100 + 128) >> 8,
    ((unsigned int)m60 * 100 + 128) >> 8,
    ((unsigned int)m75 * 100 + 128) >> 8,
    ((unsigned int)m80 * 100 + 128) >> 8
  );
#endif

  /* not really used by fonts, set to bg. */
  bg = (odk_swap_color (p->yuv[indx]) << 8) + (p->o[indx] & 0x0f);
  fg = (odk_swap_color (p->yuv[indx + 10]) << 8) + (p->o[indx + 10] & 0x0f);
  bo = (odk_swap_color (p->yuv[indx + 6]) << 8) + (p->o[indx + 6] & 0x0f);
  /* background (1) to border (6) transition */
  a1 = (bg >> 8) & 0x00ff00ff; b1 = bg & 0x00ff00ff;
  a2 = (bo >> 8) & 0x00ff00ff; b2 = bo & 0x00ff00ff;
  p->yuv[indx + 1] = odk_swap_color (bg >> 8);
  p->o[indx + 1] = bg & 255;
  v = ((m80 * a1 + m20 * a2) & 0xff00ff00)
    | (((m80 * b1 + m20 * b2) >> 8) & 0x00ff00ff);
  p->yuv[indx + 2] = odk_swap_color (v >> 8);
  p->o[indx + 2] = v & 255;
  v = ((m60 * a1 + m40 * a2) & 0xff00ff00)
    | (((m60 * b1 + m40 * b2) >> 8) & 0x00ff00ff);
  p->yuv[indx + 3] = odk_swap_color (v >> 8);
  p->o[indx + 3] = v & 255;
  v = ((m40 * a1 + m60 * a2) & 0xff00ff00)
    | (((m40 * b1 +m60 * b2) >> 8) & 0x00ff00ff);
  p->yuv[indx + 4] = odk_swap_color (v >> 8);
  p->o[indx + 4] = v & 255;
  v = ((m20 * a1 + m80 * a2) & 0xff00ff00)
    | (((m20 * b1 + m80 * b2) >> 8) & 0x00ff00ff);
  p->yuv[indx + 5] = odk_swap_color (v >> 8);
  p->o[indx + 5] = v & 255;
  /* border (6) to foreground (10) transition */
  a1 = a2; b1 = b2;
  a2 = (fg >> 8) & 0x00ff00ff; b2 = fg & 0x00ff00ff;
  v = ((m75 * a1 + m25 * a2) & 0xff00ff00)
    | (((m75 * b1 + m25 * b2) >> 8) & 0x00ff00ff);
  p->yuv[indx + 7] = odk_swap_color (v >> 8);
  p->o[indx + 7] = v & 255;
  v = ((m50 * a1 + m50 * a2) & 0xff00ff00)
    | (((m50 * b1 + m50 * b2) >> 8) & 0x00ff00ff);
  p->yuv[indx + 8] = odk_swap_color (v >> 8);
  p->o[indx + 8] = v & 255;
  v = ((m25 * a1 + m75 * a2) & 0xff00ff00)
    | (((m25 * b1 + m75 * b2) >> 8) & 0x00ff00ff);
  p->yuv[indx + 9] = odk_swap_color (v >> 8);
  p->o[indx + 9] = v & 255;
}

/*
 * adapt osd to new stream size.
 * This only affects primitives drawn after
 * this call.
 */

int odk_get_size (odk_t *odk, int size[2]) {
  int rs[2];

  if (!odk || !size)
    return 0;
  video_window_get_size (odk->gui->vwin, rs, odk->unscaled_osd ? VWIN_SIZE_OUTPUT : VWIN_SIZE_FRAME);
  if (!((rs[0] ^ size[0]) | (rs[1] ^ size[1])))
    return 0;
  size[0] = rs[0];
  size[1] = rs[1];
  return 1;
}

static void odk_adapt(odk_t *odk) {
  if (odk->osd)
    xine_osd_free (odk->osd);

  lprintf("odk_adapt to: %i, %i\n", odk->size[0], odk->size[1]);
  odk->vscale = odk->size[1] / (double)V_HEIGHT;
  odk->hscale = odk->size[0] / (double)V_WIDTH;

  odk->osd = xine_osd_new (odk->stream, 0, 0, odk->size[0], odk->size[1]);

  if (odk->palette)
    xine_osd_set_palette (odk->osd, odk->palette->yuv, odk->palette->o);
}


uint8_t *odk_get_pixmap(int type) {
  int i;
  uint8_t arrowup[] = {0, 0, 1, 1, 1, 1, 1, 1, 0, 0,
		       0, 0, 1, 1, 1, 1, 1, 1, 0, 0,
		       0, 0, 1, 0, 1, 1, 0, 1, 0, 0,
		       0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
		       0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
		       0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
		       0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
		       0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
		       0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
		       0, 0, 0, 0, 1, 1, 0, 0, 0, 0};
  uint8_t arrowdown[] = {0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
			 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
			 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
			 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
			 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
			 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
			 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
			 0, 0, 1, 0, 1, 1, 0, 1, 0, 0,
			 0, 0, 1, 1, 1, 1, 1, 1, 0, 0,
			 0, 0, 1, 1, 1, 1, 1, 1, 0, 0};

  uint8_t *p = NULL;

  p = ho_newstring(10*10 + 10);

  switch(type) {
  case PIXMAP_SIMPLE_ARROW_UP:
    memcpy(p, arrowup, 100);
    break;
  case PIXMAP_SIMPLE_ARROW_DOWN:
    memcpy(p, arrowdown, 100);
    break;
  }

  /* FIXME: check this stuff "XINE_OSD_TEXT1 + 2" */
  for(i=0;i<100;i++) p[i] = p[i] * XINE_OSD_TEXT1 + 2;

  return p;
}


/*
 * initializes drawable, xine stream and osd
 */

#ifdef ODK_COLOR_TEST
static void odk_ctest (void);
#endif

odk_t *odk_init (gGui_t *gui) {
  odk_t *odk = ho_new_tagged(odk_t, "odk object");

  lprintf("initializing\n");

  odk->gui = gui;
  odk->xitk = gui->xitk;
  odk->palette = NULL;

  odk->xine = gui->xine;
  odk->stream = odk->gui->stream;

  /* test unscaled osd support */
  odk->osd = xine_osd_new(odk->stream, 0, 0, 10, 10);
  odk->unscaled_osd = (xine_osd_get_capabilities (odk->osd) & XINE_OSD_CAP_UNSCALED);
  xine_osd_free(odk->osd);
  odk->osd = NULL;

  video_window_get_size (odk->gui->vwin, odk->size, odk->unscaled_osd ? VWIN_SIZE_OUTPUT : VWIN_SIZE_FRAME);
  odk_adapt(odk);

#ifdef ODK_COLOR_TEST
  odk_ctest ();
#endif
  return odk;
}


/*
 * primitive drawing functions
 */

void odk_draw_bitmap(odk_t *odk, uint8_t *bitmap, int x1, int y1, int width, int height,
  uint8_t *palette_map) {
  /* uint8_t a[] = {0, 0, 0, 255, 255, 255}; */
  if (!(odk->hscale&&odk->vscale)) return;

  /* FIXME: scaling? */
  xine_osd_draw_bitmap(odk->osd, bitmap, x1, y1, width, height, NULL);
}


void odk_draw_line(odk_t *odk, int x1, int y1, int x2, int y2, int color) {
 int px1, py1, px2, py2;

  if (!(odk->hscale&&odk->vscale)) return;

 px1 = (int) ((double)x1*odk->hscale);
 py1 = (int) ((double)y1*odk->vscale);
 px2 = (int) ((double)x2*odk->hscale);
 py2 = (int) ((double)y2*odk->vscale);
 lprintf("drawing line %u %u %u %u col: %u\n", px1, py1, px2, py2, color);

  xine_osd_draw_line(odk->osd,
      px1,
      py1,
      px2,
      py2,
      color);
}


void odk_draw_rect(odk_t *odk, int x1, int y1, int x2, int y2, int filled, int color) {
 int px1, py1, px2, py2;

  if (!(odk->hscale&&odk->vscale)) return;

 px1 = (int) ((double)x1*odk->hscale);
 py1 = (int) ((double)y1*odk->vscale);
 px2 = (int) ((double)x2*odk->hscale);
 py2 = (int) ((double)y2*odk->vscale);
 lprintf("drawing rect %u %u %u %u col: %u\n", px1, py1, px2, py2, color);

  xine_osd_draw_rect(odk->osd,
      px1, py1, px2, py2, color, filled);
}


void odk_draw_text(odk_t *odk, int x, int y, const char *text, int alignment, int color) {
  int px, py, w, h;

  if (!(odk->hscale&&odk->vscale)) return;

  odk_get_text_size(odk, text, &w, &h);

  if (odk->is_freetype) {
    if (alignment & ODK_ALIGN_VCENTER) y += h/2;
    if (alignment & ODK_ALIGN_TOP)  y += h;
  } else {
    if (alignment & ODK_ALIGN_VCENTER) y -= h/2;
    if (alignment & ODK_ALIGN_BOTTOM)  y -= h;
  }

  if (alignment & ODK_ALIGN_CENTER)  x -= w/2;
  if (alignment & ODK_ALIGN_RIGHT)   x -= w;

  px = (int) ((double)x*odk->hscale);
  py = (int) ((double)y*odk->vscale);

  lprintf("drawing text at %u %u (vs: %f hs: %f)\n", px, py, odk->vscale, odk->hscale);

  xine_osd_draw_text(odk->osd, px, py, text, color);
}


/*
 * overall osd control
 */

void odk_show(odk_t *odk) {
  if( odk->unscaled_osd )
    xine_osd_show_unscaled(odk->osd, 0);
  else
    xine_osd_show(odk->osd, 0);
}

void odk_hide(odk_t *odk) {
  xine_osd_hide(odk->osd, 0);
}

void odk_clear(odk_t *odk) {
  xine_osd_clear(odk->osd);
}


/*
 * font stuff
 */

void odk_get_text_size(odk_t *odk, const char *text, int *width, int *height) {
  int w, h;

  if (!(odk->hscale&&odk->vscale)) {
    *width = 0;
    *height = 0;
    return;
  }

  xine_osd_get_text_size(odk->osd, text, &w, &h);
  *width = (int) ((double)w/odk->hscale);
  *height = (int) ((double)h/odk->vscale);
}


void odk_set_font(odk_t *odk, const char *font, int size) {
  int psize;

  psize = (int)((double)size*(odk->hscale+odk->vscale)/2);

  /* smallest text size possible */
  if (psize<16) psize=16;

  lprintf("setting font to %s %u %u\n", font, size, psize);

  if (strchr(font, '.')||strchr(font, '/'))
    odk->is_freetype = 1;
  else
    odk->is_freetype = 0;

  xine_osd_set_font(odk->osd, font, psize);
}


/*
 * event stuff
 */

void odk_set_event_handler(odk_t *odk, int (*cb)(void *data, oxine_event_t *ev), void *data) {
  odk->event_handler = cb;
  odk->event_handler_data = data;
}

int odk_send_event(odk_t *odk, oxine_event_t *event) {
  switch (event->type) {
    case OXINE_EVENT_FORMAT_CHANGED:
      if (odk_get_size (odk, odk->size))
        odk_adapt(odk);
      break;

    case OXINE_EVENT_BUTTON:
    case OXINE_EVENT_MOTION:
      if( !odk->unscaled_osd ) {
        x11_rectangle_t rect;

        rect.x = event->x;
        rect.y = event->y;
        rect.w = 0;
        rect.h = 0;

        if (xine_port_send_gui_data(odk->gui->vo_port,
            XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, (void*)&rect) == -1) {
          return 0;
        }

        event->x = rect.x;
        event->y = rect.y;
      }
      event->x = (int) ((double)event->x/odk->hscale);
      event->y = (int) ((double)event->y/odk->vscale);
      if ((event->x < 0) || (event->x > V_WIDTH)) return 0;
      if ((event->y < 0) || (event->y > V_HEIGHT)) return 0;
  }

  if (odk->event_handler)
    return odk->event_handler(odk->event_handler_data, event);

  return 0;
}

/*
 * destructor
 */

void odk_free(odk_t *odk) {
  lprintf("finalizing\n");

  xine_osd_free(odk->osd);
  ho_free(odk);
}


/*
 * xine control
 */

void odk_enqueue(odk_t *odk, const char *mrl) {
  gui_playlist_add_item (odk->gui, mrl, 1, GUI_ITEM_TYPE_AUTO, 0);
}

int odk_open_and_play(odk_t *odk, const char *mrl) {
  int entry_num = odk->gui->playlist.num;

  odk_enqueue(odk, mrl);

  if((xine_get_status(odk->gui->stream) != XINE_STATUS_STOP)) {
    odk->gui->ignore_next = 1;
    xine_stop(odk->gui->stream);
    odk->gui->ignore_next = 0;
  }

  if( odk->gui->playlist.num > entry_num ) {
    odk->gui->playlist.cur = entry_num;
    gui_current_set_index (odk->gui, entry_num);

    return gui_xine_open_and_play (odk->gui, odk->gui->mmk.mrl, odk->gui->mmk.sub, 0,
           odk->gui->mmk.start, odk->gui->mmk.av_offset, odk->gui->mmk.spu_offset, 0);
  } else
    return 0;
}

#ifdef YET_UNUSED

void odk_play(odk_t *odk) {
  gui_xine_play (odk->gui, odk->stream, 0, 0, 1);
}

void odk_stop(odk_t *odk) {
  gui_stop (NULL, odk->gui);
}

static int get_pos_length(xine_stream_t *stream, int *pos, int *time, int *length) {
  int t = 0, ret = 0;

  if(stream && (xine_get_status(stream) == XINE_STATUS_PLAY)) {
    while(((ret = xine_get_pos_length(stream, pos, time, length)) == 0) && (++t < 10))
      /*printf("wait");*/
      usleep(100000); /* wait before trying again */
  }
  return ret;
}

int odk_get_pos_length_high(odk_t *odk, int *pos, int *time, int *length) {
  int ret = 0;
#if 0
  static int last_time = 0;
#endif
  xine_stream_t *stream = odk->stream;

  if(stream && (xine_get_status(stream) == XINE_STATUS_PLAY)) {
    ret = xine_get_pos_length(stream, pos, time, length);
  }

  /*
   * smoothen the times xine returns a bit. filtering out small backjumps here.
   */

#if 0
  if (ret) {
    if ((*time < last_time) && (*time+1000 > last_time)) {
      *time = last_time;
    } else {
      last_time = *time;
    }
  }
#endif

  return ret;
}

int odk_get_pos_length(odk_t *odk, int *pos, int *time, int *length) {
  return get_pos_length(odk->stream, pos, time, length);
}

/* Return stream position 1..100 */
int odk_get_seek(odk_t *odk) {
  int pos_time, length;
  int pos=1;
  if(!odk_get_pos_length_high(odk, NULL, &pos_time, &length)) {
    return -1;
  }
  if (length) pos = ((pos_time*100) / length);
  /* printf("seek pos : %d\n", pos); */
  return pos;
}


void odk_seek(odk_t *odk, int how) {
  gui_seek_relative (odk->gui, how);
}

void odk_set_speed(odk_t *odk, uint32_t speed) {
  uint32_t s = XINE_SPEED_NORMAL;

  switch(speed) {
    case ODK_SPEED_PAUSE:
      s = XINE_SPEED_PAUSE;
      break;
    case ODK_SPEED_NORMAL:
      s = XINE_SPEED_NORMAL;
      break;
    case ODK_SPEED_SLOW_4:
      s = XINE_SPEED_SLOW_4;
      break;
    case ODK_SPEED_SLOW_2:
      s = XINE_SPEED_SLOW_2;
      break;
    case ODK_SPEED_FAST_4:
      s = XINE_SPEED_FAST_4;
      break;
    case ODK_SPEED_FAST_2:
      s = XINE_SPEED_FAST_2;
      break;
    default:
      printf("odk: odk_set_speed: invalid speed\n");

  }
  xine_set_param(odk->stream, XINE_PARAM_SPEED, s);
}

uint32_t odk_get_speed(odk_t *odk) {
  int s=xine_get_param(odk->stream, XINE_PARAM_SPEED);

  switch(s) {
    case XINE_SPEED_PAUSE:
      return ODK_SPEED_PAUSE;
      break;
    case XINE_SPEED_NORMAL:
      return ODK_SPEED_NORMAL;
      break;
    case XINE_SPEED_SLOW_4:
      return ODK_SPEED_SLOW_4;
      break;
    case XINE_SPEED_SLOW_2:
      return ODK_SPEED_SLOW_2;
      break;
    case XINE_SPEED_FAST_4:
      return ODK_SPEED_FAST_4;
      break;
    case XINE_SPEED_FAST_2:
      return ODK_SPEED_FAST_2;
      break;
    default:
      printf("odk: odk_get_speed: invalid speed\n");
  }
  return -1;
}

#if 0
void odk_toggle_pause(odk_t *odk) {

  gui_pause (NULL, odk->gui, 0);
}
#endif

void odk_eject(odk_t *odk) {

  xine_eject(odk->stream);
}

const char *odk_get_mrl(odk_t *odk) {
  return mediamark_get_current_mrl (odk->gui);
}

char *odk_get_meta_info(odk_t *odk, int info) {
  const char *str = xine_get_meta_info(odk->stream, info);

  if (!str) return NULL;
  return ho_strdup(str);
}
#endif

/*
 * error handling
 */
int odk_get_error(odk_t *odk) {
  return xine_get_error(odk->stream);
}


/** ITU-R BT470 (SD color):
 rrggbb yyuuvv
 000000 108080
 0000ff 29f06e
 ff0000 515af0
 ff00ff 6bcade
 00ff00 903622
 00ffff aaa610
 ffff00 d21092
 ffffff eb8080
*/
#define _one_ld 20
#define _one_v ((float)(1<<_one_ld))
#define _y_range (235.0-16.0)
#define _y_offs 16.0
#define _uv_range (240.0-128.0)
#define _uv_offs 128.0
#define _rgb_range 255.0
#define _ry 0.3
#define _by 0.11
static const int rgb2yuv_tab[3][4] = {{
  _one_v*(_ry*_y_range/_rgb_range),
  _one_v*((1.0-_ry-_by)*_y_range/_rgb_range),
  _one_v*(_by*_y_range/_rgb_range),
  _one_v*(0.5+_y_offs)
},{ /* r - y */
  _one_v*((1.0-_ry)*_uv_range/(1.0-_ry)/_rgb_range),
  _one_v*(-(1.0-_ry-_by)*_uv_range/(1.0-_ry)/_rgb_range),
  _one_v*(-_by*_uv_range/(1.0-_ry)/_rgb_range),
  _one_v*(0.5+_uv_offs)
},{ /* b - y */
  _one_v*(-_ry*_uv_range/(1.0-_by)/_rgb_range),
  _one_v*(-(1.0-_ry-_by)*_uv_range/(1.0-_by)/_rgb_range),
  _one_v*((1.0-_by)*_uv_range/(1.0-_by)/_rgb_range),
  _one_v*(0.5+_uv_offs)
}};

static uint32_t odk_rgba2yuvo (uint32_t rgba) {
  int v[8];
  v[0] = rgba >> 24;
  v[1] = (rgba >> 16) & 255;
  v[2] = (rgba >> 8) & 255;
  v[3] = rgba & 255;
  v[4] = (v[0] * rgb2yuv_tab[0][0] + v[1] * rgb2yuv_tab[0][1] + v[2] * rgb2yuv_tab[0][2] + rgb2yuv_tab[0][3]) >> _one_ld;
  v[5] = (v[0] * rgb2yuv_tab[1][0] + v[1] * rgb2yuv_tab[1][1] + v[2] * rgb2yuv_tab[1][2] + rgb2yuv_tab[1][3]) >> _one_ld;
  v[6] = (v[0] * rgb2yuv_tab[2][0] + v[1] * rgb2yuv_tab[2][1] + v[2] * rgb2yuv_tab[2][2] + rgb2yuv_tab[2][3]) >> _one_ld;
  v[7] = (v[3] + 8) / 17;
  return (v[4] << 24) + (v[5] << 16) + (v[6] << 8) + v[7];
}

static uint32_t odk_yuvo2rgba (uint32_t yuvo) {
  int v[8];
  v[0] = yuvo >> 24;
  v[1] = (yuvo >> 16) & 255;
  v[2] = (yuvo >> 8) & 255;
  v[3] = yuvo & 15;
  v[4] = v[5] = v[6] = v[0] * _one_v*(_rgb_range/_y_range) + _one_v*(0.5-_y_offs*_rgb_range/_y_range);
  v[1] -= _uv_offs;
  v[2] -= _uv_offs;
  v[4] += v[1] * _one_v*((1.0-_ry)*_rgb_range/_uv_range);
  v[5] -= v[2] * _one_v*(_by*(1.0-_by)/(1.0-_by-_ry)*_rgb_range/_uv_range)
       +  v[1] * _one_v*(_ry*(1.0-_ry)/(1.0-_ry-_by)*_rgb_range/_uv_range);
  v[6] += v[2] * _one_v*((1.0-_by)*_rgb_range/_uv_range);
  v[4] >>= _one_ld;
  v[4] = (v[4] & ~255) ? ~(v[4] >> (8 * sizeof (v[4]) - 1)) & 255 : v[4];
  v[5] >>= _one_ld;
  v[5] = (v[5] & ~255) ? ~(v[5] >> (8 * sizeof (v[5]) - 1)) & 255 : v[5];
  v[6] >>= _one_ld;
  v[6] = (v[6] & ~255) ? ~(v[6] >> (8 * sizeof (v[6]) - 1)) & 255 : v[6];
  v[7] = 17 * (v[3] & 15);
  return (v[4] << 24) + (v[5] << 16) + (v[6] << 8) + v[7];
}

#ifdef ODK_COLOR_TEST
static void odk_ctest (void) {
  int i;
  for (i = 0; i < 16; i++) {
    uint32_t rgba = ((((i >> 2) & 1) * 255) << 24) + ((((i >> 3) & 1) * 255) << 16) + ((((i >> 1) & 1) * 255) << 8) + (i * 17);
    uint32_t yuvo, rgb2;
    if (i & 1)
      rgba = (rgba >> 1) & 0x7f7f7f7f;
    yuvo = odk_rgba2yuvo (rgba);
    rgb2 = odk_yuvo2rgba (yuvo);
    printf ("gui.oxine.odk.rgba2yuvo (%08x) = %08x, %08x.\n", (unsigned int)rgba, (unsigned int)yuvo, (unsigned int)rgb2);
  }
}
#endif

static uint32_t odk_str2yuvo (const char *s) {
  char buf[12];
  const char *p = buf;
  uint32_t v;

  buf[0] = '#';
  strlcpy (buf + 1, s + (s[0] == '#'), sizeof (buf) - 2);
  v = xitk_str2int32 (&p, 0);
  if ((p[0] == '-') && p[1] && !p[2]) {
    uint32_t o;

    buf[p - buf] = '#';
    o = xitk_str2int32 (&p, 0);
    v = (v << 8) | o;
#ifdef ODK_COLOR_TEST
    printf ("gui.oxine.odk.str2yuvo.yuvo (%s) = %08x.\n", s, (unsigned int)v);
#endif
  } else {
    v = odk_rgba2yuvo (v);
#ifdef ODK_COLOR_TEST
    printf ("gui.oxine.odk.str2yuvo.rgba (%s) = %08x.\n", s, (unsigned int)v);
#endif
  }
  return v;
}

static void odk_update_sharp (void *data, xine_cfg_entry_t *cfg) {
  odk_palette_t *p = data;

  p->sharpness = cfg->num_value;
  p->changed = 1;
}

static void odk_update_p_yuvo (void *data, xine_cfg_entry_t *cfg) {
  uint8_t *indx = data, *first;
  odk_palette_t *p;
  uint32_t yuvo;

  for (first = indx; *first; first--) ;
  xitk_container (p, first, index[0]);
  yuvo = odk_str2yuvo (cfg->str_value);
  p->yuv[*indx] = odk_swap_color (yuvo >> 8);
  p->o[*indx] = yuvo & 0x0f;
  p->changed = 1;
}

void odk_palette_init (odk_palette_t *p, xine_t *xine, const char *key_prefix, const odk_user_color_t *user_colors, int num) {
  char id[512], value[64], *q = id, *e = id + sizeof (id);
  int uindx, pindx, findx;

  if (!p)
    return;
  memset (p, 0, sizeof (*p));
  /* p->used = 0; */
  if (!user_colors)
    return;

  if (key_prefix)
    q += strlcpy (q, key_prefix, e - q);

  for (uindx = pindx = findx = 0; uindx < num; uindx++) {
    int fitem, set_pindx;

    if (!user_colors[uindx].key)
      continue;
    fitem = (int)user_colors[uindx].key[0] - (int)'0';
    if ((uint32_t)fitem >= 5u)
      continue;
    strlcpy (q, user_colors[uindx].key + 1, e - q);
    if (fitem == 0) { /* sharpness */
      p->sharpness = user_colors[uindx].yuvo_default;
      if (user_colors[uindx].desc && xine)
        p->sharpness = xine_config_register_range (xine, id, p->sharpness, -127, 127,
          gettext (user_colors[uindx].desc), NULL, 10, odk_update_sharp, p);
    } else {
      const char *s;
      uint32_t yuvo;
      if (fitem == 1) { /* single color */
        set_pindx = pindx++;
      } else { /* font cmponent */
        static const uint8_t tab_fitem[5] = { 0, 0, 0, 10, 6 };
        if (fitem == 2)
          findx = pindx, pindx += 11;
        set_pindx = findx + tab_fitem[fitem];
      }
      p->index[uindx] = set_pindx;
      yuvo = user_colors[uindx].yuvo_default;
      snprintf (value, sizeof (value) - 1, "#%08x", (unsigned int)odk_yuvo2rgba (yuvo));
      s = xine_config_register_string (xine, id, value,
        gettext (user_colors[uindx].desc), "\"#rrggbbaa\" / \"yyuuvv-o\"", 10, odk_update_p_yuvo, p->index + uindx);
      if (s) {
        yuvo = odk_str2yuvo (s);
        if (s[0] != '#') {
          snprintf (value, sizeof (value) - 1, "#%08x", (unsigned int)odk_yuvo2rgba (yuvo));
          config_update_string (xine, id, value);
        }
      }
      p->yuv[set_pindx] = odk_swap_color (yuvo >> 8);
      p->o[set_pindx] = yuvo & 0x0f;
    }
  }
  p->used = pindx;
  p->changed = 1;
}

int odk_palette_update (odk_palette_t *p) {
  uint8_t tab[ODK_PALETTE_SIZE];
  int i, used;

  if (!p)
    return 0;
  if (!p->changed)
    return 0;

  memset (tab, 0, sizeof (tab));
  i = 0;
  used = (p->used < ODK_PALETTE_SIZE - 10) ? p->used : ODK_PALETTE_SIZE - 10;
  do {
    tab[p->index[i]] = 1;
    i++;
  } while (i < used);

  for (i = 0; i < used; i++) {
    if (!memcmp (tab + i, "\x01\x00\x00\x00", 4)) {
      _odk_palette_fill_text (p, i);
      i += 11 - 1;
    }
  }
  p->changed = 0;
  return 1;
}

uint32_t odk_palette_get_index (odk_palette_t *p, int indx) {
  if (!p)
    return 0;
  indx &= ODK_PALETTE_SIZE - 1;
  return (odk_swap_color (p->yuv[indx]) << 8) + p->o[indx];
}
