#!/bin/bash
# This file is part of TALER
# Copyright (C) 2014-2023 Taler Systems SA
#
# TALER is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 3, or
# (at your option) any later version.
#
# TALER is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with TALER; see the file COPYING.  If not, see
# <http://www.gnu.org/licenses/>
#

. setup.sh

# Launch only the merchant.
setup -c test_template.conf -m

LAST_RESPONSE=$(mktemp -p "${TMPDIR:-/tmp}" test_response.conf-XXXXXX)

STATUS=$(curl -H "Content-Type: application/json" -X OPTIONS \
    http://localhost:9966/private/products \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "204" ]
then
    exit_fail "Expected 204 when admin instance does not exist yet. got: $STATUS"
fi

STATUS=$(curl -H "Content-Type: application/json" -X GET \
    http://localhost:9966/private/products \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "404" ]
then
    exit_fail "Expected 404 when the admin instance is not yet created. got: $STATUS"
fi

STATUS=$(curl -H "Content-Type: application/json" -X POST \
    http://localhost:9966/management/instances \
    -d '{"auth":{"method":"token","password":"other_secret"},"id":"admin","name":"default","user_type":"business","address":{},"jurisdiction":{},"use_stefan":true,"default_wire_transfer_delay":{"d_us" : 3600000000},"default_pay_delay":{"d_us": 3600000000}}' \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "204" ]
then
    exit_fail "Expected 204, instance created. got: $STATUS"
fi

STATUS=$(curl -H "Content-Type: application/json" -X GET \
    http://localhost:9966/private/products \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "401" ]
then
    exit_fail "Expected 401 without the token for the list of product when the admin instance was created. got: $STATUS"
fi

BASIC_AUTH=$(echo -n admin:other_secret | base64)

STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H "Authorization: Basic $BASIC_AUTH" \
    http://localhost:9966/private/token \
    -d '{"scope":"write"}' \
    -w "%{http_code}" -s -o $LAST_RESPONSE)


if [ "$STATUS" != "200" ]
then
    exit_fail "Expected 200 OK. Got: $STATUS"
fi

TOKEN=$(jq -e -r .access_token < $LAST_RESPONSE)

STATUS=$(curl -H "Content-Type: application/json" -X GET \
    -H "Authorization: Bearer $TOKEN" \
    http://localhost:9966/private/products \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "200" ]
then
    exit_fail "Expected 200 for the list of product when the admin instance was created. got: $STATUS"
fi

STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H "Authorization: Bearer $TOKEN" \
    http://localhost:9966/private/auth \
    -d '{"method":"token","password":"zxc"}' \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "204" ]
then
    exit_fail "Expected 204, instance auth token changed. got: $STATUS"
fi


STATUS=$(curl -H "Content-Type: application/json" -X DELETE \
    "http://localhost:9966/private" \
    -w "%{http_code}" -s -o /dev/null)


if [ "$STATUS" != "401" ]
then
    exit_fail "Expected 401 without the token, when purging the instance. got: $STATUS"
fi

# FIXME: what we probably want here is that when changing the instance authentication
# settings all tokens are invalidated. We would have to add another DB operation
# for that. For now, we simply check here that we cannot get a new token with the
# old password.
STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H "Authorization: Basic $BASIC_AUTH" \
    http://localhost:9966/private/token \
    -d '{"scope":"write"}' \
    -w "%{http_code}" -s -o $LAST_RESPONSE)


if [ "$STATUS" != "401" ]
then
    exit_fail "Expected 401 with old password. Got: $STATUS"
fi

BASIC_AUTH=$(echo -n admin:zxc | base64)

STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H "Authorization: Basic $BASIC_AUTH" \
    http://localhost:9966/private/token \
    -d '{"scope":"write"}' \
    -w "%{http_code}" -s -o $LAST_RESPONSE)


if [ "$STATUS" != "200" ]
then
    exit_fail "Expected 200 OK. Got: $STATUS"
fi

TOKEN=$(jq -e -r .access_token < $LAST_RESPONSE)

STATUS=$(curl -H "Content-Type: application/json" -X DELETE \
    -H "Authorization: Bearer $TOKEN" \
    "http://localhost:9966/private" \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "204" ]
then
    exit_fail "Expected 204 when purging the instance. got: $STATUS"
fi

STATUS=$(curl -H "Content-Type: application/json" -X GET \
    -H "Authorization: Bearer $TOKEN" \
    http://localhost:9966/private/products \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "404" ]
then
    exit_fail "Expected 404 when trying to list the product and the admin instance was deleted. got: $STATUS"
fi

echo "Test PASSED"

exit 0
