/*
  This file is part of TALER
  Copyright (C) 2014-2023 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file test_merchant_api.c
 * @brief testcase to test exchange's HTTP API interface
 * @author Sree Harsha Totakura <sreeharsha@totakura.in>
 * @author Christian Grothoff
 * @author Marcello Stanisci
 */
#include "platform.h"
#include <gnunet/gnunet_time_lib.h>
#include <taler/taler_util.h>
#include <taler/taler_signatures.h>
#include <taler/taler_exchange_service.h>
#include <taler/taler_json_lib.h>
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_testing_lib.h>
#include <microhttpd.h>
#include <taler/taler_bank_service.h>
#include <taler/taler_fakebank_lib.h>
#include <taler/taler_testing_lib.h>
#include <taler/taler_error_codes.h>
#include "taler_merchant_testing_lib.h"


/**
 * The 'poll-orders-conclude-1' and other 'conclude'
 * commands should NOT wait for this timeout!
 */
#define POLL_ORDER_TIMEOUT \
        GNUNET_TIME_relative_multiply (GNUNET_TIME_UNIT_SECONDS, 60)

/**
 * The 'poll-orders-conclude-1x' and other 'conclude'
 * commands that should (!) wait for this timeout! Hence,
 * here we use a short value!
 */
#define POLL_ORDER_SHORT_TIMEOUT \
        GNUNET_TIME_relative_multiply (GNUNET_TIME_UNIT_SECONDS, 2)

/**
 * Configuration file we use.  One (big) configuration is used
 * for the various components for this test.
 */
static char *config_file;

/**
 * Exchange base URL.  Could also be taken from config.
 */
#define EXCHANGE_URL "http://localhost:8081/"

/**
 * Payto URI of the customer (payer).
 */
static struct TALER_FullPayto payer_payto;

/**
 * Payto URI of the exchange (escrow account).
 */
static struct TALER_FullPayto exchange_payto;

/**
 * Payto URI of the merchant (receiver).
 */
static struct TALER_FullPayto merchant_payto;

/**
 * Credentials for the test.
 */
static struct TALER_TESTING_Credentials cred;

/**
 * Merchant base URL.
 */
static const char *merchant_url;

/**
 * Merchant instance "i1a" base URL.
 */
static char *merchant_url_i1a;

/**
 * Account number of the exchange at the bank.
 */
#define EXCHANGE_ACCOUNT_NAME "2"

/**
 * Account number of some user.
 */
#define USER_ACCOUNT_NAME "62"

/**
 * Account number of some other user.
 */
#define USER_ACCOUNT_NAME2 "63"

/**
 * Account number used by the merchant
 */
#define MERCHANT_ACCOUNT_NAME "3"

static const char *order_1_transfers[] = {
  "post-transfer-1",
  NULL
};


/**
 * Run wire transfer of funds from some user's account to the
 * exchange.
 *
 * @param label label to use for the command.
 * @param amount amount to transfer, i.e. "EUR:1"
 * @param url exchange_url
 */
static struct TALER_TESTING_Command
cmd_transfer_to_exchange (const char *label,
                          const char *amount)
{
  return TALER_TESTING_cmd_admin_add_incoming (label,
                                               amount,
                                               &cred.ba,
                                               payer_payto);
}


/**
 * Main function that will tell the interpreter what commands to
 * run.
 *
 * @param cls closure
 */
static void
run (void *cls,
     struct TALER_TESTING_Interpreter *is)
{
  struct TALER_TESTING_Command pay[] = {
    /**
     * Move money to the exchange's bank account.
     */
    cmd_transfer_to_exchange ("create-reserve-1",
                              "EUR:5.01"),
    /**
     * Make a reserve exist, according to the previous transfer.
     */
    TALER_TESTING_cmd_exec_wirewatch ("wirewatch-1",
                                      config_file),
    TALER_TESTING_cmd_check_bank_admin_transfer ("check_bank_transfer-2",
                                                 "EUR:5.01",
                                                 payer_payto,
                                                 exchange_payto,
                                                 "create-reserve-1"),
    TALER_TESTING_cmd_withdraw_amount ("withdraw-coin-1",
                                       "create-reserve-1",
                                       "EUR:5",
                                       0,
                                       MHD_HTTP_OK),
    /**
     * Check the reserve is depleted.
     */
    TALER_TESTING_cmd_status ("withdraw-status-1",
                              "create-reserve-1",
                              "EUR:0",
                              MHD_HTTP_OK),
    TALER_TESTING_cmd_merchant_post_orders2 ("create-proposal-1",
                                             cred.cfg,
                                             merchant_url,
                                             MHD_HTTP_OK,
                                             "1",
                                             GNUNET_TIME_UNIT_ZERO_TS, /* refund deadline */
                                             GNUNET_TIME_UNIT_FOREVER_TS, /* pay deadline */
                                             true, /* use claim token */
                                             "EUR:5.0",
                                             "x-taler-bank",
                                             "",
                                             "",
                                             NULL),
    TALER_TESTING_cmd_merchant_claim_order ("reclaim-1",
                                            merchant_url,
                                            MHD_HTTP_OK,
                                            "create-proposal-1",
                                            NULL),
    TALER_TESTING_cmd_wallet_get_order ("get-order-wallet-1",
                                        merchant_url,
                                        "create-proposal-1",
                                        false,
                                        false,
                                        false,
                                        MHD_HTTP_PAYMENT_REQUIRED),
    TALER_TESTING_cmd_merchant_get_order ("get-order-merchant-1",
                                          merchant_url,
                                          "create-proposal-1",
                                          TALER_MERCHANT_OSC_CLAIMED,
                                          false,
                                          MHD_HTTP_OK,
                                          NULL),
    TALER_TESTING_cmd_merchant_pay_order ("deposit-simple",
                                          merchant_url,
                                          MHD_HTTP_OK,
                                          "create-proposal-1",
                                          "withdraw-coin-1",
                                          "EUR:5",
                                          "EUR:4.99",
                                          "session-0"),
    TALER_TESTING_cmd_merchant_get_order ("get-order-merchant-1-2a",
                                          merchant_url,
                                          "create-proposal-1",
                                          TALER_MERCHANT_OSC_PAID,
                                          false,
                                          MHD_HTTP_OK,
                                          NULL),
    TALER_TESTING_cmd_sleep (
      "Wait for wire transfer deadline",
      3),
    TALER_TESTING_cmd_exec_aggregator ("run-aggregator",
                                       config_file),
    TALER_TESTING_cmd_exec_transfer ("run-transfer",
                                     config_file),
    TALER_TESTING_cmd_check_bank_transfer ("check_bank_transfer-498c",
                                           EXCHANGE_URL,
                                           "EUR:4.98",
                                           exchange_payto,
                                           merchant_payto),
    // FIXME: replace by taler-merchant-wirewatch!
    TALER_TESTING_cmd_merchant_post_transfer ("post-transfer-1",
                                              &cred.ba,
                                              merchant_payto,
                                              merchant_url,
                                              "EUR:4.98",
                                              MHD_HTTP_NO_CONTENT,
                                              "deposit-simple",
                                              NULL),
    TALER_TESTING_cmd_depositcheck ("run-depositcheck",
                                    config_file),
    TALER_TESTING_cmd_run_tme ("run taler-merchant-reconciliation-1",
                               config_file),
    TALER_TESTING_cmd_merchant_get_order2 ("get-order-merchant-1-2b",
                                           merchant_url,
                                           "create-proposal-1",
                                           // FIXME: we won't want paid, we want settled!
                                           TALER_MERCHANT_OSC_PAID,
                                           true, /* wired */
                                           order_1_transfers, /* "post-transfer-1" */
                                           false, /* refunded */
                                           NULL,
                                           NULL,
                                           MHD_HTTP_OK),
    TALER_TESTING_cmd_check_bank_empty ("check_bank_empty-2"),
    TALER_TESTING_cmd_end ()
  };

  struct TALER_TESTING_Command commands[] = {
    /* general setup */
    TALER_TESTING_cmd_run_fakebank (
      "run-fakebank",
      cred.cfg,
      "exchange-account-exchange"),
    TALER_TESTING_cmd_system_start (
      "start-taler",
      config_file,
      "-em",
      "-u", "exchange-account-exchange",
      "-r", "merchant-exchange-test",
      NULL),
    TALER_TESTING_cmd_get_exchange (
      "get-exchange",
      cred.cfg,
      NULL,
      true,
      true),
    TALER_TESTING_cmd_config (
      "config",
      merchant_url,
      MHD_HTTP_OK),
    TALER_TESTING_cmd_merchant_get_instances (
      "instances-empty",
      merchant_url,
      MHD_HTTP_OK,
      NULL),
    TALER_TESTING_cmd_merchant_post_instances (
      "instance-create-default-setup",
      merchant_url,
      "admin",
      MHD_HTTP_NO_CONTENT),
    // FIXME: setup full account for taler-merchant-wirewatch
    TALER_TESTING_cmd_merchant_post_account (
      "instance-create-default-account",
      merchant_url,
      merchant_payto,
      NULL, NULL,
      MHD_HTTP_OK),
    TALER_TESTING_cmd_merchant_post_instances (
      "instance-create-i1",
      merchant_url,
      "i1",
      MHD_HTTP_NO_CONTENT),
    TALER_TESTING_cmd_batch ("pay",
                             pay),
    /**
     * End the suite.
     */
    TALER_TESTING_cmd_end ()
  };

  TALER_TESTING_run (is,
                     commands);
}


int
main (int argc,
      char *const *argv)
{
  {
    char *cipher;

    cipher = GNUNET_STRINGS_get_suffix_from_binary_name (argv[0]);
    GNUNET_assert (NULL != cipher);
    GNUNET_asprintf (&config_file,
                     "test_merchant_api-%s.conf",
                     cipher);
    GNUNET_free (cipher);
  }
  payer_payto.full_payto =
    (char *) "payto://x-taler-bank/localhost/" USER_ACCOUNT_NAME
    "?receiver-name=" USER_ACCOUNT_NAME;
  exchange_payto.full_payto =
    (char *) "payto://x-taler-bank/localhost/" EXCHANGE_ACCOUNT_NAME
    "?receiver-name="
    EXCHANGE_ACCOUNT_NAME;
  merchant_payto.full_payto =
    (char *) "payto://x-taler-bank/localhost/" MERCHANT_ACCOUNT_NAME
    "?receiver-name=" MERCHANT_ACCOUNT_NAME;
  merchant_url = "http://localhost:8080/";
  GNUNET_asprintf (&merchant_url_i1a,
                   "%sinstances/i1a/",
                   merchant_url);
  return TALER_TESTING_main (argv,
                             "INFO",
                             config_file,
                             "exchange-account-exchange",
                             TALER_TESTING_BS_FAKEBANK,
                             &cred,
                             &run,
                             NULL);
}


/* end of test_merchant_api.c */
