/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-merchant-httpd_private-get-instances-ID-tokens.c
 * @brief implement GET /tokens
 * @author Martin Schanzenbach
 */
#include "platform.h"
#include "taler-merchant-httpd_auth.h"
#include "taler-merchant-httpd_private-get-instances-ID-tokens.h"


/**
 * Add token details to our JSON array.
 *
 * @param cls a `json_t *` JSON array to build
 * @param creation_time when the token was created
 * @param expiration_time when the token will expire
 * @param scope internal scope identifier for the token (mapped to string)
 * @param description human-readable purpose or context of the token
 * @param serial serial (row) number of the product in the database
 */
static void
add_token (void *cls,
           struct GNUNET_TIME_Timestamp creation_time,
           struct GNUNET_TIME_Timestamp expiration_time,
           uint32_t scope,
           const char *description,
           uint64_t serial)
{
  json_t *pa = cls;
  bool refreshable;
  const char *as;

  as = TMH_get_name_by_scope (scope,
                              &refreshable);
  if (NULL == as)
  {
    GNUNET_break (0);
    return;
  }
  GNUNET_assert (0 ==
                 json_array_append_new (
                   pa,
                   GNUNET_JSON_PACK (
                     GNUNET_JSON_pack_timestamp ("creation_time",
                                                 creation_time),
                     GNUNET_JSON_pack_timestamp ("expiration",
                                                 expiration_time),
                     GNUNET_JSON_pack_string ("scope",
                                              as),
                     GNUNET_JSON_pack_bool ("refreshable",
                                            refreshable),
                     GNUNET_JSON_pack_string ("description",
                                              description),
                     GNUNET_JSON_pack_uint64 ("serial",
                                              serial))));
}


MHD_RESULT
TMH_private_get_instances_ID_tokens (const struct TMH_RequestHandler *rh,
                                     struct MHD_Connection *connection,
                                     struct TMH_HandlerContext *hc)
{
  json_t *ta;
  enum GNUNET_DB_QueryStatus qs;
  int64_t limit = -20; /* default */
  uint64_t offset;

  TALER_MHD_parse_request_snumber (connection,
                                   "limit",
                                   &limit);
  if (limit > 0)
    offset = 0;
  else
    offset = INT64_MAX;
  TALER_MHD_parse_request_number (connection,
                                  "offset",
                                  &offset);
  ta = json_array ();
  GNUNET_assert (NULL != ta);
  qs = TMH_db->lookup_login_tokens (TMH_db->cls,
                                    hc->instance->settings.id,
                                    offset,
                                    limit,
                                    &add_token,
                                    ta);
  if (0 > qs)
  {
    GNUNET_break (0);
    json_decref (ta);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_FETCH_FAILED,
                                       NULL);
  }
  return TALER_MHD_REPLY_JSON_PACK (connection,
                                    MHD_HTTP_OK,
                                    GNUNET_JSON_pack_array_steal ("tokens",
                                                                  ta));
}


/* end of taler-merchant-httpd_private-get-instances-ID-tokens.c */
