/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_select_money_pots.c
 * @brief Implementation of the select_money_pots function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_select_money_pots.h"
#include "pg_helper.h"


/**
 * Context used for TMH_PG_lookup_money_pots().
 */
struct LookupMoneyPotsContext
{
  /**
   * DB context.
   */
  struct PostgresClosure *pg;

  /**
   * Function to call with the results.
   */
  TALER_MERCHANTDB_MoneyPotsCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Did database result extraction fail?
   */
  bool extract_failed;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about money pots.
 *
 * @param[in,out] cls of type `struct LookupMoneyPotsContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
lookup_money_pots_cb (void *cls,
                      PGresult *result,
                      unsigned int num_results)
{
  struct LookupMoneyPotsContext *plc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    uint64_t money_pot_serial;
    char *money_pot_name;
    size_t pot_total_len;
    struct TALER_Amount *pot_totals;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_uint64 ("money_pot_serial",
                                    &money_pot_serial),
      GNUNET_PQ_result_spec_string ("money_pot_name",
                                    &money_pot_name),
      TALER_PQ_result_spec_array_amount_with_currency (plc->pg->conn,
                                                       "pot_totals",
                                                       &pot_total_len,
                                                       &pot_totals),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      plc->extract_failed = true;
      return;
    }
    plc->cb (plc->cb_cls,
             money_pot_serial,
             money_pot_name,
             pot_total_len,
             pot_totals);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_select_money_pots (void *cls,
                          const char *instance_id,
                          int64_t limit,
                          uint64_t offset,
                          TALER_MERCHANTDB_MoneyPotsCallback cb,
                          void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  uint64_t plimit = (uint64_t) ((limit < 0) ? -limit : limit);
  struct LookupMoneyPotsContext plc = {
    .pg = pg,
    .cb = cb,
    .cb_cls = cb_cls,
    /* Can be overwritten by the lookup_money_pots_cb */
    .extract_failed = false,
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_uint64 (&offset),
    GNUNET_PQ_query_param_uint64 (&plimit),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "lookup_money_pots_asc",
           "SELECT"
           "  money_pot_serial"
           " ,money_pot_name"
           " ,pot_totals"
           " FROM merchant_money_pots"
           " JOIN merchant_instances"
           "   USING (merchant_serial)"
           " WHERE merchant_instances.merchant_id=$1"
           "   AND money_pot_serial > $2"
           " ORDER BY money_pot_serial ASC"
           " LIMIT $3");
  PREPARE (pg,
           "lookup_money_pots_desc",
           "SELECT"
           "  money_pot_serial"
           " ,money_pot_name"
           " ,pot_totals"
           " FROM merchant_money_pots"
           " JOIN merchant_instances"
           "   USING (merchant_serial)"
           " WHERE merchant_instances.merchant_id=$1"
           "   AND money_pot_serial < $2"
           " ORDER BY money_pot_serial DESC"
           " LIMIT $3");
  qs = GNUNET_PQ_eval_prepared_multi_select (
    pg->conn,
    (limit > 0)
    ? "lookup_money_pots_asc"
    : "lookup_money_pots_desc",
    params,
    &lookup_money_pots_cb,
    &plc);
  /* If there was an error inside lookup_money_pots_cb, return a hard error. */
  if (plc.extract_failed)
  {
    GNUNET_break (0);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  return qs;
}
