﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/CustomParameter.h>
#include <aws/datazone/model/ProvisioningProperties.h>

#include <utility>

namespace Aws {
namespace DataZone {
namespace Model {

/**
 */
class UpdateEnvironmentBlueprintRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API UpdateEnvironmentBlueprintRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateEnvironmentBlueprint"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone domain in which an environment
   * blueprint is to be updated.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  UpdateEnvironmentBlueprintRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the environment blueprint to be updated.</p>
   */
  inline const Aws::String& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = Aws::String>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = Aws::String>
  UpdateEnvironmentBlueprintRequest& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description to be updated as part of the
   * <code>UpdateEnvironmentBlueprint</code> action.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateEnvironmentBlueprintRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The provisioning properties to be updated as part of the
   * <code>UpdateEnvironmentBlueprint</code> action.</p>
   */
  inline const ProvisioningProperties& GetProvisioningProperties() const { return m_provisioningProperties; }
  inline bool ProvisioningPropertiesHasBeenSet() const { return m_provisioningPropertiesHasBeenSet; }
  template <typename ProvisioningPropertiesT = ProvisioningProperties>
  void SetProvisioningProperties(ProvisioningPropertiesT&& value) {
    m_provisioningPropertiesHasBeenSet = true;
    m_provisioningProperties = std::forward<ProvisioningPropertiesT>(value);
  }
  template <typename ProvisioningPropertiesT = ProvisioningProperties>
  UpdateEnvironmentBlueprintRequest& WithProvisioningProperties(ProvisioningPropertiesT&& value) {
    SetProvisioningProperties(std::forward<ProvisioningPropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user parameters to be updated as part of the
   * <code>UpdateEnvironmentBlueprint</code> action.</p>
   */
  inline const Aws::Vector<CustomParameter>& GetUserParameters() const { return m_userParameters; }
  inline bool UserParametersHasBeenSet() const { return m_userParametersHasBeenSet; }
  template <typename UserParametersT = Aws::Vector<CustomParameter>>
  void SetUserParameters(UserParametersT&& value) {
    m_userParametersHasBeenSet = true;
    m_userParameters = std::forward<UserParametersT>(value);
  }
  template <typename UserParametersT = Aws::Vector<CustomParameter>>
  UpdateEnvironmentBlueprintRequest& WithUserParameters(UserParametersT&& value) {
    SetUserParameters(std::forward<UserParametersT>(value));
    return *this;
  }
  template <typename UserParametersT = CustomParameter>
  UpdateEnvironmentBlueprintRequest& AddUserParameters(UserParametersT&& value) {
    m_userParametersHasBeenSet = true;
    m_userParameters.emplace_back(std::forward<UserParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainIdentifier;

  Aws::String m_identifier;

  Aws::String m_description;

  ProvisioningProperties m_provisioningProperties;

  Aws::Vector<CustomParameter> m_userParameters;
  bool m_domainIdentifierHasBeenSet = false;
  bool m_identifierHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_provisioningPropertiesHasBeenSet = false;
  bool m_userParametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
