(defun inject-goals (new-goals goals)
  (cond ((endp new-goals) goals)
	(t (let* ((goal (car new-goals))
		  (key  (car goal))
		  (rest (cdr new-goals)))
	     (begin
	      (cw "you have ~x0 and ~x1~%" goals new-goals)
	     (inject-goals
	      rest
	      (put-assoc-equal key 
			       (append (cdr (assoc-equal key goals))
				       (cdr goal))
			       goals)))))))

;; (new-thm symbol symbol) -> acl2-internal-symbol
;; returns concatenated thm-x-y as new acl2 internal symbol,
;; suitable for using as defthm or defun name
(defun new-thm (x y)
  (intern (concatenate 'string 
		       "THM-" 
		       (stringify x) 
		       "-" 
		       (stringify y))
	  "ACL2"))

(defun catsym (x y)
  (intern (concatenate 'string 
		       (stringify x) 
		       (stringify y))
	  "ACL2"))

;; (make-returns-symbol-alistp-thm symbol) -> thm-func-returns-symbol-alistp
;; Makes theorem which theorizes that a feature function returns
;; a symbol-alistp
(defmacro make-*/mail-returns-symbol-alistp-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-symbol-alistp)
     (implies
       (and
	(message-p msg)
	(symbol-alistp env)
	)
       (mv-let (s new-msg new-env)
	       (,func msg env)
	       (symbol-alistp new-env)))

     :hints ,(inject-goals (list (list "Goal" 
                 ':in-theory (list 'enable func))) hints)))


(defmacro make-*/mail-returns-symbolp-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-symbolp)
       (mv-let (s new-msg new-env)
	       (,func msg env)
	       (symbolp s))
     :hints ,(inject-goals (list (list "Goal" 
                 ':in-theory (list 'enable func))) hints)))

;; (make-returns-same-env-thm symbol) -> thm-func-returns-same-env
;; Makes theorem which theorizes that a feature function returns the
;; same env passed into it.
;;
(defmacro make-*/mail-returns-same-env-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-same-env)
     (implies
      (and
       (message-p msg)
       (symbol-alistp env))
      (mv-let (s new-msg new-env)
	      (,func msg env)
	      (equal new-env env)))
     :hints ,(inject-goals (list (list "Goal" 
                ':in-theory (list 'enable func))) hints)))

;; (make-*/mail-returns-message-p-thm symbol) -> thm-func-returns-message-p-thm
;; Makes theorem which theorized that a feature function returns
;; always a message-p message.  
(defmacro make-*/mail-returns-message-p-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-message-p)
     (implies
      (and
       (message-p msg)
       )
      (message-p (mv-nth 1 (,func msg env))))
     :rule-classes (:rewrite (:FORWARD-CHAINING :TRIGGER-TERMS  
                        ((mv-nth 1 (,func msg env)))))
     :hints ,(inject-goals (list (list "Goal" 
             ':in-theory (list 'enable func))) hints)))

;; (add-has-vars-conditions list[symbol]) -> 
;; Helper function for make-adds-x-variables-thm 
(defmacro add-has-var-conditions (vars)
  (let ((variable (car vars))
	(rst      (cdr vars)))
  (cond ((endp rst) `(has-var (quote ,variable)  new-env))
	(t `(and (has-var (quote ,variable) new-env)
		 (add-has-var-conditions ,rst))))))

;; (make-*/init-adds-x-variables-thm symbol list[symbol])
;; Makes theorem which theorizes that feature function adds 
;; only particular variables to env
(defmacro make-*/init-adds-x-variables-thm (func vars &optional hints)
  `(defthm ,(new-thm func 'adds-x-variables)
     (implies 
      (symbol-alistp env)
      (let ((new-env 
	     (,func env)))
	(add-has-var-conditions ,@(cdr vars))))
     :hints ,(inject-goals (list (list "Goal" 
                ':in-theory (list 'enable func))) hints)))


;; (make-changes-only-x-variables-thm symbol list[symbol])
;; Makes theorem which theorizes that feature function does not change
;; variables other than specified
(defmacro make-*/init-changes-only-x-variables-thm (func variables)
  `(defthm ,(new-thm func 'changes-only-x-variables)
     (implies 
      (and 
       (symbol-alistp env)
       (equal (get-var key env)
	      var)
       (not (member key ,variables))
      )
     (let ((new-env 
	    (,func env)))
       (equal (get-var key new-env)
	      var)))))

;; (make-add-and-changes-only-x-variables-thm symbol list[symbol])
;; Makes theorem which theorizes that feature function adds only
;; variables specified, and that no other variables are added
(defmacro make-*/init-add-and-changes-only-x-variables-thm 
                           (func variables &optional hints)
  `(defthm ,(new-thm func 'adds-and-changes-only-x-variables)
     (implies 
      (and 
       (symbol-alistp env)
       (equal (get-var key env)
	      var)
       (not (member key ,variables))
      )
     (let ((new-env 
	    (,func env)))
       (and
	(equal (get-var key new-env)
	       var)
	(add-has-var-conditions ,@(cdr variables))
       )))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))


(defmacro make-f/comm-add-and-changes-only-x-variables-thm 
      (func variables &optional hints)
  `(defthm ,(new-thm func 'adds-and-changes-only-x-variables)
     (implies 
      (and 
       (symbol-alistp env)
       (equal (get-var key env)
	      var)
       (not (member key ,variables))
      )
     (let ((new-env 
	    (,func cmd args env)))
       (and
	(equal (get-var key new-env)
	       var)
	(add-has-var-conditions ,@(cdr variables))
       )))
     :hints ,(inject-goals (list (list "Goal" 
         ':in-theory (list 'enable func))) hints)))

(defmacro make-*/mail-returns-same-message-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-same-message)
     (implies
      (and
       (symbol-alistp user)
       msg)
      (mv-let (new-status new-msg new-user)
	       (,func msg user)
	       (equal new-msg msg)))
     :hints ,(inject-goals (list (list "Goal" 
              ':in-theory (list 'enable func))) hints)))


;(defmacro make-user-*-returns-action-listp-thm (func &optional hints)
; un-needed due to not dealing with action-list

(defmacro make-d/that-returns-action-listp-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-action-listp)
     (implies
      (and
       (action-p action)
       (action-listp rest)
       (symbol-alistp users)
       (symbol-alistp hosts)
       )
      (mv-let (new-rest new-users new-hosts)
	      (,func action rest users hosts)
	      (action-listp new-rest)))
     :hints ,(inject-goals (list (list "Goal" 
       ':in-theory (list 'enable func))) hints)))


(defmacro make-d/that-returns-action-listp-if-message-p-thm  
   (func &optional hints)
  `(defthm ,(new-thm func 'returns-action-listp)
     (implies
      (and
       (action-p action)
       (message-p (action-arg1 action))
       (action-listp rest)
       (symbol-alistp users)
       (symbol-alistp hosts)
       )
      (mv-let (new-rest new-users new-hosts)
	      (,func action rest users hosts)
	      (action-listp new-rest)))
     :hints ,(inject-goals (list (list "Goal" 
       ':in-theory (list 'enable func))) hints)))

(defmacro make-*/mail-returns-superset-of-user-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-superset-of-user)
     (implies
      (and
       (symbol-alistp user)
       msg
       )
       (mv-let (new-status new-msg new-user)
	       (,func msg user)
	       (env-subset user new-user)))
     :rule-classes :forward-chaining
     :hints ,(inject-goals (list (list "Goal" 
         ':in-theory (list 'enable func))) hints)))


(defmacro make-f/comm-returns-superset-of-user-thm 
        (func &optional hints)
  `(defthm ,(new-thm func 'returns-superset-of-user)
     (implies
      (and
       (symbol-alistp user)
       args
       cmd
)
       (let ((new-user (,func cmd args user)))
	 (env-subset user new-user)))
     :rule-classes :forward-chaining
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))



(defmacro make-d/that-returns-superset-of-users-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-superset-of-users)
     (implies
      (and
       (symbol-alistp users)
       (symbol-alistp hosts)
       msg)
       (mv-let (new-rest new-users new-hosts)
	       (,func action rest users hosts)
	       (env-subset users new-users)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))


(defmacro make-d/that-returns-symbol-alistp-users-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-symbol-alistp-users)
     (implies
      (and
       (symbol-alistp users)
       (symbol-alistp hosts)
       (symbolp (action-name action))
       ;msg
       )
       (mv-let (new-rest new-users new-hosts)
	       (,func action rest users hosts)
	       (symbol-alistp new-users)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))


(defmacro make-*/init-returns-symbol-alistp-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-symbol-alistp)
     (implies
      (and
       (symbol-alistp env)
       msg)
       (let ((new-env (,func env)))
	 (symbol-alistp new-env)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))


(defmacro make-d/that-returns-symbol-alistp-users-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-symbol-alistp-users)
     (implies
      (and
       (symbol-alistp users)
       (symbol-alistp hosts)
       (symbolp (action-name action))
       )
       (mv-let (new-rest new-users new-hosts)
	       (,func action rest users hosts)
	       (symbol-alistp new-users)))
     :hints ,(inject-goals (list (list "Goal" 
         ':in-theory (list 'enable func))) hints)))


(defmacro make-d/that-returns-symbol-alistp-hosts-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-symbol-alistp-hosts)
     (implies
      (and
       (symbol-alistp hosts)
       (symbol-alistp users)
       (symbolp (action-name action))
       )
       (mv-let (new-rest new-users new-hosts)
	       (,func action rest users hosts)
	       (symbol-alistp new-hosts)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))



(defmacro make-d/that-returns-superset-of-hosts-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-superset-of-hosts)
     (implies
      (and
       (or rest (null rest))
       (or action (null action))
       (symbol-alistp users)
       (symbol-alistp hosts)
       msg)
       (mv-let (new-rest new-users new-hosts)
	       (,func action rest users hosts)
	       (env-subset hosts new-hosts)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))



(defmacro make-*/init-returns-superset-of-env-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-superset-of-env)
     (implies
      (and
       (symbol-alistp env)
       )
      (let ((new-env (,func env)))
	(env-subset env new-env)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))



(defmacro make-*/mail-returns-same-user-thm (func &optional hints)
  `(defthm ,(new-thm func 'returns-same-user)
     (implies
      (and
       (symbol-alistp user)
       )
       (mv-let (new-status new-msg new-user)
	       (,func msg user)
	       (equal new-user user)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))


(defmacro make-*/mail-returns-superset-of-user-if-message-p-thm 
  (func &optional hints)
  `(defthm ,(new-thm func 'returns-superset-of-user-if-message-p)
     (implies
      (and
       (symbol-alistp user)
       (message-p msg))
       (mv-let (new-status new-msg new-user)
	       (,func msg user)
	       (env-subset user new-user)))
     :hints ,(inject-goals (list (list "Goal" 
        ':in-theory (list 'enable func))) hints)))


(defmacro make-d/that-returns-no-actions (func &optional hints)
  `(defthm ,(new-thm func 'returns-no-actions)
     (implies
      t
      (mv-let (new-rest new-users new-hosts)
	      (,func action rest users hosts)
	      (equal new-rest rest)))
     :hints ,hints))

(defmacro make-d/that-returns-at-most-one-action (func &optional hints)
  `(defthm ,(new-thm func 'returns-no-actions)
     (implies
      t
      (mv-let (new-rest new-users new-hosts)
	      (,func action rest users hosts)
	      (equal-or-one-off new-rest rest)))
     :hints ,hints))

(defun sender-becomes-recipient (a b)
  (equal
   (message-sender    a)
   (recipient         b)))

