/*
 *  $Id: xyz_channels.c 21089 2018-05-09 06:47:51Z yeti-dn $
 *  Copyright (C) 2003-2018 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/arithmetic.h>
#include <libprocess/inttrans.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define XYZ_CHANNELS_RUN_MODES (GWY_RUN_INTERACTIVE)

typedef struct {
    GwyAppDataId xdata;
    GwyAppDataId ydata;
    GwyAppDataId zdata;
} XyzChannelsArgs;

typedef struct {
    XyzChannelsArgs *args;
    GtkWidget *xdata;
    GtkWidget *ydata;
    GtkWidget *zdata;
} XyzChannelsControls;

static gboolean module_register          (void);
static void     xyz_channels                      (GwyContainer *data,
                                          GwyRunType run);
static gboolean xyz_channels_dialog               (XyzChannelsArgs *args,
                                          GwyDataField *dfield);
static gboolean xyz_channels_data_filter(GwyContainer *data,
                                          gint id,
                                          gpointer user_data);
static void     xyz_channels_data_chosen  (GwyDataChooser *chooser,
                                           GwyAppDataId *dataid);


static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Converts three channels to XYZ data."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2018",
};

GWY_MODULE_QUERY2(module_info, xyz_channels)

static gboolean
module_register(void)
{
    gwy_process_func_register("xyz_channels",
                              (GwyProcessFunc)&xyz_channels,
                              N_("/_Basic Operations/XYZ from C_hannels..."),
                              NULL,
                              XYZ_CHANNELS_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Convert three channels to XYZ data"));

    return TRUE;
}

static void
xyz_channels(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *xdfield, *ydfield, *zdfield;
    GwyContainer *xdata, *ydata, *zdata;
    gdouble *xd, *yd, *zd;
    GwySurface *surface;
    GwyXYZ *xyz;
    gint xres, yres;
    XyzChannelsArgs args;
    gboolean ok;
    gint id, datano, i, newid;
    GQuark quark;

    g_return_if_fail(run & XYZ_CHANNELS_RUN_MODES);

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_CONTAINER_ID, &datano,
                                     0);
    g_return_if_fail(dfield);

    ok = xyz_channels_dialog(&args, dfield);
    if (!ok)
        return;

    xdata = gwy_app_data_browser_get(args.xdata.datano);
    quark = gwy_app_get_data_key_for_id(args.xdata.id);
    xdfield = GWY_DATA_FIELD(gwy_container_get_object(xdata, quark));

    ydata = gwy_app_data_browser_get(args.ydata.datano);
    quark = gwy_app_get_data_key_for_id(args.ydata.id);
    ydfield = GWY_DATA_FIELD(gwy_container_get_object(ydata, quark));

    zdata = gwy_app_data_browser_get(args.zdata.datano);
    quark = gwy_app_get_data_key_for_id(args.zdata.id);
    zdfield = GWY_DATA_FIELD(gwy_container_get_object(zdata, quark));

    xres = gwy_data_field_get_xres(xdfield);
    yres = gwy_data_field_get_yres(xdfield);

    surface = gwy_surface_new_sized(xres*yres);

    xyz = gwy_surface_get_data(surface);
    xd = gwy_data_field_get_data(xdfield);
    yd = gwy_data_field_get_data(ydfield);
    zd = gwy_data_field_get_data(zdfield);

    for (i = 0; i < xres*yres; i++) {
        xyz[i].x = xd[i];
        xyz[i].y = yd[i];
        xyz[i].z = zd[i];
    }

    /* FIXME: Should probably only allow Y coordinates that are the same
     * physical quantity as X coordinates. */
    gwy_si_unit_assign(gwy_surface_get_si_unit_xy(surface),
                       gwy_data_field_get_si_unit_z(xdfield));
    gwy_si_unit_assign(gwy_surface_get_si_unit_z(surface),
                       gwy_data_field_get_si_unit_z(zdfield));

    newid = gwy_app_data_browser_add_surface(surface, data, TRUE);
    g_object_unref(surface);
    gwy_app_xyz_log_add(data, -1, newid, "proc::xyzize", NULL);
}

static gboolean
xyz_channels_dialog(XyzChannelsArgs *args,
                    GwyDataField *dfield)
{
    GtkWidget *dialog, *table;
    XyzChannelsControls controls;
    gint response, row;

    controls.args = args;

    dialog = gtk_dialog_new_with_buttons(_("XYZ Channels"), NULL, 0,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    table = gtk_table_new(7, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table,
                       FALSE, FALSE, 4);
    row = 0;

    controls.xdata = gwy_data_chooser_new_channels();
    gwy_data_chooser_set_filter(GWY_DATA_CHOOSER(controls.xdata),
                                xyz_channels_data_filter, dfield, NULL);
    gwy_data_chooser_set_active_id(GWY_DATA_CHOOSER(controls.xdata),
                                   &args->xdata);
    gwy_table_attach_adjbar(table, row, _("_X data:"), NULL,
                            GTK_OBJECT(controls.xdata),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    g_signal_connect(controls.xdata, "changed",
                         G_CALLBACK(xyz_channels_data_chosen), &(args->xdata));
    row++;

    controls.ydata = gwy_data_chooser_new_channels();
    gwy_data_chooser_set_filter(GWY_DATA_CHOOSER(controls.ydata),
                                xyz_channels_data_filter, dfield, NULL);
    gwy_data_chooser_set_active_id(GWY_DATA_CHOOSER(controls.ydata),
                                   &args->ydata);
    gwy_table_attach_adjbar(table, row, _("_Y data:"), NULL,
                            GTK_OBJECT(controls.ydata),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    g_signal_connect(controls.ydata, "changed",
                         G_CALLBACK(xyz_channels_data_chosen), &(args->ydata));
     row++;

    controls.zdata = gwy_data_chooser_new_channels();
    gwy_data_chooser_set_filter(GWY_DATA_CHOOSER(controls.zdata),
                                xyz_channels_data_filter, dfield, NULL);
    gwy_data_chooser_set_active_id(GWY_DATA_CHOOSER(controls.zdata),
                                   &args->zdata);
    gwy_table_attach_adjbar(table, row, _("_Z data:"), NULL,
                            GTK_OBJECT(controls.zdata),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    g_signal_connect(controls.zdata, "changed",
                         G_CALLBACK(xyz_channels_data_chosen), &(args->zdata));
    row++;

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            gwy_data_chooser_get_active_id(GWY_DATA_CHOOSER(controls.xdata),
                                           &(args->xdata));
            gwy_data_chooser_get_active_id(GWY_DATA_CHOOSER(controls.ydata),
                                           &(args->ydata));
            gwy_data_chooser_get_active_id(GWY_DATA_CHOOSER(controls.zdata),
                                           &(args->zdata));
            break;


            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    return TRUE;
}

static void
xyz_channels_data_chosen(GwyDataChooser *chooser,
                         GwyAppDataId *dataid)
{
    gwy_data_chooser_get_active_id(chooser, dataid);
}

static gboolean
xyz_channels_data_filter(GwyContainer *data,
                         gint id,
                         gpointer user_data)
{
    GwyDataField *dfield = (GwyDataField*)user_data;
    GwyDataField *myfield;
    GQuark quark;

    if (!data || id < 0)
        return FALSE;

    quark = gwy_app_get_data_key_for_id(id);
    if (!gwy_container_gis_object(data, quark, (GObject**)&myfield))
        return FALSE;

    return !gwy_data_field_check_compatibility(myfield, dfield,
                                               GWY_DATA_COMPATIBILITY_RES
                                               | GWY_DATA_COMPATIBILITY_REAL);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
