// Take a look at the license at the top of the repository in the LICENSE file.

use crate::{ffi, prelude::*};
use glib::{translate::*, value::FromValue, GString, Type, Value};

#[derive(Clone, Copy, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
// rustdoc-stripper-ignore-next
/// A wrapper around `GDK_KEY_` constants and keyval methods.
pub struct Key(u32);

#[allow(non_upper_case_globals)]
impl Key {
    #[doc(alias = "gdk_keyval_from_name")]
    pub fn from_name(name: impl IntoGStr) -> Option<Self> {
        skip_assert_initialized!();
        unsafe {
            name.run_with_gstr(|name| {
                let key: Self = from_glib(ffi::gdk_keyval_from_name(name.as_ptr()));
                if key == Self::VoidSymbol {
                    None
                } else {
                    Some(key)
                }
            })
        }
    }

    #[doc(alias = "gdk_keyval_convert_case")]
    pub fn convert_case(&self) -> (Self, Self) {
        assert_initialized_main_thread!();
        unsafe {
            let mut lower = std::mem::MaybeUninit::uninit();
            let mut upper = std::mem::MaybeUninit::uninit();
            ffi::gdk_keyval_convert_case(self.into_glib(), lower.as_mut_ptr(), upper.as_mut_ptr());
            let lower = lower.assume_init();
            let upper = upper.assume_init();
            (from_glib(lower), from_glib(upper))
        }
    }

    #[doc(alias = "gdk_keyval_to_unicode")]
    pub fn to_unicode(&self) -> Option<char> {
        skip_assert_initialized!();
        unsafe {
            ::std::char::from_u32(ffi::gdk_keyval_to_unicode(self.into_glib()))
                .filter(|x| *x != '\0')
        }
    }

    #[doc(alias = "gdk_keyval_name")]
    pub fn name(&self) -> Option<GString> {
        skip_assert_initialized!();
        unsafe { from_glib_none(ffi::gdk_keyval_name(self.into_glib())) }
    }

    #[doc(alias = "gdk_keyval_is_upper")]
    pub fn is_upper(&self) -> bool {
        skip_assert_initialized!();
        unsafe { from_glib(ffi::gdk_keyval_is_upper(self.into_glib())) }
    }

    #[doc(alias = "gdk_keyval_is_lower")]
    pub fn is_lower(&self) -> bool {
        skip_assert_initialized!();
        unsafe { from_glib(ffi::gdk_keyval_is_lower(self.into_glib())) }
    }

    #[doc(alias = "gdk_keyval_to_upper")]
    #[must_use]
    pub fn to_upper(&self) -> Self {
        skip_assert_initialized!();
        unsafe { from_glib(ffi::gdk_keyval_to_upper(self.into_glib())) }
    }

    #[doc(alias = "gdk_keyval_to_lower")]
    #[must_use]
    pub fn to_lower(&self) -> Self {
        skip_assert_initialized!();
        unsafe { from_glib(ffi::gdk_keyval_to_lower(self.into_glib())) }
    }

    #[doc(alias = "GDK_KEY_0")]
    pub const _0: Self = Self(ffi::GDK_KEY_0 as u32);
    #[doc(alias = "GDK_KEY_1")]
    pub const _1: Self = Self(ffi::GDK_KEY_1 as u32);
    #[doc(alias = "GDK_KEY_10ChannelsDown")]
    pub const _10ChannelsDown: Self = Self(ffi::GDK_KEY_10ChannelsDown as u32);
    #[doc(alias = "GDK_KEY_10ChannelsUp")]
    pub const _10ChannelsUp: Self = Self(ffi::GDK_KEY_10ChannelsUp as u32);
    #[doc(alias = "GDK_KEY_2")]
    pub const _2: Self = Self(ffi::GDK_KEY_2 as u32);
    #[doc(alias = "GDK_KEY_3")]
    pub const _3: Self = Self(ffi::GDK_KEY_3 as u32);
    #[doc(alias = "GDK_KEY_3270_AltCursor")]
    pub const _3270_AltCursor: Self = Self(ffi::GDK_KEY_3270_AltCursor as u32);
    #[doc(alias = "GDK_KEY_3270_Attn")]
    pub const _3270_Attn: Self = Self(ffi::GDK_KEY_3270_Attn as u32);
    #[doc(alias = "GDK_KEY_3270_BackTab")]
    pub const _3270_BackTab: Self = Self(ffi::GDK_KEY_3270_BackTab as u32);
    #[doc(alias = "GDK_KEY_3270_ChangeScreen")]
    pub const _3270_ChangeScreen: Self = Self(ffi::GDK_KEY_3270_ChangeScreen as u32);
    #[doc(alias = "GDK_KEY_3270_Copy")]
    pub const _3270_Copy: Self = Self(ffi::GDK_KEY_3270_Copy as u32);
    #[doc(alias = "GDK_KEY_3270_CursorBlink")]
    pub const _3270_CursorBlink: Self = Self(ffi::GDK_KEY_3270_CursorBlink as u32);
    #[doc(alias = "GDK_KEY_3270_CursorSelect")]
    pub const _3270_CursorSelect: Self = Self(ffi::GDK_KEY_3270_CursorSelect as u32);
    #[doc(alias = "GDK_KEY_3270_DeleteWord")]
    pub const _3270_DeleteWord: Self = Self(ffi::GDK_KEY_3270_DeleteWord as u32);
    #[doc(alias = "GDK_KEY_3270_Duplicate")]
    pub const _3270_Duplicate: Self = Self(ffi::GDK_KEY_3270_Duplicate as u32);
    #[doc(alias = "GDK_KEY_3270_Enter")]
    pub const _3270_Enter: Self = Self(ffi::GDK_KEY_3270_Enter as u32);
    #[doc(alias = "GDK_KEY_3270_EraseEOF")]
    pub const _3270_EraseEOF: Self = Self(ffi::GDK_KEY_3270_EraseEOF as u32);
    #[doc(alias = "GDK_KEY_3270_EraseInput")]
    pub const _3270_EraseInput: Self = Self(ffi::GDK_KEY_3270_EraseInput as u32);
    #[doc(alias = "GDK_KEY_3270_ExSelect")]
    pub const _3270_ExSelect: Self = Self(ffi::GDK_KEY_3270_ExSelect as u32);
    #[doc(alias = "GDK_KEY_3270_FieldMark")]
    pub const _3270_FieldMark: Self = Self(ffi::GDK_KEY_3270_FieldMark as u32);
    #[doc(alias = "GDK_KEY_3270_Ident")]
    pub const _3270_Ident: Self = Self(ffi::GDK_KEY_3270_Ident as u32);
    #[doc(alias = "GDK_KEY_3270_Jump")]
    pub const _3270_Jump: Self = Self(ffi::GDK_KEY_3270_Jump as u32);
    #[doc(alias = "GDK_KEY_3270_KeyClick")]
    pub const _3270_KeyClick: Self = Self(ffi::GDK_KEY_3270_KeyClick as u32);
    #[doc(alias = "GDK_KEY_3270_Left2")]
    pub const _3270_Left2: Self = Self(ffi::GDK_KEY_3270_Left2 as u32);
    #[doc(alias = "GDK_KEY_3270_PA1")]
    pub const _3270_PA1: Self = Self(ffi::GDK_KEY_3270_PA1 as u32);
    #[doc(alias = "GDK_KEY_3270_PA2")]
    pub const _3270_PA2: Self = Self(ffi::GDK_KEY_3270_PA2 as u32);
    #[doc(alias = "GDK_KEY_3270_PA3")]
    pub const _3270_PA3: Self = Self(ffi::GDK_KEY_3270_PA3 as u32);
    #[doc(alias = "GDK_KEY_3270_Play")]
    pub const _3270_Play: Self = Self(ffi::GDK_KEY_3270_Play as u32);
    #[doc(alias = "GDK_KEY_3270_PrintScreen")]
    pub const _3270_PrintScreen: Self = Self(ffi::GDK_KEY_3270_PrintScreen as u32);
    #[doc(alias = "GDK_KEY_3270_Quit")]
    pub const _3270_Quit: Self = Self(ffi::GDK_KEY_3270_Quit as u32);
    #[doc(alias = "GDK_KEY_3270_Record")]
    pub const _3270_Record: Self = Self(ffi::GDK_KEY_3270_Record as u32);
    #[doc(alias = "GDK_KEY_3270_Reset")]
    pub const _3270_Reset: Self = Self(ffi::GDK_KEY_3270_Reset as u32);
    #[doc(alias = "GDK_KEY_3270_Right2")]
    pub const _3270_Right2: Self = Self(ffi::GDK_KEY_3270_Right2 as u32);
    #[doc(alias = "GDK_KEY_3270_Rule")]
    pub const _3270_Rule: Self = Self(ffi::GDK_KEY_3270_Rule as u32);
    #[doc(alias = "GDK_KEY_3270_Setup")]
    pub const _3270_Setup: Self = Self(ffi::GDK_KEY_3270_Setup as u32);
    #[doc(alias = "GDK_KEY_3270_Test")]
    pub const _3270_Test: Self = Self(ffi::GDK_KEY_3270_Test as u32);
    #[doc(alias = "GDK_KEY_3DMode")]
    pub const _3DMode: Self = Self(ffi::GDK_KEY_3DMode as u32);
    #[doc(alias = "GDK_KEY_4")]
    pub const _4: Self = Self(ffi::GDK_KEY_4 as u32);
    #[doc(alias = "GDK_KEY_5")]
    pub const _5: Self = Self(ffi::GDK_KEY_5 as u32);
    #[doc(alias = "GDK_KEY_6")]
    pub const _6: Self = Self(ffi::GDK_KEY_6 as u32);
    #[doc(alias = "GDK_KEY_7")]
    pub const _7: Self = Self(ffi::GDK_KEY_7 as u32);
    #[doc(alias = "GDK_KEY_8")]
    pub const _8: Self = Self(ffi::GDK_KEY_8 as u32);
    #[doc(alias = "GDK_KEY_9")]
    pub const _9: Self = Self(ffi::GDK_KEY_9 as u32);
    #[doc(alias = "GDK_KEY_A")]
    pub const A: Self = Self(ffi::GDK_KEY_A as u32);
    #[doc(alias = "GDK_KEY_AE")]
    pub const AE: Self = Self(ffi::GDK_KEY_AE as u32);
    #[doc(alias = "GDK_KEY_ALSToggle")]
    pub const ALSToggle: Self = Self(ffi::GDK_KEY_ALSToggle as u32);
    #[doc(alias = "GDK_KEY_Aacute")]
    pub const Aacute: Self = Self(ffi::GDK_KEY_Aacute as u32);
    #[doc(alias = "GDK_KEY_Abelowdot")]
    pub const Abelowdot: Self = Self(ffi::GDK_KEY_Abelowdot as u32);
    #[doc(alias = "GDK_KEY_Abreve")]
    pub const Abreve: Self = Self(ffi::GDK_KEY_Abreve as u32);
    #[doc(alias = "GDK_KEY_Abreveacute")]
    pub const Abreveacute: Self = Self(ffi::GDK_KEY_Abreveacute as u32);
    #[doc(alias = "GDK_KEY_Abrevebelowdot")]
    pub const Abrevebelowdot: Self = Self(ffi::GDK_KEY_Abrevebelowdot as u32);
    #[doc(alias = "GDK_KEY_Abrevegrave")]
    pub const Abrevegrave: Self = Self(ffi::GDK_KEY_Abrevegrave as u32);
    #[doc(alias = "GDK_KEY_Abrevehook")]
    pub const Abrevehook: Self = Self(ffi::GDK_KEY_Abrevehook as u32);
    #[doc(alias = "GDK_KEY_Abrevetilde")]
    pub const Abrevetilde: Self = Self(ffi::GDK_KEY_Abrevetilde as u32);
    #[doc(alias = "GDK_KEY_AccessX_Enable")]
    pub const AccessX_Enable: Self = Self(ffi::GDK_KEY_AccessX_Enable as u32);
    #[doc(alias = "GDK_KEY_AccessX_Feedback_Enable")]
    pub const AccessX_Feedback_Enable: Self = Self(ffi::GDK_KEY_AccessX_Feedback_Enable as u32);
    #[doc(alias = "GDK_KEY_Accessibility")]
    pub const Accessibility: Self = Self(ffi::GDK_KEY_Accessibility as u32);
    #[doc(alias = "GDK_KEY_Acircumflex")]
    pub const Acircumflex: Self = Self(ffi::GDK_KEY_Acircumflex as u32);
    #[doc(alias = "GDK_KEY_Acircumflexacute")]
    pub const Acircumflexacute: Self = Self(ffi::GDK_KEY_Acircumflexacute as u32);
    #[doc(alias = "GDK_KEY_Acircumflexbelowdot")]
    pub const Acircumflexbelowdot: Self = Self(ffi::GDK_KEY_Acircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_Acircumflexgrave")]
    pub const Acircumflexgrave: Self = Self(ffi::GDK_KEY_Acircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_Acircumflexhook")]
    pub const Acircumflexhook: Self = Self(ffi::GDK_KEY_Acircumflexhook as u32);
    #[doc(alias = "GDK_KEY_Acircumflextilde")]
    pub const Acircumflextilde: Self = Self(ffi::GDK_KEY_Acircumflextilde as u32);
    #[doc(alias = "GDK_KEY_AddFavorite")]
    pub const AddFavorite: Self = Self(ffi::GDK_KEY_AddFavorite as u32);
    #[doc(alias = "GDK_KEY_Addressbook")]
    pub const Addressbook: Self = Self(ffi::GDK_KEY_Addressbook as u32);
    #[doc(alias = "GDK_KEY_Adiaeresis")]
    pub const Adiaeresis: Self = Self(ffi::GDK_KEY_Adiaeresis as u32);
    #[doc(alias = "GDK_KEY_Agrave")]
    pub const Agrave: Self = Self(ffi::GDK_KEY_Agrave as u32);
    #[doc(alias = "GDK_KEY_Ahook")]
    pub const Ahook: Self = Self(ffi::GDK_KEY_Ahook as u32);
    #[doc(alias = "GDK_KEY_Alt_L")]
    pub const Alt_L: Self = Self(ffi::GDK_KEY_Alt_L as u32);
    #[doc(alias = "GDK_KEY_Alt_R")]
    pub const Alt_R: Self = Self(ffi::GDK_KEY_Alt_R as u32);
    #[doc(alias = "GDK_KEY_Amacron")]
    pub const Amacron: Self = Self(ffi::GDK_KEY_Amacron as u32);
    #[doc(alias = "GDK_KEY_Aogonek")]
    pub const Aogonek: Self = Self(ffi::GDK_KEY_Aogonek as u32);
    #[doc(alias = "GDK_KEY_AppSelect")]
    pub const AppSelect: Self = Self(ffi::GDK_KEY_AppSelect as u32);
    #[doc(alias = "GDK_KEY_ApplicationLeft")]
    pub const ApplicationLeft: Self = Self(ffi::GDK_KEY_ApplicationLeft as u32);
    #[doc(alias = "GDK_KEY_ApplicationRight")]
    pub const ApplicationRight: Self = Self(ffi::GDK_KEY_ApplicationRight as u32);
    #[doc(alias = "GDK_KEY_Arabic_0")]
    pub const Arabic_0: Self = Self(ffi::GDK_KEY_Arabic_0 as u32);
    #[doc(alias = "GDK_KEY_Arabic_1")]
    pub const Arabic_1: Self = Self(ffi::GDK_KEY_Arabic_1 as u32);
    #[doc(alias = "GDK_KEY_Arabic_2")]
    pub const Arabic_2: Self = Self(ffi::GDK_KEY_Arabic_2 as u32);
    #[doc(alias = "GDK_KEY_Arabic_3")]
    pub const Arabic_3: Self = Self(ffi::GDK_KEY_Arabic_3 as u32);
    #[doc(alias = "GDK_KEY_Arabic_4")]
    pub const Arabic_4: Self = Self(ffi::GDK_KEY_Arabic_4 as u32);
    #[doc(alias = "GDK_KEY_Arabic_5")]
    pub const Arabic_5: Self = Self(ffi::GDK_KEY_Arabic_5 as u32);
    #[doc(alias = "GDK_KEY_Arabic_6")]
    pub const Arabic_6: Self = Self(ffi::GDK_KEY_Arabic_6 as u32);
    #[doc(alias = "GDK_KEY_Arabic_7")]
    pub const Arabic_7: Self = Self(ffi::GDK_KEY_Arabic_7 as u32);
    #[doc(alias = "GDK_KEY_Arabic_8")]
    pub const Arabic_8: Self = Self(ffi::GDK_KEY_Arabic_8 as u32);
    #[doc(alias = "GDK_KEY_Arabic_9")]
    pub const Arabic_9: Self = Self(ffi::GDK_KEY_Arabic_9 as u32);
    #[doc(alias = "GDK_KEY_Arabic_ain")]
    pub const Arabic_ain: Self = Self(ffi::GDK_KEY_Arabic_ain as u32);
    #[doc(alias = "GDK_KEY_Arabic_alef")]
    pub const Arabic_alef: Self = Self(ffi::GDK_KEY_Arabic_alef as u32);
    #[doc(alias = "GDK_KEY_Arabic_alefmaksura")]
    pub const Arabic_alefmaksura: Self = Self(ffi::GDK_KEY_Arabic_alefmaksura as u32);
    #[doc(alias = "GDK_KEY_Arabic_beh")]
    pub const Arabic_beh: Self = Self(ffi::GDK_KEY_Arabic_beh as u32);
    #[doc(alias = "GDK_KEY_Arabic_comma")]
    pub const Arabic_comma: Self = Self(ffi::GDK_KEY_Arabic_comma as u32);
    #[doc(alias = "GDK_KEY_Arabic_dad")]
    pub const Arabic_dad: Self = Self(ffi::GDK_KEY_Arabic_dad as u32);
    #[doc(alias = "GDK_KEY_Arabic_dal")]
    pub const Arabic_dal: Self = Self(ffi::GDK_KEY_Arabic_dal as u32);
    #[doc(alias = "GDK_KEY_Arabic_damma")]
    pub const Arabic_damma: Self = Self(ffi::GDK_KEY_Arabic_damma as u32);
    #[doc(alias = "GDK_KEY_Arabic_dammatan")]
    pub const Arabic_dammatan: Self = Self(ffi::GDK_KEY_Arabic_dammatan as u32);
    #[doc(alias = "GDK_KEY_Arabic_ddal")]
    pub const Arabic_ddal: Self = Self(ffi::GDK_KEY_Arabic_ddal as u32);
    #[doc(alias = "GDK_KEY_Arabic_farsi_yeh")]
    pub const Arabic_farsi_yeh: Self = Self(ffi::GDK_KEY_Arabic_farsi_yeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_fatha")]
    pub const Arabic_fatha: Self = Self(ffi::GDK_KEY_Arabic_fatha as u32);
    #[doc(alias = "GDK_KEY_Arabic_fathatan")]
    pub const Arabic_fathatan: Self = Self(ffi::GDK_KEY_Arabic_fathatan as u32);
    #[doc(alias = "GDK_KEY_Arabic_feh")]
    pub const Arabic_feh: Self = Self(ffi::GDK_KEY_Arabic_feh as u32);
    #[doc(alias = "GDK_KEY_Arabic_fullstop")]
    pub const Arabic_fullstop: Self = Self(ffi::GDK_KEY_Arabic_fullstop as u32);
    #[doc(alias = "GDK_KEY_Arabic_gaf")]
    pub const Arabic_gaf: Self = Self(ffi::GDK_KEY_Arabic_gaf as u32);
    #[doc(alias = "GDK_KEY_Arabic_ghain")]
    pub const Arabic_ghain: Self = Self(ffi::GDK_KEY_Arabic_ghain as u32);
    #[doc(alias = "GDK_KEY_Arabic_ha")]
    pub const Arabic_ha: Self = Self(ffi::GDK_KEY_Arabic_ha as u32);
    #[doc(alias = "GDK_KEY_Arabic_hah")]
    pub const Arabic_hah: Self = Self(ffi::GDK_KEY_Arabic_hah as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamza")]
    pub const Arabic_hamza: Self = Self(ffi::GDK_KEY_Arabic_hamza as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamza_above")]
    pub const Arabic_hamza_above: Self = Self(ffi::GDK_KEY_Arabic_hamza_above as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamza_below")]
    pub const Arabic_hamza_below: Self = Self(ffi::GDK_KEY_Arabic_hamza_below as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaonalef")]
    pub const Arabic_hamzaonalef: Self = Self(ffi::GDK_KEY_Arabic_hamzaonalef as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaonwaw")]
    pub const Arabic_hamzaonwaw: Self = Self(ffi::GDK_KEY_Arabic_hamzaonwaw as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaonyeh")]
    pub const Arabic_hamzaonyeh: Self = Self(ffi::GDK_KEY_Arabic_hamzaonyeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaunderalef")]
    pub const Arabic_hamzaunderalef: Self = Self(ffi::GDK_KEY_Arabic_hamzaunderalef as u32);
    #[doc(alias = "GDK_KEY_Arabic_heh")]
    pub const Arabic_heh: Self = Self(ffi::GDK_KEY_Arabic_heh as u32);
    #[doc(alias = "GDK_KEY_Arabic_heh_doachashmee")]
    pub const Arabic_heh_doachashmee: Self = Self(ffi::GDK_KEY_Arabic_heh_doachashmee as u32);
    #[doc(alias = "GDK_KEY_Arabic_heh_goal")]
    pub const Arabic_heh_goal: Self = Self(ffi::GDK_KEY_Arabic_heh_goal as u32);
    #[doc(alias = "GDK_KEY_Arabic_jeem")]
    pub const Arabic_jeem: Self = Self(ffi::GDK_KEY_Arabic_jeem as u32);
    #[doc(alias = "GDK_KEY_Arabic_jeh")]
    pub const Arabic_jeh: Self = Self(ffi::GDK_KEY_Arabic_jeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_kaf")]
    pub const Arabic_kaf: Self = Self(ffi::GDK_KEY_Arabic_kaf as u32);
    #[doc(alias = "GDK_KEY_Arabic_kasra")]
    pub const Arabic_kasra: Self = Self(ffi::GDK_KEY_Arabic_kasra as u32);
    #[doc(alias = "GDK_KEY_Arabic_kasratan")]
    pub const Arabic_kasratan: Self = Self(ffi::GDK_KEY_Arabic_kasratan as u32);
    #[doc(alias = "GDK_KEY_Arabic_keheh")]
    pub const Arabic_keheh: Self = Self(ffi::GDK_KEY_Arabic_keheh as u32);
    #[doc(alias = "GDK_KEY_Arabic_khah")]
    pub const Arabic_khah: Self = Self(ffi::GDK_KEY_Arabic_khah as u32);
    #[doc(alias = "GDK_KEY_Arabic_lam")]
    pub const Arabic_lam: Self = Self(ffi::GDK_KEY_Arabic_lam as u32);
    #[doc(alias = "GDK_KEY_Arabic_madda_above")]
    pub const Arabic_madda_above: Self = Self(ffi::GDK_KEY_Arabic_madda_above as u32);
    #[doc(alias = "GDK_KEY_Arabic_maddaonalef")]
    pub const Arabic_maddaonalef: Self = Self(ffi::GDK_KEY_Arabic_maddaonalef as u32);
    #[doc(alias = "GDK_KEY_Arabic_meem")]
    pub const Arabic_meem: Self = Self(ffi::GDK_KEY_Arabic_meem as u32);
    #[doc(alias = "GDK_KEY_Arabic_noon")]
    pub const Arabic_noon: Self = Self(ffi::GDK_KEY_Arabic_noon as u32);
    #[doc(alias = "GDK_KEY_Arabic_noon_ghunna")]
    pub const Arabic_noon_ghunna: Self = Self(ffi::GDK_KEY_Arabic_noon_ghunna as u32);
    #[doc(alias = "GDK_KEY_Arabic_peh")]
    pub const Arabic_peh: Self = Self(ffi::GDK_KEY_Arabic_peh as u32);
    #[doc(alias = "GDK_KEY_Arabic_percent")]
    pub const Arabic_percent: Self = Self(ffi::GDK_KEY_Arabic_percent as u32);
    #[doc(alias = "GDK_KEY_Arabic_qaf")]
    pub const Arabic_qaf: Self = Self(ffi::GDK_KEY_Arabic_qaf as u32);
    #[doc(alias = "GDK_KEY_Arabic_question_mark")]
    pub const Arabic_question_mark: Self = Self(ffi::GDK_KEY_Arabic_question_mark as u32);
    #[doc(alias = "GDK_KEY_Arabic_ra")]
    pub const Arabic_ra: Self = Self(ffi::GDK_KEY_Arabic_ra as u32);
    #[doc(alias = "GDK_KEY_Arabic_rreh")]
    pub const Arabic_rreh: Self = Self(ffi::GDK_KEY_Arabic_rreh as u32);
    #[doc(alias = "GDK_KEY_Arabic_sad")]
    pub const Arabic_sad: Self = Self(ffi::GDK_KEY_Arabic_sad as u32);
    #[doc(alias = "GDK_KEY_Arabic_seen")]
    pub const Arabic_seen: Self = Self(ffi::GDK_KEY_Arabic_seen as u32);
    #[doc(alias = "GDK_KEY_Arabic_semicolon")]
    pub const Arabic_semicolon: Self = Self(ffi::GDK_KEY_Arabic_semicolon as u32);
    #[doc(alias = "GDK_KEY_Arabic_shadda")]
    pub const Arabic_shadda: Self = Self(ffi::GDK_KEY_Arabic_shadda as u32);
    #[doc(alias = "GDK_KEY_Arabic_sheen")]
    pub const Arabic_sheen: Self = Self(ffi::GDK_KEY_Arabic_sheen as u32);
    #[doc(alias = "GDK_KEY_Arabic_sukun")]
    pub const Arabic_sukun: Self = Self(ffi::GDK_KEY_Arabic_sukun as u32);
    #[doc(alias = "GDK_KEY_Arabic_superscript_alef")]
    pub const Arabic_superscript_alef: Self = Self(ffi::GDK_KEY_Arabic_superscript_alef as u32);
    #[doc(alias = "GDK_KEY_Arabic_switch")]
    pub const Arabic_switch: Self = Self(ffi::GDK_KEY_Arabic_switch as u32);
    #[doc(alias = "GDK_KEY_Arabic_tah")]
    pub const Arabic_tah: Self = Self(ffi::GDK_KEY_Arabic_tah as u32);
    #[doc(alias = "GDK_KEY_Arabic_tatweel")]
    pub const Arabic_tatweel: Self = Self(ffi::GDK_KEY_Arabic_tatweel as u32);
    #[doc(alias = "GDK_KEY_Arabic_tcheh")]
    pub const Arabic_tcheh: Self = Self(ffi::GDK_KEY_Arabic_tcheh as u32);
    #[doc(alias = "GDK_KEY_Arabic_teh")]
    pub const Arabic_teh: Self = Self(ffi::GDK_KEY_Arabic_teh as u32);
    #[doc(alias = "GDK_KEY_Arabic_tehmarbuta")]
    pub const Arabic_tehmarbuta: Self = Self(ffi::GDK_KEY_Arabic_tehmarbuta as u32);
    #[doc(alias = "GDK_KEY_Arabic_thal")]
    pub const Arabic_thal: Self = Self(ffi::GDK_KEY_Arabic_thal as u32);
    #[doc(alias = "GDK_KEY_Arabic_theh")]
    pub const Arabic_theh: Self = Self(ffi::GDK_KEY_Arabic_theh as u32);
    #[doc(alias = "GDK_KEY_Arabic_tteh")]
    pub const Arabic_tteh: Self = Self(ffi::GDK_KEY_Arabic_tteh as u32);
    #[doc(alias = "GDK_KEY_Arabic_veh")]
    pub const Arabic_veh: Self = Self(ffi::GDK_KEY_Arabic_veh as u32);
    #[doc(alias = "GDK_KEY_Arabic_waw")]
    pub const Arabic_waw: Self = Self(ffi::GDK_KEY_Arabic_waw as u32);
    #[doc(alias = "GDK_KEY_Arabic_yeh")]
    pub const Arabic_yeh: Self = Self(ffi::GDK_KEY_Arabic_yeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_yeh_baree")]
    pub const Arabic_yeh_baree: Self = Self(ffi::GDK_KEY_Arabic_yeh_baree as u32);
    #[doc(alias = "GDK_KEY_Arabic_zah")]
    pub const Arabic_zah: Self = Self(ffi::GDK_KEY_Arabic_zah as u32);
    #[doc(alias = "GDK_KEY_Arabic_zain")]
    pub const Arabic_zain: Self = Self(ffi::GDK_KEY_Arabic_zain as u32);
    #[doc(alias = "GDK_KEY_Aring")]
    pub const Aring: Self = Self(ffi::GDK_KEY_Aring as u32);
    #[doc(alias = "GDK_KEY_Armenian_AT")]
    pub const Armenian_AT: Self = Self(ffi::GDK_KEY_Armenian_AT as u32);
    #[doc(alias = "GDK_KEY_Armenian_AYB")]
    pub const Armenian_AYB: Self = Self(ffi::GDK_KEY_Armenian_AYB as u32);
    #[doc(alias = "GDK_KEY_Armenian_BEN")]
    pub const Armenian_BEN: Self = Self(ffi::GDK_KEY_Armenian_BEN as u32);
    #[doc(alias = "GDK_KEY_Armenian_CHA")]
    pub const Armenian_CHA: Self = Self(ffi::GDK_KEY_Armenian_CHA as u32);
    #[doc(alias = "GDK_KEY_Armenian_DA")]
    pub const Armenian_DA: Self = Self(ffi::GDK_KEY_Armenian_DA as u32);
    #[doc(alias = "GDK_KEY_Armenian_DZA")]
    pub const Armenian_DZA: Self = Self(ffi::GDK_KEY_Armenian_DZA as u32);
    #[doc(alias = "GDK_KEY_Armenian_E")]
    pub const Armenian_E: Self = Self(ffi::GDK_KEY_Armenian_E as u32);
    #[doc(alias = "GDK_KEY_Armenian_FE")]
    pub const Armenian_FE: Self = Self(ffi::GDK_KEY_Armenian_FE as u32);
    #[doc(alias = "GDK_KEY_Armenian_GHAT")]
    pub const Armenian_GHAT: Self = Self(ffi::GDK_KEY_Armenian_GHAT as u32);
    #[doc(alias = "GDK_KEY_Armenian_GIM")]
    pub const Armenian_GIM: Self = Self(ffi::GDK_KEY_Armenian_GIM as u32);
    #[doc(alias = "GDK_KEY_Armenian_HI")]
    pub const Armenian_HI: Self = Self(ffi::GDK_KEY_Armenian_HI as u32);
    #[doc(alias = "GDK_KEY_Armenian_HO")]
    pub const Armenian_HO: Self = Self(ffi::GDK_KEY_Armenian_HO as u32);
    #[doc(alias = "GDK_KEY_Armenian_INI")]
    pub const Armenian_INI: Self = Self(ffi::GDK_KEY_Armenian_INI as u32);
    #[doc(alias = "GDK_KEY_Armenian_JE")]
    pub const Armenian_JE: Self = Self(ffi::GDK_KEY_Armenian_JE as u32);
    #[doc(alias = "GDK_KEY_Armenian_KE")]
    pub const Armenian_KE: Self = Self(ffi::GDK_KEY_Armenian_KE as u32);
    #[doc(alias = "GDK_KEY_Armenian_KEN")]
    pub const Armenian_KEN: Self = Self(ffi::GDK_KEY_Armenian_KEN as u32);
    #[doc(alias = "GDK_KEY_Armenian_KHE")]
    pub const Armenian_KHE: Self = Self(ffi::GDK_KEY_Armenian_KHE as u32);
    #[doc(alias = "GDK_KEY_Armenian_LYUN")]
    pub const Armenian_LYUN: Self = Self(ffi::GDK_KEY_Armenian_LYUN as u32);
    #[doc(alias = "GDK_KEY_Armenian_MEN")]
    pub const Armenian_MEN: Self = Self(ffi::GDK_KEY_Armenian_MEN as u32);
    #[doc(alias = "GDK_KEY_Armenian_NU")]
    pub const Armenian_NU: Self = Self(ffi::GDK_KEY_Armenian_NU as u32);
    #[doc(alias = "GDK_KEY_Armenian_O")]
    pub const Armenian_O: Self = Self(ffi::GDK_KEY_Armenian_O as u32);
    #[doc(alias = "GDK_KEY_Armenian_PE")]
    pub const Armenian_PE: Self = Self(ffi::GDK_KEY_Armenian_PE as u32);
    #[doc(alias = "GDK_KEY_Armenian_PYUR")]
    pub const Armenian_PYUR: Self = Self(ffi::GDK_KEY_Armenian_PYUR as u32);
    #[doc(alias = "GDK_KEY_Armenian_RA")]
    pub const Armenian_RA: Self = Self(ffi::GDK_KEY_Armenian_RA as u32);
    #[doc(alias = "GDK_KEY_Armenian_RE")]
    pub const Armenian_RE: Self = Self(ffi::GDK_KEY_Armenian_RE as u32);
    #[doc(alias = "GDK_KEY_Armenian_SE")]
    pub const Armenian_SE: Self = Self(ffi::GDK_KEY_Armenian_SE as u32);
    #[doc(alias = "GDK_KEY_Armenian_SHA")]
    pub const Armenian_SHA: Self = Self(ffi::GDK_KEY_Armenian_SHA as u32);
    #[doc(alias = "GDK_KEY_Armenian_TCHE")]
    pub const Armenian_TCHE: Self = Self(ffi::GDK_KEY_Armenian_TCHE as u32);
    #[doc(alias = "GDK_KEY_Armenian_TO")]
    pub const Armenian_TO: Self = Self(ffi::GDK_KEY_Armenian_TO as u32);
    #[doc(alias = "GDK_KEY_Armenian_TSA")]
    pub const Armenian_TSA: Self = Self(ffi::GDK_KEY_Armenian_TSA as u32);
    #[doc(alias = "GDK_KEY_Armenian_TSO")]
    pub const Armenian_TSO: Self = Self(ffi::GDK_KEY_Armenian_TSO as u32);
    #[doc(alias = "GDK_KEY_Armenian_TYUN")]
    pub const Armenian_TYUN: Self = Self(ffi::GDK_KEY_Armenian_TYUN as u32);
    #[doc(alias = "GDK_KEY_Armenian_VEV")]
    pub const Armenian_VEV: Self = Self(ffi::GDK_KEY_Armenian_VEV as u32);
    #[doc(alias = "GDK_KEY_Armenian_VO")]
    pub const Armenian_VO: Self = Self(ffi::GDK_KEY_Armenian_VO as u32);
    #[doc(alias = "GDK_KEY_Armenian_VYUN")]
    pub const Armenian_VYUN: Self = Self(ffi::GDK_KEY_Armenian_VYUN as u32);
    #[doc(alias = "GDK_KEY_Armenian_YECH")]
    pub const Armenian_YECH: Self = Self(ffi::GDK_KEY_Armenian_YECH as u32);
    #[doc(alias = "GDK_KEY_Armenian_ZA")]
    pub const Armenian_ZA: Self = Self(ffi::GDK_KEY_Armenian_ZA as u32);
    #[doc(alias = "GDK_KEY_Armenian_ZHE")]
    pub const Armenian_ZHE: Self = Self(ffi::GDK_KEY_Armenian_ZHE as u32);
    #[doc(alias = "GDK_KEY_Armenian_accent")]
    pub const Armenian_accent: Self = Self(ffi::GDK_KEY_Armenian_accent as u32);
    #[doc(alias = "GDK_KEY_Armenian_amanak")]
    pub const Armenian_amanak: Self = Self(ffi::GDK_KEY_Armenian_amanak as u32);
    #[doc(alias = "GDK_KEY_Armenian_apostrophe")]
    pub const Armenian_apostrophe: Self = Self(ffi::GDK_KEY_Armenian_apostrophe as u32);
    #[doc(alias = "GDK_KEY_Armenian_at")]
    pub const Armenian_at: Self = Self(ffi::GDK_KEY_Armenian_at as u32);
    #[doc(alias = "GDK_KEY_Armenian_ayb")]
    pub const Armenian_ayb: Self = Self(ffi::GDK_KEY_Armenian_ayb as u32);
    #[doc(alias = "GDK_KEY_Armenian_ben")]
    pub const Armenian_ben: Self = Self(ffi::GDK_KEY_Armenian_ben as u32);
    #[doc(alias = "GDK_KEY_Armenian_but")]
    pub const Armenian_but: Self = Self(ffi::GDK_KEY_Armenian_but as u32);
    #[doc(alias = "GDK_KEY_Armenian_cha")]
    pub const Armenian_cha: Self = Self(ffi::GDK_KEY_Armenian_cha as u32);
    #[doc(alias = "GDK_KEY_Armenian_da")]
    pub const Armenian_da: Self = Self(ffi::GDK_KEY_Armenian_da as u32);
    #[doc(alias = "GDK_KEY_Armenian_dza")]
    pub const Armenian_dza: Self = Self(ffi::GDK_KEY_Armenian_dza as u32);
    #[doc(alias = "GDK_KEY_Armenian_e")]
    pub const Armenian_e: Self = Self(ffi::GDK_KEY_Armenian_e as u32);
    #[doc(alias = "GDK_KEY_Armenian_exclam")]
    pub const Armenian_exclam: Self = Self(ffi::GDK_KEY_Armenian_exclam as u32);
    #[doc(alias = "GDK_KEY_Armenian_fe")]
    pub const Armenian_fe: Self = Self(ffi::GDK_KEY_Armenian_fe as u32);
    #[doc(alias = "GDK_KEY_Armenian_full_stop")]
    pub const Armenian_full_stop: Self = Self(ffi::GDK_KEY_Armenian_full_stop as u32);
    #[doc(alias = "GDK_KEY_Armenian_ghat")]
    pub const Armenian_ghat: Self = Self(ffi::GDK_KEY_Armenian_ghat as u32);
    #[doc(alias = "GDK_KEY_Armenian_gim")]
    pub const Armenian_gim: Self = Self(ffi::GDK_KEY_Armenian_gim as u32);
    #[doc(alias = "GDK_KEY_Armenian_hi")]
    pub const Armenian_hi: Self = Self(ffi::GDK_KEY_Armenian_hi as u32);
    #[doc(alias = "GDK_KEY_Armenian_ho")]
    pub const Armenian_ho: Self = Self(ffi::GDK_KEY_Armenian_ho as u32);
    #[doc(alias = "GDK_KEY_Armenian_hyphen")]
    pub const Armenian_hyphen: Self = Self(ffi::GDK_KEY_Armenian_hyphen as u32);
    #[doc(alias = "GDK_KEY_Armenian_ini")]
    pub const Armenian_ini: Self = Self(ffi::GDK_KEY_Armenian_ini as u32);
    #[doc(alias = "GDK_KEY_Armenian_je")]
    pub const Armenian_je: Self = Self(ffi::GDK_KEY_Armenian_je as u32);
    #[doc(alias = "GDK_KEY_Armenian_ke")]
    pub const Armenian_ke: Self = Self(ffi::GDK_KEY_Armenian_ke as u32);
    #[doc(alias = "GDK_KEY_Armenian_ken")]
    pub const Armenian_ken: Self = Self(ffi::GDK_KEY_Armenian_ken as u32);
    #[doc(alias = "GDK_KEY_Armenian_khe")]
    pub const Armenian_khe: Self = Self(ffi::GDK_KEY_Armenian_khe as u32);
    #[doc(alias = "GDK_KEY_Armenian_ligature_ew")]
    pub const Armenian_ligature_ew: Self = Self(ffi::GDK_KEY_Armenian_ligature_ew as u32);
    #[doc(alias = "GDK_KEY_Armenian_lyun")]
    pub const Armenian_lyun: Self = Self(ffi::GDK_KEY_Armenian_lyun as u32);
    #[doc(alias = "GDK_KEY_Armenian_men")]
    pub const Armenian_men: Self = Self(ffi::GDK_KEY_Armenian_men as u32);
    #[doc(alias = "GDK_KEY_Armenian_nu")]
    pub const Armenian_nu: Self = Self(ffi::GDK_KEY_Armenian_nu as u32);
    #[doc(alias = "GDK_KEY_Armenian_o")]
    pub const Armenian_o: Self = Self(ffi::GDK_KEY_Armenian_o as u32);
    #[doc(alias = "GDK_KEY_Armenian_paruyk")]
    pub const Armenian_paruyk: Self = Self(ffi::GDK_KEY_Armenian_paruyk as u32);
    #[doc(alias = "GDK_KEY_Armenian_pe")]
    pub const Armenian_pe: Self = Self(ffi::GDK_KEY_Armenian_pe as u32);
    #[doc(alias = "GDK_KEY_Armenian_pyur")]
    pub const Armenian_pyur: Self = Self(ffi::GDK_KEY_Armenian_pyur as u32);
    #[doc(alias = "GDK_KEY_Armenian_question")]
    pub const Armenian_question: Self = Self(ffi::GDK_KEY_Armenian_question as u32);
    #[doc(alias = "GDK_KEY_Armenian_ra")]
    pub const Armenian_ra: Self = Self(ffi::GDK_KEY_Armenian_ra as u32);
    #[doc(alias = "GDK_KEY_Armenian_re")]
    pub const Armenian_re: Self = Self(ffi::GDK_KEY_Armenian_re as u32);
    #[doc(alias = "GDK_KEY_Armenian_se")]
    pub const Armenian_se: Self = Self(ffi::GDK_KEY_Armenian_se as u32);
    #[doc(alias = "GDK_KEY_Armenian_separation_mark")]
    pub const Armenian_separation_mark: Self = Self(ffi::GDK_KEY_Armenian_separation_mark as u32);
    #[doc(alias = "GDK_KEY_Armenian_sha")]
    pub const Armenian_sha: Self = Self(ffi::GDK_KEY_Armenian_sha as u32);
    #[doc(alias = "GDK_KEY_Armenian_shesht")]
    pub const Armenian_shesht: Self = Self(ffi::GDK_KEY_Armenian_shesht as u32);
    #[doc(alias = "GDK_KEY_Armenian_tche")]
    pub const Armenian_tche: Self = Self(ffi::GDK_KEY_Armenian_tche as u32);
    #[doc(alias = "GDK_KEY_Armenian_to")]
    pub const Armenian_to: Self = Self(ffi::GDK_KEY_Armenian_to as u32);
    #[doc(alias = "GDK_KEY_Armenian_tsa")]
    pub const Armenian_tsa: Self = Self(ffi::GDK_KEY_Armenian_tsa as u32);
    #[doc(alias = "GDK_KEY_Armenian_tso")]
    pub const Armenian_tso: Self = Self(ffi::GDK_KEY_Armenian_tso as u32);
    #[doc(alias = "GDK_KEY_Armenian_tyun")]
    pub const Armenian_tyun: Self = Self(ffi::GDK_KEY_Armenian_tyun as u32);
    #[doc(alias = "GDK_KEY_Armenian_verjaket")]
    pub const Armenian_verjaket: Self = Self(ffi::GDK_KEY_Armenian_verjaket as u32);
    #[doc(alias = "GDK_KEY_Armenian_vev")]
    pub const Armenian_vev: Self = Self(ffi::GDK_KEY_Armenian_vev as u32);
    #[doc(alias = "GDK_KEY_Armenian_vo")]
    pub const Armenian_vo: Self = Self(ffi::GDK_KEY_Armenian_vo as u32);
    #[doc(alias = "GDK_KEY_Armenian_vyun")]
    pub const Armenian_vyun: Self = Self(ffi::GDK_KEY_Armenian_vyun as u32);
    #[doc(alias = "GDK_KEY_Armenian_yech")]
    pub const Armenian_yech: Self = Self(ffi::GDK_KEY_Armenian_yech as u32);
    #[doc(alias = "GDK_KEY_Armenian_yentamna")]
    pub const Armenian_yentamna: Self = Self(ffi::GDK_KEY_Armenian_yentamna as u32);
    #[doc(alias = "GDK_KEY_Armenian_za")]
    pub const Armenian_za: Self = Self(ffi::GDK_KEY_Armenian_za as u32);
    #[doc(alias = "GDK_KEY_Armenian_zhe")]
    pub const Armenian_zhe: Self = Self(ffi::GDK_KEY_Armenian_zhe as u32);
    #[doc(alias = "GDK_KEY_AspectRatio")]
    pub const AspectRatio: Self = Self(ffi::GDK_KEY_AspectRatio as u32);
    #[doc(alias = "GDK_KEY_Assistant")]
    pub const Assistant: Self = Self(ffi::GDK_KEY_Assistant as u32);
    #[doc(alias = "GDK_KEY_Atilde")]
    pub const Atilde: Self = Self(ffi::GDK_KEY_Atilde as u32);
    #[doc(alias = "GDK_KEY_AttendantOff")]
    pub const AttendantOff: Self = Self(ffi::GDK_KEY_AttendantOff as u32);
    #[doc(alias = "GDK_KEY_AttendantOn")]
    pub const AttendantOn: Self = Self(ffi::GDK_KEY_AttendantOn as u32);
    #[doc(alias = "GDK_KEY_AttendantToggle")]
    pub const AttendantToggle: Self = Self(ffi::GDK_KEY_AttendantToggle as u32);
    #[doc(alias = "GDK_KEY_AudibleBell_Enable")]
    pub const AudibleBell_Enable: Self = Self(ffi::GDK_KEY_AudibleBell_Enable as u32);
    #[doc(alias = "GDK_KEY_Audio")]
    pub const Audio: Self = Self(ffi::GDK_KEY_Audio as u32);
    #[doc(alias = "GDK_KEY_AudioCycleTrack")]
    pub const AudioCycleTrack: Self = Self(ffi::GDK_KEY_AudioCycleTrack as u32);
    #[doc(alias = "GDK_KEY_AudioDesc")]
    pub const AudioDesc: Self = Self(ffi::GDK_KEY_AudioDesc as u32);
    #[doc(alias = "GDK_KEY_AudioForward")]
    pub const AudioForward: Self = Self(ffi::GDK_KEY_AudioForward as u32);
    #[doc(alias = "GDK_KEY_AudioLowerVolume")]
    pub const AudioLowerVolume: Self = Self(ffi::GDK_KEY_AudioLowerVolume as u32);
    #[doc(alias = "GDK_KEY_AudioMedia")]
    pub const AudioMedia: Self = Self(ffi::GDK_KEY_AudioMedia as u32);
    #[doc(alias = "GDK_KEY_AudioMicMute")]
    pub const AudioMicMute: Self = Self(ffi::GDK_KEY_AudioMicMute as u32);
    #[doc(alias = "GDK_KEY_AudioMute")]
    pub const AudioMute: Self = Self(ffi::GDK_KEY_AudioMute as u32);
    #[doc(alias = "GDK_KEY_AudioNext")]
    pub const AudioNext: Self = Self(ffi::GDK_KEY_AudioNext as u32);
    #[doc(alias = "GDK_KEY_AudioPause")]
    pub const AudioPause: Self = Self(ffi::GDK_KEY_AudioPause as u32);
    #[doc(alias = "GDK_KEY_AudioPlay")]
    pub const AudioPlay: Self = Self(ffi::GDK_KEY_AudioPlay as u32);
    #[doc(alias = "GDK_KEY_AudioPreset")]
    pub const AudioPreset: Self = Self(ffi::GDK_KEY_AudioPreset as u32);
    #[doc(alias = "GDK_KEY_AudioPrev")]
    pub const AudioPrev: Self = Self(ffi::GDK_KEY_AudioPrev as u32);
    #[doc(alias = "GDK_KEY_AudioRaiseVolume")]
    pub const AudioRaiseVolume: Self = Self(ffi::GDK_KEY_AudioRaiseVolume as u32);
    #[doc(alias = "GDK_KEY_AudioRandomPlay")]
    pub const AudioRandomPlay: Self = Self(ffi::GDK_KEY_AudioRandomPlay as u32);
    #[doc(alias = "GDK_KEY_AudioRecord")]
    pub const AudioRecord: Self = Self(ffi::GDK_KEY_AudioRecord as u32);
    #[doc(alias = "GDK_KEY_AudioRepeat")]
    pub const AudioRepeat: Self = Self(ffi::GDK_KEY_AudioRepeat as u32);
    #[doc(alias = "GDK_KEY_AudioRewind")]
    pub const AudioRewind: Self = Self(ffi::GDK_KEY_AudioRewind as u32);
    #[doc(alias = "GDK_KEY_AudioStop")]
    pub const AudioStop: Self = Self(ffi::GDK_KEY_AudioStop as u32);
    #[doc(alias = "GDK_KEY_AutopilotEngageToggle")]
    pub const AutopilotEngageToggle: Self = Self(ffi::GDK_KEY_AutopilotEngageToggle as u32);
    #[doc(alias = "GDK_KEY_Away")]
    pub const Away: Self = Self(ffi::GDK_KEY_Away as u32);
    #[doc(alias = "GDK_KEY_B")]
    pub const B: Self = Self(ffi::GDK_KEY_B as u32);
    #[doc(alias = "GDK_KEY_Babovedot")]
    pub const Babovedot: Self = Self(ffi::GDK_KEY_Babovedot as u32);
    #[doc(alias = "GDK_KEY_Back")]
    pub const Back: Self = Self(ffi::GDK_KEY_Back as u32);
    #[doc(alias = "GDK_KEY_BackForward")]
    pub const BackForward: Self = Self(ffi::GDK_KEY_BackForward as u32);
    #[doc(alias = "GDK_KEY_BackSpace")]
    pub const BackSpace: Self = Self(ffi::GDK_KEY_BackSpace as u32);
    #[doc(alias = "GDK_KEY_Battery")]
    pub const Battery: Self = Self(ffi::GDK_KEY_Battery as u32);
    #[doc(alias = "GDK_KEY_Begin")]
    pub const Begin: Self = Self(ffi::GDK_KEY_Begin as u32);
    #[doc(alias = "GDK_KEY_Blue")]
    pub const Blue: Self = Self(ffi::GDK_KEY_Blue as u32);
    #[doc(alias = "GDK_KEY_Bluetooth")]
    pub const Bluetooth: Self = Self(ffi::GDK_KEY_Bluetooth as u32);
    #[doc(alias = "GDK_KEY_Book")]
    pub const Book: Self = Self(ffi::GDK_KEY_Book as u32);
    #[doc(alias = "GDK_KEY_BounceKeys_Enable")]
    pub const BounceKeys_Enable: Self = Self(ffi::GDK_KEY_BounceKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_Break")]
    pub const Break: Self = Self(ffi::GDK_KEY_Break as u32);
    #[doc(alias = "GDK_KEY_BrightnessAdjust")]
    pub const BrightnessAdjust: Self = Self(ffi::GDK_KEY_BrightnessAdjust as u32);
    #[doc(alias = "GDK_KEY_BrightnessAuto")]
    pub const BrightnessAuto: Self = Self(ffi::GDK_KEY_BrightnessAuto as u32);
    #[doc(alias = "GDK_KEY_BrightnessMax")]
    pub const BrightnessMax: Self = Self(ffi::GDK_KEY_BrightnessMax as u32);
    #[doc(alias = "GDK_KEY_BrightnessMin")]
    pub const BrightnessMin: Self = Self(ffi::GDK_KEY_BrightnessMin as u32);
    #[doc(alias = "GDK_KEY_Buttonconfig")]
    pub const Buttonconfig: Self = Self(ffi::GDK_KEY_Buttonconfig as u32);
    #[doc(alias = "GDK_KEY_Byelorussian_SHORTU")]
    pub const Byelorussian_SHORTU: Self = Self(ffi::GDK_KEY_Byelorussian_SHORTU as u32);
    #[doc(alias = "GDK_KEY_Byelorussian_shortu")]
    pub const Byelorussian_shortu: Self = Self(ffi::GDK_KEY_Byelorussian_shortu as u32);
    #[doc(alias = "GDK_KEY_C")]
    pub const C: Self = Self(ffi::GDK_KEY_C as u32);
    #[doc(alias = "GDK_KEY_CD")]
    pub const CD: Self = Self(ffi::GDK_KEY_CD as u32);
    #[doc(alias = "GDK_KEY_CH")]
    pub const CH: Self = Self(ffi::GDK_KEY_CH as u32);
    #[doc(alias = "GDK_KEY_C_H")]
    pub const C_H: Self = Self(ffi::GDK_KEY_C_H as u32);
    #[doc(alias = "GDK_KEY_C_h")]
    pub const C_h: Self = Self(ffi::GDK_KEY_C_h as u32);
    #[doc(alias = "GDK_KEY_Cabovedot")]
    pub const Cabovedot: Self = Self(ffi::GDK_KEY_Cabovedot as u32);
    #[doc(alias = "GDK_KEY_Cacute")]
    pub const Cacute: Self = Self(ffi::GDK_KEY_Cacute as u32);
    #[doc(alias = "GDK_KEY_Calculator")]
    pub const Calculator: Self = Self(ffi::GDK_KEY_Calculator as u32);
    #[doc(alias = "GDK_KEY_Calendar")]
    pub const Calendar: Self = Self(ffi::GDK_KEY_Calendar as u32);
    #[doc(alias = "GDK_KEY_CameraAccessDisable")]
    pub const CameraAccessDisable: Self = Self(ffi::GDK_KEY_CameraAccessDisable as u32);
    #[doc(alias = "GDK_KEY_CameraAccessEnable")]
    pub const CameraAccessEnable: Self = Self(ffi::GDK_KEY_CameraAccessEnable as u32);
    #[doc(alias = "GDK_KEY_CameraAccessToggle")]
    pub const CameraAccessToggle: Self = Self(ffi::GDK_KEY_CameraAccessToggle as u32);
    #[doc(alias = "GDK_KEY_CameraDown")]
    pub const CameraDown: Self = Self(ffi::GDK_KEY_CameraDown as u32);
    #[doc(alias = "GDK_KEY_CameraFocus")]
    pub const CameraFocus: Self = Self(ffi::GDK_KEY_CameraFocus as u32);
    #[doc(alias = "GDK_KEY_CameraLeft")]
    pub const CameraLeft: Self = Self(ffi::GDK_KEY_CameraLeft as u32);
    #[doc(alias = "GDK_KEY_CameraRight")]
    pub const CameraRight: Self = Self(ffi::GDK_KEY_CameraRight as u32);
    #[doc(alias = "GDK_KEY_CameraUp")]
    pub const CameraUp: Self = Self(ffi::GDK_KEY_CameraUp as u32);
    #[doc(alias = "GDK_KEY_CameraZoomIn")]
    pub const CameraZoomIn: Self = Self(ffi::GDK_KEY_CameraZoomIn as u32);
    #[doc(alias = "GDK_KEY_CameraZoomOut")]
    pub const CameraZoomOut: Self = Self(ffi::GDK_KEY_CameraZoomOut as u32);
    #[doc(alias = "GDK_KEY_Cancel")]
    pub const Cancel: Self = Self(ffi::GDK_KEY_Cancel as u32);
    #[doc(alias = "GDK_KEY_Caps_Lock")]
    pub const Caps_Lock: Self = Self(ffi::GDK_KEY_Caps_Lock as u32);
    #[doc(alias = "GDK_KEY_Ccaron")]
    pub const Ccaron: Self = Self(ffi::GDK_KEY_Ccaron as u32);
    #[doc(alias = "GDK_KEY_Ccedilla")]
    pub const Ccedilla: Self = Self(ffi::GDK_KEY_Ccedilla as u32);
    #[doc(alias = "GDK_KEY_Ccircumflex")]
    pub const Ccircumflex: Self = Self(ffi::GDK_KEY_Ccircumflex as u32);
    #[doc(alias = "GDK_KEY_Ch")]
    pub const Ch: Self = Self(ffi::GDK_KEY_Ch as u32);
    #[doc(alias = "GDK_KEY_ChannelDown")]
    pub const ChannelDown: Self = Self(ffi::GDK_KEY_ChannelDown as u32);
    #[doc(alias = "GDK_KEY_ChannelUp")]
    pub const ChannelUp: Self = Self(ffi::GDK_KEY_ChannelUp as u32);
    #[doc(alias = "GDK_KEY_Clear")]
    pub const Clear: Self = Self(ffi::GDK_KEY_Clear as u32);
    #[doc(alias = "GDK_KEY_ClearGrab")]
    pub const ClearGrab: Self = Self(ffi::GDK_KEY_ClearGrab as u32);
    #[doc(alias = "GDK_KEY_ClearvuSonar")]
    pub const ClearvuSonar: Self = Self(ffi::GDK_KEY_ClearvuSonar as u32);
    #[doc(alias = "GDK_KEY_Close")]
    pub const Close: Self = Self(ffi::GDK_KEY_Close as u32);
    #[doc(alias = "GDK_KEY_Codeinput")]
    pub const Codeinput: Self = Self(ffi::GDK_KEY_Codeinput as u32);
    #[doc(alias = "GDK_KEY_ColonSign")]
    pub const ColonSign: Self = Self(ffi::GDK_KEY_ColonSign as u32);
    #[doc(alias = "GDK_KEY_Community")]
    pub const Community: Self = Self(ffi::GDK_KEY_Community as u32);
    #[doc(alias = "GDK_KEY_ContextMenu")]
    pub const ContextMenu: Self = Self(ffi::GDK_KEY_ContextMenu as u32);
    #[doc(alias = "GDK_KEY_ContrastAdjust")]
    pub const ContrastAdjust: Self = Self(ffi::GDK_KEY_ContrastAdjust as u32);
    #[doc(alias = "GDK_KEY_ControlPanel")]
    pub const ControlPanel: Self = Self(ffi::GDK_KEY_ControlPanel as u32);
    #[doc(alias = "GDK_KEY_Control_L")]
    pub const Control_L: Self = Self(ffi::GDK_KEY_Control_L as u32);
    #[doc(alias = "GDK_KEY_Control_R")]
    pub const Control_R: Self = Self(ffi::GDK_KEY_Control_R as u32);
    #[doc(alias = "GDK_KEY_Copy")]
    pub const Copy: Self = Self(ffi::GDK_KEY_Copy as u32);
    #[doc(alias = "GDK_KEY_CruzeiroSign")]
    pub const CruzeiroSign: Self = Self(ffi::GDK_KEY_CruzeiroSign as u32);
    #[doc(alias = "GDK_KEY_Cut")]
    pub const Cut: Self = Self(ffi::GDK_KEY_Cut as u32);
    #[doc(alias = "GDK_KEY_CycleAngle")]
    pub const CycleAngle: Self = Self(ffi::GDK_KEY_CycleAngle as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_A")]
    pub const Cyrillic_A: Self = Self(ffi::GDK_KEY_Cyrillic_A as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_BE")]
    pub const Cyrillic_BE: Self = Self(ffi::GDK_KEY_Cyrillic_BE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_CHE")]
    pub const Cyrillic_CHE: Self = Self(ffi::GDK_KEY_Cyrillic_CHE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_CHE_descender")]
    pub const Cyrillic_CHE_descender: Self = Self(ffi::GDK_KEY_Cyrillic_CHE_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_CHE_vertstroke")]
    pub const Cyrillic_CHE_vertstroke: Self = Self(ffi::GDK_KEY_Cyrillic_CHE_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_DE")]
    pub const Cyrillic_DE: Self = Self(ffi::GDK_KEY_Cyrillic_DE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_DZHE")]
    pub const Cyrillic_DZHE: Self = Self(ffi::GDK_KEY_Cyrillic_DZHE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_E")]
    pub const Cyrillic_E: Self = Self(ffi::GDK_KEY_Cyrillic_E as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EF")]
    pub const Cyrillic_EF: Self = Self(ffi::GDK_KEY_Cyrillic_EF as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EL")]
    pub const Cyrillic_EL: Self = Self(ffi::GDK_KEY_Cyrillic_EL as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EM")]
    pub const Cyrillic_EM: Self = Self(ffi::GDK_KEY_Cyrillic_EM as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EN")]
    pub const Cyrillic_EN: Self = Self(ffi::GDK_KEY_Cyrillic_EN as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EN_descender")]
    pub const Cyrillic_EN_descender: Self = Self(ffi::GDK_KEY_Cyrillic_EN_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ER")]
    pub const Cyrillic_ER: Self = Self(ffi::GDK_KEY_Cyrillic_ER as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ES")]
    pub const Cyrillic_ES: Self = Self(ffi::GDK_KEY_Cyrillic_ES as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_GHE")]
    pub const Cyrillic_GHE: Self = Self(ffi::GDK_KEY_Cyrillic_GHE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_GHE_bar")]
    pub const Cyrillic_GHE_bar: Self = Self(ffi::GDK_KEY_Cyrillic_GHE_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_HA")]
    pub const Cyrillic_HA: Self = Self(ffi::GDK_KEY_Cyrillic_HA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_HARDSIGN")]
    pub const Cyrillic_HARDSIGN: Self = Self(ffi::GDK_KEY_Cyrillic_HARDSIGN as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_HA_descender")]
    pub const Cyrillic_HA_descender: Self = Self(ffi::GDK_KEY_Cyrillic_HA_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_I")]
    pub const Cyrillic_I: Self = Self(ffi::GDK_KEY_Cyrillic_I as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_IE")]
    pub const Cyrillic_IE: Self = Self(ffi::GDK_KEY_Cyrillic_IE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_IO")]
    pub const Cyrillic_IO: Self = Self(ffi::GDK_KEY_Cyrillic_IO as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_I_macron")]
    pub const Cyrillic_I_macron: Self = Self(ffi::GDK_KEY_Cyrillic_I_macron as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_JE")]
    pub const Cyrillic_JE: Self = Self(ffi::GDK_KEY_Cyrillic_JE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_KA")]
    pub const Cyrillic_KA: Self = Self(ffi::GDK_KEY_Cyrillic_KA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_KA_descender")]
    pub const Cyrillic_KA_descender: Self = Self(ffi::GDK_KEY_Cyrillic_KA_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_KA_vertstroke")]
    pub const Cyrillic_KA_vertstroke: Self = Self(ffi::GDK_KEY_Cyrillic_KA_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_LJE")]
    pub const Cyrillic_LJE: Self = Self(ffi::GDK_KEY_Cyrillic_LJE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_NJE")]
    pub const Cyrillic_NJE: Self = Self(ffi::GDK_KEY_Cyrillic_NJE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_O")]
    pub const Cyrillic_O: Self = Self(ffi::GDK_KEY_Cyrillic_O as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_O_bar")]
    pub const Cyrillic_O_bar: Self = Self(ffi::GDK_KEY_Cyrillic_O_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_PE")]
    pub const Cyrillic_PE: Self = Self(ffi::GDK_KEY_Cyrillic_PE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SCHWA")]
    pub const Cyrillic_SCHWA: Self = Self(ffi::GDK_KEY_Cyrillic_SCHWA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHA")]
    pub const Cyrillic_SHA: Self = Self(ffi::GDK_KEY_Cyrillic_SHA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHCHA")]
    pub const Cyrillic_SHCHA: Self = Self(ffi::GDK_KEY_Cyrillic_SHCHA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHHA")]
    pub const Cyrillic_SHHA: Self = Self(ffi::GDK_KEY_Cyrillic_SHHA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHORTI")]
    pub const Cyrillic_SHORTI: Self = Self(ffi::GDK_KEY_Cyrillic_SHORTI as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SOFTSIGN")]
    pub const Cyrillic_SOFTSIGN: Self = Self(ffi::GDK_KEY_Cyrillic_SOFTSIGN as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_TE")]
    pub const Cyrillic_TE: Self = Self(ffi::GDK_KEY_Cyrillic_TE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_TSE")]
    pub const Cyrillic_TSE: Self = Self(ffi::GDK_KEY_Cyrillic_TSE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U")]
    pub const Cyrillic_U: Self = Self(ffi::GDK_KEY_Cyrillic_U as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U_macron")]
    pub const Cyrillic_U_macron: Self = Self(ffi::GDK_KEY_Cyrillic_U_macron as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U_straight")]
    pub const Cyrillic_U_straight: Self = Self(ffi::GDK_KEY_Cyrillic_U_straight as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U_straight_bar")]
    pub const Cyrillic_U_straight_bar: Self = Self(ffi::GDK_KEY_Cyrillic_U_straight_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_VE")]
    pub const Cyrillic_VE: Self = Self(ffi::GDK_KEY_Cyrillic_VE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_YA")]
    pub const Cyrillic_YA: Self = Self(ffi::GDK_KEY_Cyrillic_YA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_YERU")]
    pub const Cyrillic_YERU: Self = Self(ffi::GDK_KEY_Cyrillic_YERU as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_YU")]
    pub const Cyrillic_YU: Self = Self(ffi::GDK_KEY_Cyrillic_YU as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ZE")]
    pub const Cyrillic_ZE: Self = Self(ffi::GDK_KEY_Cyrillic_ZE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ZHE")]
    pub const Cyrillic_ZHE: Self = Self(ffi::GDK_KEY_Cyrillic_ZHE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ZHE_descender")]
    pub const Cyrillic_ZHE_descender: Self = Self(ffi::GDK_KEY_Cyrillic_ZHE_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_a")]
    pub const Cyrillic_a: Self = Self(ffi::GDK_KEY_Cyrillic_a as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_be")]
    pub const Cyrillic_be: Self = Self(ffi::GDK_KEY_Cyrillic_be as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_che")]
    pub const Cyrillic_che: Self = Self(ffi::GDK_KEY_Cyrillic_che as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_che_descender")]
    pub const Cyrillic_che_descender: Self = Self(ffi::GDK_KEY_Cyrillic_che_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_che_vertstroke")]
    pub const Cyrillic_che_vertstroke: Self = Self(ffi::GDK_KEY_Cyrillic_che_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_de")]
    pub const Cyrillic_de: Self = Self(ffi::GDK_KEY_Cyrillic_de as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_dzhe")]
    pub const Cyrillic_dzhe: Self = Self(ffi::GDK_KEY_Cyrillic_dzhe as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_e")]
    pub const Cyrillic_e: Self = Self(ffi::GDK_KEY_Cyrillic_e as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ef")]
    pub const Cyrillic_ef: Self = Self(ffi::GDK_KEY_Cyrillic_ef as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_el")]
    pub const Cyrillic_el: Self = Self(ffi::GDK_KEY_Cyrillic_el as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_em")]
    pub const Cyrillic_em: Self = Self(ffi::GDK_KEY_Cyrillic_em as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_en")]
    pub const Cyrillic_en: Self = Self(ffi::GDK_KEY_Cyrillic_en as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_en_descender")]
    pub const Cyrillic_en_descender: Self = Self(ffi::GDK_KEY_Cyrillic_en_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_er")]
    pub const Cyrillic_er: Self = Self(ffi::GDK_KEY_Cyrillic_er as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_es")]
    pub const Cyrillic_es: Self = Self(ffi::GDK_KEY_Cyrillic_es as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ghe")]
    pub const Cyrillic_ghe: Self = Self(ffi::GDK_KEY_Cyrillic_ghe as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ghe_bar")]
    pub const Cyrillic_ghe_bar: Self = Self(ffi::GDK_KEY_Cyrillic_ghe_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ha")]
    pub const Cyrillic_ha: Self = Self(ffi::GDK_KEY_Cyrillic_ha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ha_descender")]
    pub const Cyrillic_ha_descender: Self = Self(ffi::GDK_KEY_Cyrillic_ha_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_hardsign")]
    pub const Cyrillic_hardsign: Self = Self(ffi::GDK_KEY_Cyrillic_hardsign as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_i")]
    pub const Cyrillic_i: Self = Self(ffi::GDK_KEY_Cyrillic_i as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_i_macron")]
    pub const Cyrillic_i_macron: Self = Self(ffi::GDK_KEY_Cyrillic_i_macron as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ie")]
    pub const Cyrillic_ie: Self = Self(ffi::GDK_KEY_Cyrillic_ie as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_io")]
    pub const Cyrillic_io: Self = Self(ffi::GDK_KEY_Cyrillic_io as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_je")]
    pub const Cyrillic_je: Self = Self(ffi::GDK_KEY_Cyrillic_je as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ka")]
    pub const Cyrillic_ka: Self = Self(ffi::GDK_KEY_Cyrillic_ka as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ka_descender")]
    pub const Cyrillic_ka_descender: Self = Self(ffi::GDK_KEY_Cyrillic_ka_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ka_vertstroke")]
    pub const Cyrillic_ka_vertstroke: Self = Self(ffi::GDK_KEY_Cyrillic_ka_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_lje")]
    pub const Cyrillic_lje: Self = Self(ffi::GDK_KEY_Cyrillic_lje as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_nje")]
    pub const Cyrillic_nje: Self = Self(ffi::GDK_KEY_Cyrillic_nje as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_o")]
    pub const Cyrillic_o: Self = Self(ffi::GDK_KEY_Cyrillic_o as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_o_bar")]
    pub const Cyrillic_o_bar: Self = Self(ffi::GDK_KEY_Cyrillic_o_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_pe")]
    pub const Cyrillic_pe: Self = Self(ffi::GDK_KEY_Cyrillic_pe as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_schwa")]
    pub const Cyrillic_schwa: Self = Self(ffi::GDK_KEY_Cyrillic_schwa as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_sha")]
    pub const Cyrillic_sha: Self = Self(ffi::GDK_KEY_Cyrillic_sha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_shcha")]
    pub const Cyrillic_shcha: Self = Self(ffi::GDK_KEY_Cyrillic_shcha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_shha")]
    pub const Cyrillic_shha: Self = Self(ffi::GDK_KEY_Cyrillic_shha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_shorti")]
    pub const Cyrillic_shorti: Self = Self(ffi::GDK_KEY_Cyrillic_shorti as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_softsign")]
    pub const Cyrillic_softsign: Self = Self(ffi::GDK_KEY_Cyrillic_softsign as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_te")]
    pub const Cyrillic_te: Self = Self(ffi::GDK_KEY_Cyrillic_te as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_tse")]
    pub const Cyrillic_tse: Self = Self(ffi::GDK_KEY_Cyrillic_tse as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u")]
    pub const Cyrillic_u: Self = Self(ffi::GDK_KEY_Cyrillic_u as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u_macron")]
    pub const Cyrillic_u_macron: Self = Self(ffi::GDK_KEY_Cyrillic_u_macron as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u_straight")]
    pub const Cyrillic_u_straight: Self = Self(ffi::GDK_KEY_Cyrillic_u_straight as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u_straight_bar")]
    pub const Cyrillic_u_straight_bar: Self = Self(ffi::GDK_KEY_Cyrillic_u_straight_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ve")]
    pub const Cyrillic_ve: Self = Self(ffi::GDK_KEY_Cyrillic_ve as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ya")]
    pub const Cyrillic_ya: Self = Self(ffi::GDK_KEY_Cyrillic_ya as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_yeru")]
    pub const Cyrillic_yeru: Self = Self(ffi::GDK_KEY_Cyrillic_yeru as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_yu")]
    pub const Cyrillic_yu: Self = Self(ffi::GDK_KEY_Cyrillic_yu as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ze")]
    pub const Cyrillic_ze: Self = Self(ffi::GDK_KEY_Cyrillic_ze as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_zhe")]
    pub const Cyrillic_zhe: Self = Self(ffi::GDK_KEY_Cyrillic_zhe as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_zhe_descender")]
    pub const Cyrillic_zhe_descender: Self = Self(ffi::GDK_KEY_Cyrillic_zhe_descender as u32);
    #[doc(alias = "GDK_KEY_D")]
    pub const D: Self = Self(ffi::GDK_KEY_D as u32);
    #[doc(alias = "GDK_KEY_DOS")]
    pub const DOS: Self = Self(ffi::GDK_KEY_DOS as u32);
    #[doc(alias = "GDK_KEY_DVD")]
    pub const DVD: Self = Self(ffi::GDK_KEY_DVD as u32);
    #[doc(alias = "GDK_KEY_Dabovedot")]
    pub const Dabovedot: Self = Self(ffi::GDK_KEY_Dabovedot as u32);
    #[doc(alias = "GDK_KEY_Data")]
    pub const Data: Self = Self(ffi::GDK_KEY_Data as u32);
    #[doc(alias = "GDK_KEY_Database")]
    pub const Database: Self = Self(ffi::GDK_KEY_Database as u32);
    #[doc(alias = "GDK_KEY_Dcaron")]
    pub const Dcaron: Self = Self(ffi::GDK_KEY_Dcaron as u32);
    #[doc(alias = "GDK_KEY_Delete")]
    pub const Delete: Self = Self(ffi::GDK_KEY_Delete as u32);
    #[doc(alias = "GDK_KEY_Dictate")]
    pub const Dictate: Self = Self(ffi::GDK_KEY_Dictate as u32);
    #[doc(alias = "GDK_KEY_Display")]
    pub const Display: Self = Self(ffi::GDK_KEY_Display as u32);
    #[doc(alias = "GDK_KEY_DisplayOff")]
    pub const DisplayOff: Self = Self(ffi::GDK_KEY_DisplayOff as u32);
    #[doc(alias = "GDK_KEY_DisplayToggle")]
    pub const DisplayToggle: Self = Self(ffi::GDK_KEY_DisplayToggle as u32);
    #[doc(alias = "GDK_KEY_DoNotDisturb")]
    pub const DoNotDisturb: Self = Self(ffi::GDK_KEY_DoNotDisturb as u32);
    #[doc(alias = "GDK_KEY_Documents")]
    pub const Documents: Self = Self(ffi::GDK_KEY_Documents as u32);
    #[doc(alias = "GDK_KEY_DongSign")]
    pub const DongSign: Self = Self(ffi::GDK_KEY_DongSign as u32);
    #[doc(alias = "GDK_KEY_Down")]
    pub const Down: Self = Self(ffi::GDK_KEY_Down as u32);
    #[doc(alias = "GDK_KEY_Dstroke")]
    pub const Dstroke: Self = Self(ffi::GDK_KEY_Dstroke as u32);
    #[doc(alias = "GDK_KEY_DualRangeRadar")]
    pub const DualRangeRadar: Self = Self(ffi::GDK_KEY_DualRangeRadar as u32);
    #[doc(alias = "GDK_KEY_E")]
    pub const E: Self = Self(ffi::GDK_KEY_E as u32);
    #[doc(alias = "GDK_KEY_ENG")]
    pub const ENG: Self = Self(ffi::GDK_KEY_ENG as u32);
    #[doc(alias = "GDK_KEY_ETH")]
    pub const ETH: Self = Self(ffi::GDK_KEY_ETH as u32);
    #[doc(alias = "GDK_KEY_EZH")]
    pub const EZH: Self = Self(ffi::GDK_KEY_EZH as u32);
    #[doc(alias = "GDK_KEY_Eabovedot")]
    pub const Eabovedot: Self = Self(ffi::GDK_KEY_Eabovedot as u32);
    #[doc(alias = "GDK_KEY_Eacute")]
    pub const Eacute: Self = Self(ffi::GDK_KEY_Eacute as u32);
    #[doc(alias = "GDK_KEY_Ebelowdot")]
    pub const Ebelowdot: Self = Self(ffi::GDK_KEY_Ebelowdot as u32);
    #[doc(alias = "GDK_KEY_Ecaron")]
    pub const Ecaron: Self = Self(ffi::GDK_KEY_Ecaron as u32);
    #[doc(alias = "GDK_KEY_Ecircumflex")]
    pub const Ecircumflex: Self = Self(ffi::GDK_KEY_Ecircumflex as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexacute")]
    pub const Ecircumflexacute: Self = Self(ffi::GDK_KEY_Ecircumflexacute as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexbelowdot")]
    pub const Ecircumflexbelowdot: Self = Self(ffi::GDK_KEY_Ecircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexgrave")]
    pub const Ecircumflexgrave: Self = Self(ffi::GDK_KEY_Ecircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexhook")]
    pub const Ecircumflexhook: Self = Self(ffi::GDK_KEY_Ecircumflexhook as u32);
    #[doc(alias = "GDK_KEY_Ecircumflextilde")]
    pub const Ecircumflextilde: Self = Self(ffi::GDK_KEY_Ecircumflextilde as u32);
    #[doc(alias = "GDK_KEY_EcuSign")]
    pub const EcuSign: Self = Self(ffi::GDK_KEY_EcuSign as u32);
    #[doc(alias = "GDK_KEY_Ediaeresis")]
    pub const Ediaeresis: Self = Self(ffi::GDK_KEY_Ediaeresis as u32);
    #[doc(alias = "GDK_KEY_Editor")]
    pub const Editor: Self = Self(ffi::GDK_KEY_Editor as u32);
    #[doc(alias = "GDK_KEY_Egrave")]
    pub const Egrave: Self = Self(ffi::GDK_KEY_Egrave as u32);
    #[doc(alias = "GDK_KEY_Ehook")]
    pub const Ehook: Self = Self(ffi::GDK_KEY_Ehook as u32);
    #[doc(alias = "GDK_KEY_Eisu_Shift")]
    pub const Eisu_Shift: Self = Self(ffi::GDK_KEY_Eisu_Shift as u32);
    #[doc(alias = "GDK_KEY_Eisu_toggle")]
    pub const Eisu_toggle: Self = Self(ffi::GDK_KEY_Eisu_toggle as u32);
    #[doc(alias = "GDK_KEY_Eject")]
    pub const Eject: Self = Self(ffi::GDK_KEY_Eject as u32);
    #[doc(alias = "GDK_KEY_Emacron")]
    pub const Emacron: Self = Self(ffi::GDK_KEY_Emacron as u32);
    #[doc(alias = "GDK_KEY_EmojiPicker")]
    pub const EmojiPicker: Self = Self(ffi::GDK_KEY_EmojiPicker as u32);
    #[doc(alias = "GDK_KEY_End")]
    pub const End: Self = Self(ffi::GDK_KEY_End as u32);
    #[doc(alias = "GDK_KEY_Eogonek")]
    pub const Eogonek: Self = Self(ffi::GDK_KEY_Eogonek as u32);
    #[doc(alias = "GDK_KEY_Escape")]
    pub const Escape: Self = Self(ffi::GDK_KEY_Escape as u32);
    #[doc(alias = "GDK_KEY_Eth")]
    pub const Eth: Self = Self(ffi::GDK_KEY_Eth as u32);
    #[doc(alias = "GDK_KEY_Etilde")]
    pub const Etilde: Self = Self(ffi::GDK_KEY_Etilde as u32);
    #[doc(alias = "GDK_KEY_EuroSign")]
    pub const EuroSign: Self = Self(ffi::GDK_KEY_EuroSign as u32);
    #[doc(alias = "GDK_KEY_Excel")]
    pub const Excel: Self = Self(ffi::GDK_KEY_Excel as u32);
    #[doc(alias = "GDK_KEY_Execute")]
    pub const Execute: Self = Self(ffi::GDK_KEY_Execute as u32);
    #[doc(alias = "GDK_KEY_Explorer")]
    pub const Explorer: Self = Self(ffi::GDK_KEY_Explorer as u32);
    #[doc(alias = "GDK_KEY_F")]
    pub const F: Self = Self(ffi::GDK_KEY_F as u32);
    #[doc(alias = "GDK_KEY_F1")]
    pub const F1: Self = Self(ffi::GDK_KEY_F1 as u32);
    #[doc(alias = "GDK_KEY_F10")]
    pub const F10: Self = Self(ffi::GDK_KEY_F10 as u32);
    #[doc(alias = "GDK_KEY_F11")]
    pub const F11: Self = Self(ffi::GDK_KEY_F11 as u32);
    #[doc(alias = "GDK_KEY_F12")]
    pub const F12: Self = Self(ffi::GDK_KEY_F12 as u32);
    #[doc(alias = "GDK_KEY_F13")]
    pub const F13: Self = Self(ffi::GDK_KEY_F13 as u32);
    #[doc(alias = "GDK_KEY_F14")]
    pub const F14: Self = Self(ffi::GDK_KEY_F14 as u32);
    #[doc(alias = "GDK_KEY_F15")]
    pub const F15: Self = Self(ffi::GDK_KEY_F15 as u32);
    #[doc(alias = "GDK_KEY_F16")]
    pub const F16: Self = Self(ffi::GDK_KEY_F16 as u32);
    #[doc(alias = "GDK_KEY_F17")]
    pub const F17: Self = Self(ffi::GDK_KEY_F17 as u32);
    #[doc(alias = "GDK_KEY_F18")]
    pub const F18: Self = Self(ffi::GDK_KEY_F18 as u32);
    #[doc(alias = "GDK_KEY_F19")]
    pub const F19: Self = Self(ffi::GDK_KEY_F19 as u32);
    #[doc(alias = "GDK_KEY_F2")]
    pub const F2: Self = Self(ffi::GDK_KEY_F2 as u32);
    #[doc(alias = "GDK_KEY_F20")]
    pub const F20: Self = Self(ffi::GDK_KEY_F20 as u32);
    #[doc(alias = "GDK_KEY_F21")]
    pub const F21: Self = Self(ffi::GDK_KEY_F21 as u32);
    #[doc(alias = "GDK_KEY_F22")]
    pub const F22: Self = Self(ffi::GDK_KEY_F22 as u32);
    #[doc(alias = "GDK_KEY_F23")]
    pub const F23: Self = Self(ffi::GDK_KEY_F23 as u32);
    #[doc(alias = "GDK_KEY_F24")]
    pub const F24: Self = Self(ffi::GDK_KEY_F24 as u32);
    #[doc(alias = "GDK_KEY_F25")]
    pub const F25: Self = Self(ffi::GDK_KEY_F25 as u32);
    #[doc(alias = "GDK_KEY_F26")]
    pub const F26: Self = Self(ffi::GDK_KEY_F26 as u32);
    #[doc(alias = "GDK_KEY_F27")]
    pub const F27: Self = Self(ffi::GDK_KEY_F27 as u32);
    #[doc(alias = "GDK_KEY_F28")]
    pub const F28: Self = Self(ffi::GDK_KEY_F28 as u32);
    #[doc(alias = "GDK_KEY_F29")]
    pub const F29: Self = Self(ffi::GDK_KEY_F29 as u32);
    #[doc(alias = "GDK_KEY_F3")]
    pub const F3: Self = Self(ffi::GDK_KEY_F3 as u32);
    #[doc(alias = "GDK_KEY_F30")]
    pub const F30: Self = Self(ffi::GDK_KEY_F30 as u32);
    #[doc(alias = "GDK_KEY_F31")]
    pub const F31: Self = Self(ffi::GDK_KEY_F31 as u32);
    #[doc(alias = "GDK_KEY_F32")]
    pub const F32: Self = Self(ffi::GDK_KEY_F32 as u32);
    #[doc(alias = "GDK_KEY_F33")]
    pub const F33: Self = Self(ffi::GDK_KEY_F33 as u32);
    #[doc(alias = "GDK_KEY_F34")]
    pub const F34: Self = Self(ffi::GDK_KEY_F34 as u32);
    #[doc(alias = "GDK_KEY_F35")]
    pub const F35: Self = Self(ffi::GDK_KEY_F35 as u32);
    #[doc(alias = "GDK_KEY_F4")]
    pub const F4: Self = Self(ffi::GDK_KEY_F4 as u32);
    #[doc(alias = "GDK_KEY_F5")]
    pub const F5: Self = Self(ffi::GDK_KEY_F5 as u32);
    #[doc(alias = "GDK_KEY_F6")]
    pub const F6: Self = Self(ffi::GDK_KEY_F6 as u32);
    #[doc(alias = "GDK_KEY_F7")]
    pub const F7: Self = Self(ffi::GDK_KEY_F7 as u32);
    #[doc(alias = "GDK_KEY_F8")]
    pub const F8: Self = Self(ffi::GDK_KEY_F8 as u32);
    #[doc(alias = "GDK_KEY_F9")]
    pub const F9: Self = Self(ffi::GDK_KEY_F9 as u32);
    #[doc(alias = "GDK_KEY_FFrancSign")]
    pub const FFrancSign: Self = Self(ffi::GDK_KEY_FFrancSign as u32);
    #[doc(alias = "GDK_KEY_Fabovedot")]
    pub const Fabovedot: Self = Self(ffi::GDK_KEY_Fabovedot as u32);
    #[doc(alias = "GDK_KEY_Farsi_0")]
    pub const Farsi_0: Self = Self(ffi::GDK_KEY_Farsi_0 as u32);
    #[doc(alias = "GDK_KEY_Farsi_1")]
    pub const Farsi_1: Self = Self(ffi::GDK_KEY_Farsi_1 as u32);
    #[doc(alias = "GDK_KEY_Farsi_2")]
    pub const Farsi_2: Self = Self(ffi::GDK_KEY_Farsi_2 as u32);
    #[doc(alias = "GDK_KEY_Farsi_3")]
    pub const Farsi_3: Self = Self(ffi::GDK_KEY_Farsi_3 as u32);
    #[doc(alias = "GDK_KEY_Farsi_4")]
    pub const Farsi_4: Self = Self(ffi::GDK_KEY_Farsi_4 as u32);
    #[doc(alias = "GDK_KEY_Farsi_5")]
    pub const Farsi_5: Self = Self(ffi::GDK_KEY_Farsi_5 as u32);
    #[doc(alias = "GDK_KEY_Farsi_6")]
    pub const Farsi_6: Self = Self(ffi::GDK_KEY_Farsi_6 as u32);
    #[doc(alias = "GDK_KEY_Farsi_7")]
    pub const Farsi_7: Self = Self(ffi::GDK_KEY_Farsi_7 as u32);
    #[doc(alias = "GDK_KEY_Farsi_8")]
    pub const Farsi_8: Self = Self(ffi::GDK_KEY_Farsi_8 as u32);
    #[doc(alias = "GDK_KEY_Farsi_9")]
    pub const Farsi_9: Self = Self(ffi::GDK_KEY_Farsi_9 as u32);
    #[doc(alias = "GDK_KEY_Farsi_yeh")]
    pub const Farsi_yeh: Self = Self(ffi::GDK_KEY_Farsi_yeh as u32);
    #[doc(alias = "GDK_KEY_FastReverse")]
    pub const FastReverse: Self = Self(ffi::GDK_KEY_FastReverse as u32);
    #[doc(alias = "GDK_KEY_Favorites")]
    pub const Favorites: Self = Self(ffi::GDK_KEY_Favorites as u32);
    #[doc(alias = "GDK_KEY_Finance")]
    pub const Finance: Self = Self(ffi::GDK_KEY_Finance as u32);
    #[doc(alias = "GDK_KEY_Find")]
    pub const Find: Self = Self(ffi::GDK_KEY_Find as u32);
    #[doc(alias = "GDK_KEY_First_Virtual_Screen")]
    pub const First_Virtual_Screen: Self = Self(ffi::GDK_KEY_First_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_FishingChart")]
    pub const FishingChart: Self = Self(ffi::GDK_KEY_FishingChart as u32);
    #[doc(alias = "GDK_KEY_Fn")]
    pub const Fn: Self = Self(ffi::GDK_KEY_Fn as u32);
    #[doc(alias = "GDK_KEY_FnRightShift")]
    pub const FnRightShift: Self = Self(ffi::GDK_KEY_FnRightShift as u32);
    #[doc(alias = "GDK_KEY_Fn_Esc")]
    pub const Fn_Esc: Self = Self(ffi::GDK_KEY_Fn_Esc as u32);
    #[doc(alias = "GDK_KEY_Forward")]
    pub const Forward: Self = Self(ffi::GDK_KEY_Forward as u32);
    #[doc(alias = "GDK_KEY_FrameBack")]
    pub const FrameBack: Self = Self(ffi::GDK_KEY_FrameBack as u32);
    #[doc(alias = "GDK_KEY_FrameForward")]
    pub const FrameForward: Self = Self(ffi::GDK_KEY_FrameForward as u32);
    #[doc(alias = "GDK_KEY_FullScreen")]
    pub const FullScreen: Self = Self(ffi::GDK_KEY_FullScreen as u32);
    #[doc(alias = "GDK_KEY_G")]
    pub const G: Self = Self(ffi::GDK_KEY_G as u32);
    #[doc(alias = "GDK_KEY_Gabovedot")]
    pub const Gabovedot: Self = Self(ffi::GDK_KEY_Gabovedot as u32);
    #[doc(alias = "GDK_KEY_Game")]
    pub const Game: Self = Self(ffi::GDK_KEY_Game as u32);
    #[doc(alias = "GDK_KEY_Gbreve")]
    pub const Gbreve: Self = Self(ffi::GDK_KEY_Gbreve as u32);
    #[doc(alias = "GDK_KEY_Gcaron")]
    pub const Gcaron: Self = Self(ffi::GDK_KEY_Gcaron as u32);
    #[doc(alias = "GDK_KEY_Gcedilla")]
    pub const Gcedilla: Self = Self(ffi::GDK_KEY_Gcedilla as u32);
    #[doc(alias = "GDK_KEY_Gcircumflex")]
    pub const Gcircumflex: Self = Self(ffi::GDK_KEY_Gcircumflex as u32);
    #[doc(alias = "GDK_KEY_Georgian_an")]
    pub const Georgian_an: Self = Self(ffi::GDK_KEY_Georgian_an as u32);
    #[doc(alias = "GDK_KEY_Georgian_ban")]
    pub const Georgian_ban: Self = Self(ffi::GDK_KEY_Georgian_ban as u32);
    #[doc(alias = "GDK_KEY_Georgian_can")]
    pub const Georgian_can: Self = Self(ffi::GDK_KEY_Georgian_can as u32);
    #[doc(alias = "GDK_KEY_Georgian_char")]
    pub const Georgian_char: Self = Self(ffi::GDK_KEY_Georgian_char as u32);
    #[doc(alias = "GDK_KEY_Georgian_chin")]
    pub const Georgian_chin: Self = Self(ffi::GDK_KEY_Georgian_chin as u32);
    #[doc(alias = "GDK_KEY_Georgian_cil")]
    pub const Georgian_cil: Self = Self(ffi::GDK_KEY_Georgian_cil as u32);
    #[doc(alias = "GDK_KEY_Georgian_don")]
    pub const Georgian_don: Self = Self(ffi::GDK_KEY_Georgian_don as u32);
    #[doc(alias = "GDK_KEY_Georgian_en")]
    pub const Georgian_en: Self = Self(ffi::GDK_KEY_Georgian_en as u32);
    #[doc(alias = "GDK_KEY_Georgian_fi")]
    pub const Georgian_fi: Self = Self(ffi::GDK_KEY_Georgian_fi as u32);
    #[doc(alias = "GDK_KEY_Georgian_gan")]
    pub const Georgian_gan: Self = Self(ffi::GDK_KEY_Georgian_gan as u32);
    #[doc(alias = "GDK_KEY_Georgian_ghan")]
    pub const Georgian_ghan: Self = Self(ffi::GDK_KEY_Georgian_ghan as u32);
    #[doc(alias = "GDK_KEY_Georgian_hae")]
    pub const Georgian_hae: Self = Self(ffi::GDK_KEY_Georgian_hae as u32);
    #[doc(alias = "GDK_KEY_Georgian_har")]
    pub const Georgian_har: Self = Self(ffi::GDK_KEY_Georgian_har as u32);
    #[doc(alias = "GDK_KEY_Georgian_he")]
    pub const Georgian_he: Self = Self(ffi::GDK_KEY_Georgian_he as u32);
    #[doc(alias = "GDK_KEY_Georgian_hie")]
    pub const Georgian_hie: Self = Self(ffi::GDK_KEY_Georgian_hie as u32);
    #[doc(alias = "GDK_KEY_Georgian_hoe")]
    pub const Georgian_hoe: Self = Self(ffi::GDK_KEY_Georgian_hoe as u32);
    #[doc(alias = "GDK_KEY_Georgian_in")]
    pub const Georgian_in: Self = Self(ffi::GDK_KEY_Georgian_in as u32);
    #[doc(alias = "GDK_KEY_Georgian_jhan")]
    pub const Georgian_jhan: Self = Self(ffi::GDK_KEY_Georgian_jhan as u32);
    #[doc(alias = "GDK_KEY_Georgian_jil")]
    pub const Georgian_jil: Self = Self(ffi::GDK_KEY_Georgian_jil as u32);
    #[doc(alias = "GDK_KEY_Georgian_kan")]
    pub const Georgian_kan: Self = Self(ffi::GDK_KEY_Georgian_kan as u32);
    #[doc(alias = "GDK_KEY_Georgian_khar")]
    pub const Georgian_khar: Self = Self(ffi::GDK_KEY_Georgian_khar as u32);
    #[doc(alias = "GDK_KEY_Georgian_las")]
    pub const Georgian_las: Self = Self(ffi::GDK_KEY_Georgian_las as u32);
    #[doc(alias = "GDK_KEY_Georgian_man")]
    pub const Georgian_man: Self = Self(ffi::GDK_KEY_Georgian_man as u32);
    #[doc(alias = "GDK_KEY_Georgian_nar")]
    pub const Georgian_nar: Self = Self(ffi::GDK_KEY_Georgian_nar as u32);
    #[doc(alias = "GDK_KEY_Georgian_on")]
    pub const Georgian_on: Self = Self(ffi::GDK_KEY_Georgian_on as u32);
    #[doc(alias = "GDK_KEY_Georgian_par")]
    pub const Georgian_par: Self = Self(ffi::GDK_KEY_Georgian_par as u32);
    #[doc(alias = "GDK_KEY_Georgian_phar")]
    pub const Georgian_phar: Self = Self(ffi::GDK_KEY_Georgian_phar as u32);
    #[doc(alias = "GDK_KEY_Georgian_qar")]
    pub const Georgian_qar: Self = Self(ffi::GDK_KEY_Georgian_qar as u32);
    #[doc(alias = "GDK_KEY_Georgian_rae")]
    pub const Georgian_rae: Self = Self(ffi::GDK_KEY_Georgian_rae as u32);
    #[doc(alias = "GDK_KEY_Georgian_san")]
    pub const Georgian_san: Self = Self(ffi::GDK_KEY_Georgian_san as u32);
    #[doc(alias = "GDK_KEY_Georgian_shin")]
    pub const Georgian_shin: Self = Self(ffi::GDK_KEY_Georgian_shin as u32);
    #[doc(alias = "GDK_KEY_Georgian_tan")]
    pub const Georgian_tan: Self = Self(ffi::GDK_KEY_Georgian_tan as u32);
    #[doc(alias = "GDK_KEY_Georgian_tar")]
    pub const Georgian_tar: Self = Self(ffi::GDK_KEY_Georgian_tar as u32);
    #[doc(alias = "GDK_KEY_Georgian_un")]
    pub const Georgian_un: Self = Self(ffi::GDK_KEY_Georgian_un as u32);
    #[doc(alias = "GDK_KEY_Georgian_vin")]
    pub const Georgian_vin: Self = Self(ffi::GDK_KEY_Georgian_vin as u32);
    #[doc(alias = "GDK_KEY_Georgian_we")]
    pub const Georgian_we: Self = Self(ffi::GDK_KEY_Georgian_we as u32);
    #[doc(alias = "GDK_KEY_Georgian_xan")]
    pub const Georgian_xan: Self = Self(ffi::GDK_KEY_Georgian_xan as u32);
    #[doc(alias = "GDK_KEY_Georgian_zen")]
    pub const Georgian_zen: Self = Self(ffi::GDK_KEY_Georgian_zen as u32);
    #[doc(alias = "GDK_KEY_Georgian_zhar")]
    pub const Georgian_zhar: Self = Self(ffi::GDK_KEY_Georgian_zhar as u32);
    #[doc(alias = "GDK_KEY_Go")]
    pub const Go: Self = Self(ffi::GDK_KEY_Go as u32);
    #[doc(alias = "GDK_KEY_GraphicsEditor")]
    pub const GraphicsEditor: Self = Self(ffi::GDK_KEY_GraphicsEditor as u32);
    #[doc(alias = "GDK_KEY_Greek_ALPHA")]
    pub const Greek_ALPHA: Self = Self(ffi::GDK_KEY_Greek_ALPHA as u32);
    #[doc(alias = "GDK_KEY_Greek_ALPHAaccent")]
    pub const Greek_ALPHAaccent: Self = Self(ffi::GDK_KEY_Greek_ALPHAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_BETA")]
    pub const Greek_BETA: Self = Self(ffi::GDK_KEY_Greek_BETA as u32);
    #[doc(alias = "GDK_KEY_Greek_CHI")]
    pub const Greek_CHI: Self = Self(ffi::GDK_KEY_Greek_CHI as u32);
    #[doc(alias = "GDK_KEY_Greek_DELTA")]
    pub const Greek_DELTA: Self = Self(ffi::GDK_KEY_Greek_DELTA as u32);
    #[doc(alias = "GDK_KEY_Greek_EPSILON")]
    pub const Greek_EPSILON: Self = Self(ffi::GDK_KEY_Greek_EPSILON as u32);
    #[doc(alias = "GDK_KEY_Greek_EPSILONaccent")]
    pub const Greek_EPSILONaccent: Self = Self(ffi::GDK_KEY_Greek_EPSILONaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_ETA")]
    pub const Greek_ETA: Self = Self(ffi::GDK_KEY_Greek_ETA as u32);
    #[doc(alias = "GDK_KEY_Greek_ETAaccent")]
    pub const Greek_ETAaccent: Self = Self(ffi::GDK_KEY_Greek_ETAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_GAMMA")]
    pub const Greek_GAMMA: Self = Self(ffi::GDK_KEY_Greek_GAMMA as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTA")]
    pub const Greek_IOTA: Self = Self(ffi::GDK_KEY_Greek_IOTA as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTAaccent")]
    pub const Greek_IOTAaccent: Self = Self(ffi::GDK_KEY_Greek_IOTAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTAdiaeresis")]
    pub const Greek_IOTAdiaeresis: Self = Self(ffi::GDK_KEY_Greek_IOTAdiaeresis as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTAdieresis")]
    pub const Greek_IOTAdieresis: Self = Self(ffi::GDK_KEY_Greek_IOTAdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_KAPPA")]
    pub const Greek_KAPPA: Self = Self(ffi::GDK_KEY_Greek_KAPPA as u32);
    #[doc(alias = "GDK_KEY_Greek_LAMBDA")]
    pub const Greek_LAMBDA: Self = Self(ffi::GDK_KEY_Greek_LAMBDA as u32);
    #[doc(alias = "GDK_KEY_Greek_LAMDA")]
    pub const Greek_LAMDA: Self = Self(ffi::GDK_KEY_Greek_LAMDA as u32);
    #[doc(alias = "GDK_KEY_Greek_MU")]
    pub const Greek_MU: Self = Self(ffi::GDK_KEY_Greek_MU as u32);
    #[doc(alias = "GDK_KEY_Greek_NU")]
    pub const Greek_NU: Self = Self(ffi::GDK_KEY_Greek_NU as u32);
    #[doc(alias = "GDK_KEY_Greek_OMEGA")]
    pub const Greek_OMEGA: Self = Self(ffi::GDK_KEY_Greek_OMEGA as u32);
    #[doc(alias = "GDK_KEY_Greek_OMEGAaccent")]
    pub const Greek_OMEGAaccent: Self = Self(ffi::GDK_KEY_Greek_OMEGAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_OMICRON")]
    pub const Greek_OMICRON: Self = Self(ffi::GDK_KEY_Greek_OMICRON as u32);
    #[doc(alias = "GDK_KEY_Greek_OMICRONaccent")]
    pub const Greek_OMICRONaccent: Self = Self(ffi::GDK_KEY_Greek_OMICRONaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_PHI")]
    pub const Greek_PHI: Self = Self(ffi::GDK_KEY_Greek_PHI as u32);
    #[doc(alias = "GDK_KEY_Greek_PI")]
    pub const Greek_PI: Self = Self(ffi::GDK_KEY_Greek_PI as u32);
    #[doc(alias = "GDK_KEY_Greek_PSI")]
    pub const Greek_PSI: Self = Self(ffi::GDK_KEY_Greek_PSI as u32);
    #[doc(alias = "GDK_KEY_Greek_RHO")]
    pub const Greek_RHO: Self = Self(ffi::GDK_KEY_Greek_RHO as u32);
    #[doc(alias = "GDK_KEY_Greek_SIGMA")]
    pub const Greek_SIGMA: Self = Self(ffi::GDK_KEY_Greek_SIGMA as u32);
    #[doc(alias = "GDK_KEY_Greek_TAU")]
    pub const Greek_TAU: Self = Self(ffi::GDK_KEY_Greek_TAU as u32);
    #[doc(alias = "GDK_KEY_Greek_THETA")]
    pub const Greek_THETA: Self = Self(ffi::GDK_KEY_Greek_THETA as u32);
    #[doc(alias = "GDK_KEY_Greek_UPSILON")]
    pub const Greek_UPSILON: Self = Self(ffi::GDK_KEY_Greek_UPSILON as u32);
    #[doc(alias = "GDK_KEY_Greek_UPSILONaccent")]
    pub const Greek_UPSILONaccent: Self = Self(ffi::GDK_KEY_Greek_UPSILONaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_UPSILONdieresis")]
    pub const Greek_UPSILONdieresis: Self = Self(ffi::GDK_KEY_Greek_UPSILONdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_XI")]
    pub const Greek_XI: Self = Self(ffi::GDK_KEY_Greek_XI as u32);
    #[doc(alias = "GDK_KEY_Greek_ZETA")]
    pub const Greek_ZETA: Self = Self(ffi::GDK_KEY_Greek_ZETA as u32);
    #[doc(alias = "GDK_KEY_Greek_accentdieresis")]
    pub const Greek_accentdieresis: Self = Self(ffi::GDK_KEY_Greek_accentdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_alpha")]
    pub const Greek_alpha: Self = Self(ffi::GDK_KEY_Greek_alpha as u32);
    #[doc(alias = "GDK_KEY_Greek_alphaaccent")]
    pub const Greek_alphaaccent: Self = Self(ffi::GDK_KEY_Greek_alphaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_beta")]
    pub const Greek_beta: Self = Self(ffi::GDK_KEY_Greek_beta as u32);
    #[doc(alias = "GDK_KEY_Greek_chi")]
    pub const Greek_chi: Self = Self(ffi::GDK_KEY_Greek_chi as u32);
    #[doc(alias = "GDK_KEY_Greek_delta")]
    pub const Greek_delta: Self = Self(ffi::GDK_KEY_Greek_delta as u32);
    #[doc(alias = "GDK_KEY_Greek_epsilon")]
    pub const Greek_epsilon: Self = Self(ffi::GDK_KEY_Greek_epsilon as u32);
    #[doc(alias = "GDK_KEY_Greek_epsilonaccent")]
    pub const Greek_epsilonaccent: Self = Self(ffi::GDK_KEY_Greek_epsilonaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_eta")]
    pub const Greek_eta: Self = Self(ffi::GDK_KEY_Greek_eta as u32);
    #[doc(alias = "GDK_KEY_Greek_etaaccent")]
    pub const Greek_etaaccent: Self = Self(ffi::GDK_KEY_Greek_etaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_finalsmallsigma")]
    pub const Greek_finalsmallsigma: Self = Self(ffi::GDK_KEY_Greek_finalsmallsigma as u32);
    #[doc(alias = "GDK_KEY_Greek_gamma")]
    pub const Greek_gamma: Self = Self(ffi::GDK_KEY_Greek_gamma as u32);
    #[doc(alias = "GDK_KEY_Greek_horizbar")]
    pub const Greek_horizbar: Self = Self(ffi::GDK_KEY_Greek_horizbar as u32);
    #[doc(alias = "GDK_KEY_Greek_iota")]
    pub const Greek_iota: Self = Self(ffi::GDK_KEY_Greek_iota as u32);
    #[doc(alias = "GDK_KEY_Greek_iotaaccent")]
    pub const Greek_iotaaccent: Self = Self(ffi::GDK_KEY_Greek_iotaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_iotaaccentdieresis")]
    pub const Greek_iotaaccentdieresis: Self = Self(ffi::GDK_KEY_Greek_iotaaccentdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_iotadieresis")]
    pub const Greek_iotadieresis: Self = Self(ffi::GDK_KEY_Greek_iotadieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_kappa")]
    pub const Greek_kappa: Self = Self(ffi::GDK_KEY_Greek_kappa as u32);
    #[doc(alias = "GDK_KEY_Greek_lambda")]
    pub const Greek_lambda: Self = Self(ffi::GDK_KEY_Greek_lambda as u32);
    #[doc(alias = "GDK_KEY_Greek_lamda")]
    pub const Greek_lamda: Self = Self(ffi::GDK_KEY_Greek_lamda as u32);
    #[doc(alias = "GDK_KEY_Greek_mu")]
    pub const Greek_mu: Self = Self(ffi::GDK_KEY_Greek_mu as u32);
    #[doc(alias = "GDK_KEY_Greek_nu")]
    pub const Greek_nu: Self = Self(ffi::GDK_KEY_Greek_nu as u32);
    #[doc(alias = "GDK_KEY_Greek_omega")]
    pub const Greek_omega: Self = Self(ffi::GDK_KEY_Greek_omega as u32);
    #[doc(alias = "GDK_KEY_Greek_omegaaccent")]
    pub const Greek_omegaaccent: Self = Self(ffi::GDK_KEY_Greek_omegaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_omicron")]
    pub const Greek_omicron: Self = Self(ffi::GDK_KEY_Greek_omicron as u32);
    #[doc(alias = "GDK_KEY_Greek_omicronaccent")]
    pub const Greek_omicronaccent: Self = Self(ffi::GDK_KEY_Greek_omicronaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_phi")]
    pub const Greek_phi: Self = Self(ffi::GDK_KEY_Greek_phi as u32);
    #[doc(alias = "GDK_KEY_Greek_pi")]
    pub const Greek_pi: Self = Self(ffi::GDK_KEY_Greek_pi as u32);
    #[doc(alias = "GDK_KEY_Greek_psi")]
    pub const Greek_psi: Self = Self(ffi::GDK_KEY_Greek_psi as u32);
    #[doc(alias = "GDK_KEY_Greek_rho")]
    pub const Greek_rho: Self = Self(ffi::GDK_KEY_Greek_rho as u32);
    #[doc(alias = "GDK_KEY_Greek_sigma")]
    pub const Greek_sigma: Self = Self(ffi::GDK_KEY_Greek_sigma as u32);
    #[doc(alias = "GDK_KEY_Greek_switch")]
    pub const Greek_switch: Self = Self(ffi::GDK_KEY_Greek_switch as u32);
    #[doc(alias = "GDK_KEY_Greek_tau")]
    pub const Greek_tau: Self = Self(ffi::GDK_KEY_Greek_tau as u32);
    #[doc(alias = "GDK_KEY_Greek_theta")]
    pub const Greek_theta: Self = Self(ffi::GDK_KEY_Greek_theta as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilon")]
    pub const Greek_upsilon: Self = Self(ffi::GDK_KEY_Greek_upsilon as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilonaccent")]
    pub const Greek_upsilonaccent: Self = Self(ffi::GDK_KEY_Greek_upsilonaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilonaccentdieresis")]
    pub const Greek_upsilonaccentdieresis: Self =
        Self(ffi::GDK_KEY_Greek_upsilonaccentdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilondieresis")]
    pub const Greek_upsilondieresis: Self = Self(ffi::GDK_KEY_Greek_upsilondieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_xi")]
    pub const Greek_xi: Self = Self(ffi::GDK_KEY_Greek_xi as u32);
    #[doc(alias = "GDK_KEY_Greek_zeta")]
    pub const Greek_zeta: Self = Self(ffi::GDK_KEY_Greek_zeta as u32);
    #[doc(alias = "GDK_KEY_Green")]
    pub const Green: Self = Self(ffi::GDK_KEY_Green as u32);
    #[doc(alias = "GDK_KEY_H")]
    pub const H: Self = Self(ffi::GDK_KEY_H as u32);
    #[doc(alias = "GDK_KEY_Hangul")]
    pub const Hangul: Self = Self(ffi::GDK_KEY_Hangul as u32);
    #[doc(alias = "GDK_KEY_Hangul_A")]
    pub const Hangul_A: Self = Self(ffi::GDK_KEY_Hangul_A as u32);
    #[doc(alias = "GDK_KEY_Hangul_AE")]
    pub const Hangul_AE: Self = Self(ffi::GDK_KEY_Hangul_AE as u32);
    #[doc(alias = "GDK_KEY_Hangul_AraeA")]
    pub const Hangul_AraeA: Self = Self(ffi::GDK_KEY_Hangul_AraeA as u32);
    #[doc(alias = "GDK_KEY_Hangul_AraeAE")]
    pub const Hangul_AraeAE: Self = Self(ffi::GDK_KEY_Hangul_AraeAE as u32);
    #[doc(alias = "GDK_KEY_Hangul_Banja")]
    pub const Hangul_Banja: Self = Self(ffi::GDK_KEY_Hangul_Banja as u32);
    #[doc(alias = "GDK_KEY_Hangul_Cieuc")]
    pub const Hangul_Cieuc: Self = Self(ffi::GDK_KEY_Hangul_Cieuc as u32);
    #[doc(alias = "GDK_KEY_Hangul_Codeinput")]
    pub const Hangul_Codeinput: Self = Self(ffi::GDK_KEY_Hangul_Codeinput as u32);
    #[doc(alias = "GDK_KEY_Hangul_Dikeud")]
    pub const Hangul_Dikeud: Self = Self(ffi::GDK_KEY_Hangul_Dikeud as u32);
    #[doc(alias = "GDK_KEY_Hangul_E")]
    pub const Hangul_E: Self = Self(ffi::GDK_KEY_Hangul_E as u32);
    #[doc(alias = "GDK_KEY_Hangul_EO")]
    pub const Hangul_EO: Self = Self(ffi::GDK_KEY_Hangul_EO as u32);
    #[doc(alias = "GDK_KEY_Hangul_EU")]
    pub const Hangul_EU: Self = Self(ffi::GDK_KEY_Hangul_EU as u32);
    #[doc(alias = "GDK_KEY_Hangul_End")]
    pub const Hangul_End: Self = Self(ffi::GDK_KEY_Hangul_End as u32);
    #[doc(alias = "GDK_KEY_Hangul_Hanja")]
    pub const Hangul_Hanja: Self = Self(ffi::GDK_KEY_Hangul_Hanja as u32);
    #[doc(alias = "GDK_KEY_Hangul_Hieuh")]
    pub const Hangul_Hieuh: Self = Self(ffi::GDK_KEY_Hangul_Hieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_I")]
    pub const Hangul_I: Self = Self(ffi::GDK_KEY_Hangul_I as u32);
    #[doc(alias = "GDK_KEY_Hangul_Ieung")]
    pub const Hangul_Ieung: Self = Self(ffi::GDK_KEY_Hangul_Ieung as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Cieuc")]
    pub const Hangul_J_Cieuc: Self = Self(ffi::GDK_KEY_Hangul_J_Cieuc as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Dikeud")]
    pub const Hangul_J_Dikeud: Self = Self(ffi::GDK_KEY_Hangul_J_Dikeud as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Hieuh")]
    pub const Hangul_J_Hieuh: Self = Self(ffi::GDK_KEY_Hangul_J_Hieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Ieung")]
    pub const Hangul_J_Ieung: Self = Self(ffi::GDK_KEY_Hangul_J_Ieung as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Jieuj")]
    pub const Hangul_J_Jieuj: Self = Self(ffi::GDK_KEY_Hangul_J_Jieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Khieuq")]
    pub const Hangul_J_Khieuq: Self = Self(ffi::GDK_KEY_Hangul_J_Khieuq as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Kiyeog")]
    pub const Hangul_J_Kiyeog: Self = Self(ffi::GDK_KEY_Hangul_J_Kiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_KiyeogSios")]
    pub const Hangul_J_KiyeogSios: Self = Self(ffi::GDK_KEY_Hangul_J_KiyeogSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_KkogjiDalrinIeung")]
    pub const Hangul_J_KkogjiDalrinIeung: Self =
        Self(ffi::GDK_KEY_Hangul_J_KkogjiDalrinIeung as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Mieum")]
    pub const Hangul_J_Mieum: Self = Self(ffi::GDK_KEY_Hangul_J_Mieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Nieun")]
    pub const Hangul_J_Nieun: Self = Self(ffi::GDK_KEY_Hangul_J_Nieun as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_NieunHieuh")]
    pub const Hangul_J_NieunHieuh: Self = Self(ffi::GDK_KEY_Hangul_J_NieunHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_NieunJieuj")]
    pub const Hangul_J_NieunJieuj: Self = Self(ffi::GDK_KEY_Hangul_J_NieunJieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_PanSios")]
    pub const Hangul_J_PanSios: Self = Self(ffi::GDK_KEY_Hangul_J_PanSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Phieuf")]
    pub const Hangul_J_Phieuf: Self = Self(ffi::GDK_KEY_Hangul_J_Phieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Pieub")]
    pub const Hangul_J_Pieub: Self = Self(ffi::GDK_KEY_Hangul_J_Pieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_PieubSios")]
    pub const Hangul_J_PieubSios: Self = Self(ffi::GDK_KEY_Hangul_J_PieubSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Rieul")]
    pub const Hangul_J_Rieul: Self = Self(ffi::GDK_KEY_Hangul_J_Rieul as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulHieuh")]
    pub const Hangul_J_RieulHieuh: Self = Self(ffi::GDK_KEY_Hangul_J_RieulHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulKiyeog")]
    pub const Hangul_J_RieulKiyeog: Self = Self(ffi::GDK_KEY_Hangul_J_RieulKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulMieum")]
    pub const Hangul_J_RieulMieum: Self = Self(ffi::GDK_KEY_Hangul_J_RieulMieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulPhieuf")]
    pub const Hangul_J_RieulPhieuf: Self = Self(ffi::GDK_KEY_Hangul_J_RieulPhieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulPieub")]
    pub const Hangul_J_RieulPieub: Self = Self(ffi::GDK_KEY_Hangul_J_RieulPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulSios")]
    pub const Hangul_J_RieulSios: Self = Self(ffi::GDK_KEY_Hangul_J_RieulSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulTieut")]
    pub const Hangul_J_RieulTieut: Self = Self(ffi::GDK_KEY_Hangul_J_RieulTieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Sios")]
    pub const Hangul_J_Sios: Self = Self(ffi::GDK_KEY_Hangul_J_Sios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_SsangKiyeog")]
    pub const Hangul_J_SsangKiyeog: Self = Self(ffi::GDK_KEY_Hangul_J_SsangKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_SsangSios")]
    pub const Hangul_J_SsangSios: Self = Self(ffi::GDK_KEY_Hangul_J_SsangSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Tieut")]
    pub const Hangul_J_Tieut: Self = Self(ffi::GDK_KEY_Hangul_J_Tieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_YeorinHieuh")]
    pub const Hangul_J_YeorinHieuh: Self = Self(ffi::GDK_KEY_Hangul_J_YeorinHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_Jamo")]
    pub const Hangul_Jamo: Self = Self(ffi::GDK_KEY_Hangul_Jamo as u32);
    #[doc(alias = "GDK_KEY_Hangul_Jeonja")]
    pub const Hangul_Jeonja: Self = Self(ffi::GDK_KEY_Hangul_Jeonja as u32);
    #[doc(alias = "GDK_KEY_Hangul_Jieuj")]
    pub const Hangul_Jieuj: Self = Self(ffi::GDK_KEY_Hangul_Jieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_Khieuq")]
    pub const Hangul_Khieuq: Self = Self(ffi::GDK_KEY_Hangul_Khieuq as u32);
    #[doc(alias = "GDK_KEY_Hangul_Kiyeog")]
    pub const Hangul_Kiyeog: Self = Self(ffi::GDK_KEY_Hangul_Kiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_KiyeogSios")]
    pub const Hangul_KiyeogSios: Self = Self(ffi::GDK_KEY_Hangul_KiyeogSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_KkogjiDalrinIeung")]
    pub const Hangul_KkogjiDalrinIeung: Self = Self(ffi::GDK_KEY_Hangul_KkogjiDalrinIeung as u32);
    #[doc(alias = "GDK_KEY_Hangul_Mieum")]
    pub const Hangul_Mieum: Self = Self(ffi::GDK_KEY_Hangul_Mieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_MultipleCandidate")]
    pub const Hangul_MultipleCandidate: Self = Self(ffi::GDK_KEY_Hangul_MultipleCandidate as u32);
    #[doc(alias = "GDK_KEY_Hangul_Nieun")]
    pub const Hangul_Nieun: Self = Self(ffi::GDK_KEY_Hangul_Nieun as u32);
    #[doc(alias = "GDK_KEY_Hangul_NieunHieuh")]
    pub const Hangul_NieunHieuh: Self = Self(ffi::GDK_KEY_Hangul_NieunHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_NieunJieuj")]
    pub const Hangul_NieunJieuj: Self = Self(ffi::GDK_KEY_Hangul_NieunJieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_O")]
    pub const Hangul_O: Self = Self(ffi::GDK_KEY_Hangul_O as u32);
    #[doc(alias = "GDK_KEY_Hangul_OE")]
    pub const Hangul_OE: Self = Self(ffi::GDK_KEY_Hangul_OE as u32);
    #[doc(alias = "GDK_KEY_Hangul_PanSios")]
    pub const Hangul_PanSios: Self = Self(ffi::GDK_KEY_Hangul_PanSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_Phieuf")]
    pub const Hangul_Phieuf: Self = Self(ffi::GDK_KEY_Hangul_Phieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_Pieub")]
    pub const Hangul_Pieub: Self = Self(ffi::GDK_KEY_Hangul_Pieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_PieubSios")]
    pub const Hangul_PieubSios: Self = Self(ffi::GDK_KEY_Hangul_PieubSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_PostHanja")]
    pub const Hangul_PostHanja: Self = Self(ffi::GDK_KEY_Hangul_PostHanja as u32);
    #[doc(alias = "GDK_KEY_Hangul_PreHanja")]
    pub const Hangul_PreHanja: Self = Self(ffi::GDK_KEY_Hangul_PreHanja as u32);
    #[doc(alias = "GDK_KEY_Hangul_PreviousCandidate")]
    pub const Hangul_PreviousCandidate: Self = Self(ffi::GDK_KEY_Hangul_PreviousCandidate as u32);
    #[doc(alias = "GDK_KEY_Hangul_Rieul")]
    pub const Hangul_Rieul: Self = Self(ffi::GDK_KEY_Hangul_Rieul as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulHieuh")]
    pub const Hangul_RieulHieuh: Self = Self(ffi::GDK_KEY_Hangul_RieulHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulKiyeog")]
    pub const Hangul_RieulKiyeog: Self = Self(ffi::GDK_KEY_Hangul_RieulKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulMieum")]
    pub const Hangul_RieulMieum: Self = Self(ffi::GDK_KEY_Hangul_RieulMieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulPhieuf")]
    pub const Hangul_RieulPhieuf: Self = Self(ffi::GDK_KEY_Hangul_RieulPhieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulPieub")]
    pub const Hangul_RieulPieub: Self = Self(ffi::GDK_KEY_Hangul_RieulPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulSios")]
    pub const Hangul_RieulSios: Self = Self(ffi::GDK_KEY_Hangul_RieulSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulTieut")]
    pub const Hangul_RieulTieut: Self = Self(ffi::GDK_KEY_Hangul_RieulTieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulYeorinHieuh")]
    pub const Hangul_RieulYeorinHieuh: Self = Self(ffi::GDK_KEY_Hangul_RieulYeorinHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_Romaja")]
    pub const Hangul_Romaja: Self = Self(ffi::GDK_KEY_Hangul_Romaja as u32);
    #[doc(alias = "GDK_KEY_Hangul_SingleCandidate")]
    pub const Hangul_SingleCandidate: Self = Self(ffi::GDK_KEY_Hangul_SingleCandidate as u32);
    #[doc(alias = "GDK_KEY_Hangul_Sios")]
    pub const Hangul_Sios: Self = Self(ffi::GDK_KEY_Hangul_Sios as u32);
    #[doc(alias = "GDK_KEY_Hangul_Special")]
    pub const Hangul_Special: Self = Self(ffi::GDK_KEY_Hangul_Special as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangDikeud")]
    pub const Hangul_SsangDikeud: Self = Self(ffi::GDK_KEY_Hangul_SsangDikeud as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangJieuj")]
    pub const Hangul_SsangJieuj: Self = Self(ffi::GDK_KEY_Hangul_SsangJieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangKiyeog")]
    pub const Hangul_SsangKiyeog: Self = Self(ffi::GDK_KEY_Hangul_SsangKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangPieub")]
    pub const Hangul_SsangPieub: Self = Self(ffi::GDK_KEY_Hangul_SsangPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangSios")]
    pub const Hangul_SsangSios: Self = Self(ffi::GDK_KEY_Hangul_SsangSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_Start")]
    pub const Hangul_Start: Self = Self(ffi::GDK_KEY_Hangul_Start as u32);
    #[doc(alias = "GDK_KEY_Hangul_SunkyeongeumMieum")]
    pub const Hangul_SunkyeongeumMieum: Self = Self(ffi::GDK_KEY_Hangul_SunkyeongeumMieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_SunkyeongeumPhieuf")]
    pub const Hangul_SunkyeongeumPhieuf: Self = Self(ffi::GDK_KEY_Hangul_SunkyeongeumPhieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_SunkyeongeumPieub")]
    pub const Hangul_SunkyeongeumPieub: Self = Self(ffi::GDK_KEY_Hangul_SunkyeongeumPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_Tieut")]
    pub const Hangul_Tieut: Self = Self(ffi::GDK_KEY_Hangul_Tieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_U")]
    pub const Hangul_U: Self = Self(ffi::GDK_KEY_Hangul_U as u32);
    #[doc(alias = "GDK_KEY_Hangul_WA")]
    pub const Hangul_WA: Self = Self(ffi::GDK_KEY_Hangul_WA as u32);
    #[doc(alias = "GDK_KEY_Hangul_WAE")]
    pub const Hangul_WAE: Self = Self(ffi::GDK_KEY_Hangul_WAE as u32);
    #[doc(alias = "GDK_KEY_Hangul_WE")]
    pub const Hangul_WE: Self = Self(ffi::GDK_KEY_Hangul_WE as u32);
    #[doc(alias = "GDK_KEY_Hangul_WEO")]
    pub const Hangul_WEO: Self = Self(ffi::GDK_KEY_Hangul_WEO as u32);
    #[doc(alias = "GDK_KEY_Hangul_WI")]
    pub const Hangul_WI: Self = Self(ffi::GDK_KEY_Hangul_WI as u32);
    #[doc(alias = "GDK_KEY_Hangul_YA")]
    pub const Hangul_YA: Self = Self(ffi::GDK_KEY_Hangul_YA as u32);
    #[doc(alias = "GDK_KEY_Hangul_YAE")]
    pub const Hangul_YAE: Self = Self(ffi::GDK_KEY_Hangul_YAE as u32);
    #[doc(alias = "GDK_KEY_Hangul_YE")]
    pub const Hangul_YE: Self = Self(ffi::GDK_KEY_Hangul_YE as u32);
    #[doc(alias = "GDK_KEY_Hangul_YEO")]
    pub const Hangul_YEO: Self = Self(ffi::GDK_KEY_Hangul_YEO as u32);
    #[doc(alias = "GDK_KEY_Hangul_YI")]
    pub const Hangul_YI: Self = Self(ffi::GDK_KEY_Hangul_YI as u32);
    #[doc(alias = "GDK_KEY_Hangul_YO")]
    pub const Hangul_YO: Self = Self(ffi::GDK_KEY_Hangul_YO as u32);
    #[doc(alias = "GDK_KEY_Hangul_YU")]
    pub const Hangul_YU: Self = Self(ffi::GDK_KEY_Hangul_YU as u32);
    #[doc(alias = "GDK_KEY_Hangul_YeorinHieuh")]
    pub const Hangul_YeorinHieuh: Self = Self(ffi::GDK_KEY_Hangul_YeorinHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_switch")]
    pub const Hangul_switch: Self = Self(ffi::GDK_KEY_Hangul_switch as u32);
    #[doc(alias = "GDK_KEY_HangupPhone")]
    pub const HangupPhone: Self = Self(ffi::GDK_KEY_HangupPhone as u32);
    #[doc(alias = "GDK_KEY_Hankaku")]
    pub const Hankaku: Self = Self(ffi::GDK_KEY_Hankaku as u32);
    #[doc(alias = "GDK_KEY_Hcircumflex")]
    pub const Hcircumflex: Self = Self(ffi::GDK_KEY_Hcircumflex as u32);
    #[doc(alias = "GDK_KEY_Hebrew_switch")]
    pub const Hebrew_switch: Self = Self(ffi::GDK_KEY_Hebrew_switch as u32);
    #[doc(alias = "GDK_KEY_Help")]
    pub const Help: Self = Self(ffi::GDK_KEY_Help as u32);
    #[doc(alias = "GDK_KEY_Henkan")]
    pub const Henkan: Self = Self(ffi::GDK_KEY_Henkan as u32);
    #[doc(alias = "GDK_KEY_Henkan_Mode")]
    pub const Henkan_Mode: Self = Self(ffi::GDK_KEY_Henkan_Mode as u32);
    #[doc(alias = "GDK_KEY_Hibernate")]
    pub const Hibernate: Self = Self(ffi::GDK_KEY_Hibernate as u32);
    #[doc(alias = "GDK_KEY_Hiragana")]
    pub const Hiragana: Self = Self(ffi::GDK_KEY_Hiragana as u32);
    #[doc(alias = "GDK_KEY_Hiragana_Katakana")]
    pub const Hiragana_Katakana: Self = Self(ffi::GDK_KEY_Hiragana_Katakana as u32);
    #[doc(alias = "GDK_KEY_History")]
    pub const History: Self = Self(ffi::GDK_KEY_History as u32);
    #[doc(alias = "GDK_KEY_Home")]
    pub const Home: Self = Self(ffi::GDK_KEY_Home as u32);
    #[doc(alias = "GDK_KEY_HomePage")]
    pub const HomePage: Self = Self(ffi::GDK_KEY_HomePage as u32);
    #[doc(alias = "GDK_KEY_HotLinks")]
    pub const HotLinks: Self = Self(ffi::GDK_KEY_HotLinks as u32);
    #[doc(alias = "GDK_KEY_Hstroke")]
    pub const Hstroke: Self = Self(ffi::GDK_KEY_Hstroke as u32);
    #[doc(alias = "GDK_KEY_Hyper_L")]
    pub const Hyper_L: Self = Self(ffi::GDK_KEY_Hyper_L as u32);
    #[doc(alias = "GDK_KEY_Hyper_R")]
    pub const Hyper_R: Self = Self(ffi::GDK_KEY_Hyper_R as u32);
    #[doc(alias = "GDK_KEY_I")]
    pub const I: Self = Self(ffi::GDK_KEY_I as u32);
    #[doc(alias = "GDK_KEY_ISO_Center_Object")]
    pub const ISO_Center_Object: Self = Self(ffi::GDK_KEY_ISO_Center_Object as u32);
    #[doc(alias = "GDK_KEY_ISO_Continuous_Underline")]
    pub const ISO_Continuous_Underline: Self = Self(ffi::GDK_KEY_ISO_Continuous_Underline as u32);
    #[doc(alias = "GDK_KEY_ISO_Discontinuous_Underline")]
    pub const ISO_Discontinuous_Underline: Self =
        Self(ffi::GDK_KEY_ISO_Discontinuous_Underline as u32);
    #[doc(alias = "GDK_KEY_ISO_Emphasize")]
    pub const ISO_Emphasize: Self = Self(ffi::GDK_KEY_ISO_Emphasize as u32);
    #[doc(alias = "GDK_KEY_ISO_Enter")]
    pub const ISO_Enter: Self = Self(ffi::GDK_KEY_ISO_Enter as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Down")]
    pub const ISO_Fast_Cursor_Down: Self = Self(ffi::GDK_KEY_ISO_Fast_Cursor_Down as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Left")]
    pub const ISO_Fast_Cursor_Left: Self = Self(ffi::GDK_KEY_ISO_Fast_Cursor_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Right")]
    pub const ISO_Fast_Cursor_Right: Self = Self(ffi::GDK_KEY_ISO_Fast_Cursor_Right as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Up")]
    pub const ISO_Fast_Cursor_Up: Self = Self(ffi::GDK_KEY_ISO_Fast_Cursor_Up as u32);
    #[doc(alias = "GDK_KEY_ISO_First_Group")]
    pub const ISO_First_Group: Self = Self(ffi::GDK_KEY_ISO_First_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_First_Group_Lock")]
    pub const ISO_First_Group_Lock: Self = Self(ffi::GDK_KEY_ISO_First_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Group_Latch")]
    pub const ISO_Group_Latch: Self = Self(ffi::GDK_KEY_ISO_Group_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Group_Lock")]
    pub const ISO_Group_Lock: Self = Self(ffi::GDK_KEY_ISO_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Group_Shift")]
    pub const ISO_Group_Shift: Self = Self(ffi::GDK_KEY_ISO_Group_Shift as u32);
    #[doc(alias = "GDK_KEY_ISO_Last_Group")]
    pub const ISO_Last_Group: Self = Self(ffi::GDK_KEY_ISO_Last_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_Last_Group_Lock")]
    pub const ISO_Last_Group_Lock: Self = Self(ffi::GDK_KEY_ISO_Last_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Left_Tab")]
    pub const ISO_Left_Tab: Self = Self(ffi::GDK_KEY_ISO_Left_Tab as u32);
    #[doc(alias = "GDK_KEY_ISO_Level2_Latch")]
    pub const ISO_Level2_Latch: Self = Self(ffi::GDK_KEY_ISO_Level2_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Level3_Latch")]
    pub const ISO_Level3_Latch: Self = Self(ffi::GDK_KEY_ISO_Level3_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Level3_Lock")]
    pub const ISO_Level3_Lock: Self = Self(ffi::GDK_KEY_ISO_Level3_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Level3_Shift")]
    pub const ISO_Level3_Shift: Self = Self(ffi::GDK_KEY_ISO_Level3_Shift as u32);
    #[doc(alias = "GDK_KEY_ISO_Level5_Latch")]
    pub const ISO_Level5_Latch: Self = Self(ffi::GDK_KEY_ISO_Level5_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Level5_Lock")]
    pub const ISO_Level5_Lock: Self = Self(ffi::GDK_KEY_ISO_Level5_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Level5_Shift")]
    pub const ISO_Level5_Shift: Self = Self(ffi::GDK_KEY_ISO_Level5_Shift as u32);
    #[doc(alias = "GDK_KEY_ISO_Lock")]
    pub const ISO_Lock: Self = Self(ffi::GDK_KEY_ISO_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Move_Line_Down")]
    pub const ISO_Move_Line_Down: Self = Self(ffi::GDK_KEY_ISO_Move_Line_Down as u32);
    #[doc(alias = "GDK_KEY_ISO_Move_Line_Up")]
    pub const ISO_Move_Line_Up: Self = Self(ffi::GDK_KEY_ISO_Move_Line_Up as u32);
    #[doc(alias = "GDK_KEY_ISO_Next_Group")]
    pub const ISO_Next_Group: Self = Self(ffi::GDK_KEY_ISO_Next_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_Next_Group_Lock")]
    pub const ISO_Next_Group_Lock: Self = Self(ffi::GDK_KEY_ISO_Next_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Line_Down")]
    pub const ISO_Partial_Line_Down: Self = Self(ffi::GDK_KEY_ISO_Partial_Line_Down as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Line_Up")]
    pub const ISO_Partial_Line_Up: Self = Self(ffi::GDK_KEY_ISO_Partial_Line_Up as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Space_Left")]
    pub const ISO_Partial_Space_Left: Self = Self(ffi::GDK_KEY_ISO_Partial_Space_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Space_Right")]
    pub const ISO_Partial_Space_Right: Self = Self(ffi::GDK_KEY_ISO_Partial_Space_Right as u32);
    #[doc(alias = "GDK_KEY_ISO_Prev_Group")]
    pub const ISO_Prev_Group: Self = Self(ffi::GDK_KEY_ISO_Prev_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_Prev_Group_Lock")]
    pub const ISO_Prev_Group_Lock: Self = Self(ffi::GDK_KEY_ISO_Prev_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Release_Both_Margins")]
    pub const ISO_Release_Both_Margins: Self = Self(ffi::GDK_KEY_ISO_Release_Both_Margins as u32);
    #[doc(alias = "GDK_KEY_ISO_Release_Margin_Left")]
    pub const ISO_Release_Margin_Left: Self = Self(ffi::GDK_KEY_ISO_Release_Margin_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Release_Margin_Right")]
    pub const ISO_Release_Margin_Right: Self = Self(ffi::GDK_KEY_ISO_Release_Margin_Right as u32);
    #[doc(alias = "GDK_KEY_ISO_Set_Margin_Left")]
    pub const ISO_Set_Margin_Left: Self = Self(ffi::GDK_KEY_ISO_Set_Margin_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Set_Margin_Right")]
    pub const ISO_Set_Margin_Right: Self = Self(ffi::GDK_KEY_ISO_Set_Margin_Right as u32);
    #[doc(alias = "GDK_KEY_Iabovedot")]
    pub const Iabovedot: Self = Self(ffi::GDK_KEY_Iabovedot as u32);
    #[doc(alias = "GDK_KEY_Iacute")]
    pub const Iacute: Self = Self(ffi::GDK_KEY_Iacute as u32);
    #[doc(alias = "GDK_KEY_Ibelowdot")]
    pub const Ibelowdot: Self = Self(ffi::GDK_KEY_Ibelowdot as u32);
    #[doc(alias = "GDK_KEY_Ibreve")]
    pub const Ibreve: Self = Self(ffi::GDK_KEY_Ibreve as u32);
    #[doc(alias = "GDK_KEY_Icircumflex")]
    pub const Icircumflex: Self = Self(ffi::GDK_KEY_Icircumflex as u32);
    #[doc(alias = "GDK_KEY_Idiaeresis")]
    pub const Idiaeresis: Self = Self(ffi::GDK_KEY_Idiaeresis as u32);
    #[doc(alias = "GDK_KEY_Igrave")]
    pub const Igrave: Self = Self(ffi::GDK_KEY_Igrave as u32);
    #[doc(alias = "GDK_KEY_Ihook")]
    pub const Ihook: Self = Self(ffi::GDK_KEY_Ihook as u32);
    #[doc(alias = "GDK_KEY_Imacron")]
    pub const Imacron: Self = Self(ffi::GDK_KEY_Imacron as u32);
    #[doc(alias = "GDK_KEY_Images")]
    pub const Images: Self = Self(ffi::GDK_KEY_Images as u32);
    #[doc(alias = "GDK_KEY_Info")]
    pub const Info: Self = Self(ffi::GDK_KEY_Info as u32);
    #[doc(alias = "GDK_KEY_Insert")]
    pub const Insert: Self = Self(ffi::GDK_KEY_Insert as u32);
    #[doc(alias = "GDK_KEY_Iogonek")]
    pub const Iogonek: Self = Self(ffi::GDK_KEY_Iogonek as u32);
    #[doc(alias = "GDK_KEY_Itilde")]
    pub const Itilde: Self = Self(ffi::GDK_KEY_Itilde as u32);
    #[doc(alias = "GDK_KEY_J")]
    pub const J: Self = Self(ffi::GDK_KEY_J as u32);
    #[doc(alias = "GDK_KEY_Jcircumflex")]
    pub const Jcircumflex: Self = Self(ffi::GDK_KEY_Jcircumflex as u32);
    #[doc(alias = "GDK_KEY_Journal")]
    pub const Journal: Self = Self(ffi::GDK_KEY_Journal as u32);
    #[doc(alias = "GDK_KEY_K")]
    pub const K: Self = Self(ffi::GDK_KEY_K as u32);
    #[doc(alias = "GDK_KEY_KP_0")]
    pub const KP_0: Self = Self(ffi::GDK_KEY_KP_0 as u32);
    #[doc(alias = "GDK_KEY_KP_1")]
    pub const KP_1: Self = Self(ffi::GDK_KEY_KP_1 as u32);
    #[doc(alias = "GDK_KEY_KP_2")]
    pub const KP_2: Self = Self(ffi::GDK_KEY_KP_2 as u32);
    #[doc(alias = "GDK_KEY_KP_3")]
    pub const KP_3: Self = Self(ffi::GDK_KEY_KP_3 as u32);
    #[doc(alias = "GDK_KEY_KP_4")]
    pub const KP_4: Self = Self(ffi::GDK_KEY_KP_4 as u32);
    #[doc(alias = "GDK_KEY_KP_5")]
    pub const KP_5: Self = Self(ffi::GDK_KEY_KP_5 as u32);
    #[doc(alias = "GDK_KEY_KP_6")]
    pub const KP_6: Self = Self(ffi::GDK_KEY_KP_6 as u32);
    #[doc(alias = "GDK_KEY_KP_7")]
    pub const KP_7: Self = Self(ffi::GDK_KEY_KP_7 as u32);
    #[doc(alias = "GDK_KEY_KP_8")]
    pub const KP_8: Self = Self(ffi::GDK_KEY_KP_8 as u32);
    #[doc(alias = "GDK_KEY_KP_9")]
    pub const KP_9: Self = Self(ffi::GDK_KEY_KP_9 as u32);
    #[doc(alias = "GDK_KEY_KP_Add")]
    pub const KP_Add: Self = Self(ffi::GDK_KEY_KP_Add as u32);
    #[doc(alias = "GDK_KEY_KP_Begin")]
    pub const KP_Begin: Self = Self(ffi::GDK_KEY_KP_Begin as u32);
    #[doc(alias = "GDK_KEY_KP_Decimal")]
    pub const KP_Decimal: Self = Self(ffi::GDK_KEY_KP_Decimal as u32);
    #[doc(alias = "GDK_KEY_KP_Delete")]
    pub const KP_Delete: Self = Self(ffi::GDK_KEY_KP_Delete as u32);
    #[doc(alias = "GDK_KEY_KP_Divide")]
    pub const KP_Divide: Self = Self(ffi::GDK_KEY_KP_Divide as u32);
    #[doc(alias = "GDK_KEY_KP_Down")]
    pub const KP_Down: Self = Self(ffi::GDK_KEY_KP_Down as u32);
    #[doc(alias = "GDK_KEY_KP_End")]
    pub const KP_End: Self = Self(ffi::GDK_KEY_KP_End as u32);
    #[doc(alias = "GDK_KEY_KP_Enter")]
    pub const KP_Enter: Self = Self(ffi::GDK_KEY_KP_Enter as u32);
    #[doc(alias = "GDK_KEY_KP_Equal")]
    pub const KP_Equal: Self = Self(ffi::GDK_KEY_KP_Equal as u32);
    #[doc(alias = "GDK_KEY_KP_F1")]
    pub const KP_F1: Self = Self(ffi::GDK_KEY_KP_F1 as u32);
    #[doc(alias = "GDK_KEY_KP_F2")]
    pub const KP_F2: Self = Self(ffi::GDK_KEY_KP_F2 as u32);
    #[doc(alias = "GDK_KEY_KP_F3")]
    pub const KP_F3: Self = Self(ffi::GDK_KEY_KP_F3 as u32);
    #[doc(alias = "GDK_KEY_KP_F4")]
    pub const KP_F4: Self = Self(ffi::GDK_KEY_KP_F4 as u32);
    #[doc(alias = "GDK_KEY_KP_Home")]
    pub const KP_Home: Self = Self(ffi::GDK_KEY_KP_Home as u32);
    #[doc(alias = "GDK_KEY_KP_Insert")]
    pub const KP_Insert: Self = Self(ffi::GDK_KEY_KP_Insert as u32);
    #[doc(alias = "GDK_KEY_KP_Left")]
    pub const KP_Left: Self = Self(ffi::GDK_KEY_KP_Left as u32);
    #[doc(alias = "GDK_KEY_KP_Multiply")]
    pub const KP_Multiply: Self = Self(ffi::GDK_KEY_KP_Multiply as u32);
    #[doc(alias = "GDK_KEY_KP_Next")]
    pub const KP_Next: Self = Self(ffi::GDK_KEY_KP_Next as u32);
    #[doc(alias = "GDK_KEY_KP_Page_Down")]
    pub const KP_Page_Down: Self = Self(ffi::GDK_KEY_KP_Page_Down as u32);
    #[doc(alias = "GDK_KEY_KP_Page_Up")]
    pub const KP_Page_Up: Self = Self(ffi::GDK_KEY_KP_Page_Up as u32);
    #[doc(alias = "GDK_KEY_KP_Prior")]
    pub const KP_Prior: Self = Self(ffi::GDK_KEY_KP_Prior as u32);
    #[doc(alias = "GDK_KEY_KP_Right")]
    pub const KP_Right: Self = Self(ffi::GDK_KEY_KP_Right as u32);
    #[doc(alias = "GDK_KEY_KP_Separator")]
    pub const KP_Separator: Self = Self(ffi::GDK_KEY_KP_Separator as u32);
    #[doc(alias = "GDK_KEY_KP_Space")]
    pub const KP_Space: Self = Self(ffi::GDK_KEY_KP_Space as u32);
    #[doc(alias = "GDK_KEY_KP_Subtract")]
    pub const KP_Subtract: Self = Self(ffi::GDK_KEY_KP_Subtract as u32);
    #[doc(alias = "GDK_KEY_KP_Tab")]
    pub const KP_Tab: Self = Self(ffi::GDK_KEY_KP_Tab as u32);
    #[doc(alias = "GDK_KEY_KP_Up")]
    pub const KP_Up: Self = Self(ffi::GDK_KEY_KP_Up as u32);
    #[doc(alias = "GDK_KEY_Kana_Lock")]
    pub const Kana_Lock: Self = Self(ffi::GDK_KEY_Kana_Lock as u32);
    #[doc(alias = "GDK_KEY_Kana_Shift")]
    pub const Kana_Shift: Self = Self(ffi::GDK_KEY_Kana_Shift as u32);
    #[doc(alias = "GDK_KEY_Kanji")]
    pub const Kanji: Self = Self(ffi::GDK_KEY_Kanji as u32);
    #[doc(alias = "GDK_KEY_Kanji_Bangou")]
    pub const Kanji_Bangou: Self = Self(ffi::GDK_KEY_Kanji_Bangou as u32);
    #[doc(alias = "GDK_KEY_Katakana")]
    pub const Katakana: Self = Self(ffi::GDK_KEY_Katakana as u32);
    #[doc(alias = "GDK_KEY_KbdBrightnessDown")]
    pub const KbdBrightnessDown: Self = Self(ffi::GDK_KEY_KbdBrightnessDown as u32);
    #[doc(alias = "GDK_KEY_KbdBrightnessUp")]
    pub const KbdBrightnessUp: Self = Self(ffi::GDK_KEY_KbdBrightnessUp as u32);
    #[doc(alias = "GDK_KEY_KbdInputAssistAccept")]
    pub const KbdInputAssistAccept: Self = Self(ffi::GDK_KEY_KbdInputAssistAccept as u32);
    #[doc(alias = "GDK_KEY_KbdInputAssistCancel")]
    pub const KbdInputAssistCancel: Self = Self(ffi::GDK_KEY_KbdInputAssistCancel as u32);
    #[doc(alias = "GDK_KEY_KbdInputAssistNext")]
    pub const KbdInputAssistNext: Self = Self(ffi::GDK_KEY_KbdInputAssistNext as u32);
    #[doc(alias = "GDK_KEY_KbdInputAssistNextgroup")]
    pub const KbdInputAssistNextgroup: Self = Self(ffi::GDK_KEY_KbdInputAssistNextgroup as u32);
    #[doc(alias = "GDK_KEY_KbdInputAssistPrev")]
    pub const KbdInputAssistPrev: Self = Self(ffi::GDK_KEY_KbdInputAssistPrev as u32);
    #[doc(alias = "GDK_KEY_KbdInputAssistPrevgroup")]
    pub const KbdInputAssistPrevgroup: Self = Self(ffi::GDK_KEY_KbdInputAssistPrevgroup as u32);
    #[doc(alias = "GDK_KEY_KbdLcdMenu1")]
    pub const KbdLcdMenu1: Self = Self(ffi::GDK_KEY_KbdLcdMenu1 as u32);
    #[doc(alias = "GDK_KEY_KbdLcdMenu2")]
    pub const KbdLcdMenu2: Self = Self(ffi::GDK_KEY_KbdLcdMenu2 as u32);
    #[doc(alias = "GDK_KEY_KbdLcdMenu3")]
    pub const KbdLcdMenu3: Self = Self(ffi::GDK_KEY_KbdLcdMenu3 as u32);
    #[doc(alias = "GDK_KEY_KbdLcdMenu4")]
    pub const KbdLcdMenu4: Self = Self(ffi::GDK_KEY_KbdLcdMenu4 as u32);
    #[doc(alias = "GDK_KEY_KbdLcdMenu5")]
    pub const KbdLcdMenu5: Self = Self(ffi::GDK_KEY_KbdLcdMenu5 as u32);
    #[doc(alias = "GDK_KEY_KbdLightOnOff")]
    pub const KbdLightOnOff: Self = Self(ffi::GDK_KEY_KbdLightOnOff as u32);
    #[doc(alias = "GDK_KEY_Kcedilla")]
    pub const Kcedilla: Self = Self(ffi::GDK_KEY_Kcedilla as u32);
    #[doc(alias = "GDK_KEY_Keyboard")]
    pub const Keyboard: Self = Self(ffi::GDK_KEY_Keyboard as u32);
    #[doc(alias = "GDK_KEY_Korean_Won")]
    pub const Korean_Won: Self = Self(ffi::GDK_KEY_Korean_Won as u32);
    #[doc(alias = "GDK_KEY_L")]
    pub const L: Self = Self(ffi::GDK_KEY_L as u32);
    #[doc(alias = "GDK_KEY_L1")]
    pub const L1: Self = Self(ffi::GDK_KEY_L1 as u32);
    #[doc(alias = "GDK_KEY_L10")]
    pub const L10: Self = Self(ffi::GDK_KEY_L10 as u32);
    #[doc(alias = "GDK_KEY_L2")]
    pub const L2: Self = Self(ffi::GDK_KEY_L2 as u32);
    #[doc(alias = "GDK_KEY_L3")]
    pub const L3: Self = Self(ffi::GDK_KEY_L3 as u32);
    #[doc(alias = "GDK_KEY_L4")]
    pub const L4: Self = Self(ffi::GDK_KEY_L4 as u32);
    #[doc(alias = "GDK_KEY_L5")]
    pub const L5: Self = Self(ffi::GDK_KEY_L5 as u32);
    #[doc(alias = "GDK_KEY_L6")]
    pub const L6: Self = Self(ffi::GDK_KEY_L6 as u32);
    #[doc(alias = "GDK_KEY_L7")]
    pub const L7: Self = Self(ffi::GDK_KEY_L7 as u32);
    #[doc(alias = "GDK_KEY_L8")]
    pub const L8: Self = Self(ffi::GDK_KEY_L8 as u32);
    #[doc(alias = "GDK_KEY_L9")]
    pub const L9: Self = Self(ffi::GDK_KEY_L9 as u32);
    #[doc(alias = "GDK_KEY_Lacute")]
    pub const Lacute: Self = Self(ffi::GDK_KEY_Lacute as u32);
    #[doc(alias = "GDK_KEY_Last_Virtual_Screen")]
    pub const Last_Virtual_Screen: Self = Self(ffi::GDK_KEY_Last_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_Launch0")]
    pub const Launch0: Self = Self(ffi::GDK_KEY_Launch0 as u32);
    #[doc(alias = "GDK_KEY_Launch1")]
    pub const Launch1: Self = Self(ffi::GDK_KEY_Launch1 as u32);
    #[doc(alias = "GDK_KEY_Launch2")]
    pub const Launch2: Self = Self(ffi::GDK_KEY_Launch2 as u32);
    #[doc(alias = "GDK_KEY_Launch3")]
    pub const Launch3: Self = Self(ffi::GDK_KEY_Launch3 as u32);
    #[doc(alias = "GDK_KEY_Launch4")]
    pub const Launch4: Self = Self(ffi::GDK_KEY_Launch4 as u32);
    #[doc(alias = "GDK_KEY_Launch5")]
    pub const Launch5: Self = Self(ffi::GDK_KEY_Launch5 as u32);
    #[doc(alias = "GDK_KEY_Launch6")]
    pub const Launch6: Self = Self(ffi::GDK_KEY_Launch6 as u32);
    #[doc(alias = "GDK_KEY_Launch7")]
    pub const Launch7: Self = Self(ffi::GDK_KEY_Launch7 as u32);
    #[doc(alias = "GDK_KEY_Launch8")]
    pub const Launch8: Self = Self(ffi::GDK_KEY_Launch8 as u32);
    #[doc(alias = "GDK_KEY_Launch9")]
    pub const Launch9: Self = Self(ffi::GDK_KEY_Launch9 as u32);
    #[doc(alias = "GDK_KEY_LaunchA")]
    pub const LaunchA: Self = Self(ffi::GDK_KEY_LaunchA as u32);
    #[doc(alias = "GDK_KEY_LaunchB")]
    pub const LaunchB: Self = Self(ffi::GDK_KEY_LaunchB as u32);
    #[doc(alias = "GDK_KEY_LaunchC")]
    pub const LaunchC: Self = Self(ffi::GDK_KEY_LaunchC as u32);
    #[doc(alias = "GDK_KEY_LaunchD")]
    pub const LaunchD: Self = Self(ffi::GDK_KEY_LaunchD as u32);
    #[doc(alias = "GDK_KEY_LaunchE")]
    pub const LaunchE: Self = Self(ffi::GDK_KEY_LaunchE as u32);
    #[doc(alias = "GDK_KEY_LaunchF")]
    pub const LaunchF: Self = Self(ffi::GDK_KEY_LaunchF as u32);
    #[doc(alias = "GDK_KEY_Lbelowdot")]
    pub const Lbelowdot: Self = Self(ffi::GDK_KEY_Lbelowdot as u32);
    #[doc(alias = "GDK_KEY_Lcaron")]
    pub const Lcaron: Self = Self(ffi::GDK_KEY_Lcaron as u32);
    #[doc(alias = "GDK_KEY_Lcedilla")]
    pub const Lcedilla: Self = Self(ffi::GDK_KEY_Lcedilla as u32);
    #[doc(alias = "GDK_KEY_Left")]
    pub const Left: Self = Self(ffi::GDK_KEY_Left as u32);
    #[doc(alias = "GDK_KEY_LeftDown")]
    pub const LeftDown: Self = Self(ffi::GDK_KEY_LeftDown as u32);
    #[doc(alias = "GDK_KEY_LeftUp")]
    pub const LeftUp: Self = Self(ffi::GDK_KEY_LeftUp as u32);
    #[doc(alias = "GDK_KEY_LightBulb")]
    pub const LightBulb: Self = Self(ffi::GDK_KEY_LightBulb as u32);
    #[doc(alias = "GDK_KEY_LightsToggle")]
    pub const LightsToggle: Self = Self(ffi::GDK_KEY_LightsToggle as u32);
    #[doc(alias = "GDK_KEY_Linefeed")]
    pub const Linefeed: Self = Self(ffi::GDK_KEY_Linefeed as u32);
    #[doc(alias = "GDK_KEY_LiraSign")]
    pub const LiraSign: Self = Self(ffi::GDK_KEY_LiraSign as u32);
    #[doc(alias = "GDK_KEY_LogGrabInfo")]
    pub const LogGrabInfo: Self = Self(ffi::GDK_KEY_LogGrabInfo as u32);
    #[doc(alias = "GDK_KEY_LogOff")]
    pub const LogOff: Self = Self(ffi::GDK_KEY_LogOff as u32);
    #[doc(alias = "GDK_KEY_LogWindowTree")]
    pub const LogWindowTree: Self = Self(ffi::GDK_KEY_LogWindowTree as u32);
    #[doc(alias = "GDK_KEY_Lstroke")]
    pub const Lstroke: Self = Self(ffi::GDK_KEY_Lstroke as u32);
    #[doc(alias = "GDK_KEY_M")]
    pub const M: Self = Self(ffi::GDK_KEY_M as u32);
    #[doc(alias = "GDK_KEY_Mabovedot")]
    pub const Mabovedot: Self = Self(ffi::GDK_KEY_Mabovedot as u32);
    #[doc(alias = "GDK_KEY_Macedonia_DSE")]
    pub const Macedonia_DSE: Self = Self(ffi::GDK_KEY_Macedonia_DSE as u32);
    #[doc(alias = "GDK_KEY_Macedonia_GJE")]
    pub const Macedonia_GJE: Self = Self(ffi::GDK_KEY_Macedonia_GJE as u32);
    #[doc(alias = "GDK_KEY_Macedonia_KJE")]
    pub const Macedonia_KJE: Self = Self(ffi::GDK_KEY_Macedonia_KJE as u32);
    #[doc(alias = "GDK_KEY_Macedonia_dse")]
    pub const Macedonia_dse: Self = Self(ffi::GDK_KEY_Macedonia_dse as u32);
    #[doc(alias = "GDK_KEY_Macedonia_gje")]
    pub const Macedonia_gje: Self = Self(ffi::GDK_KEY_Macedonia_gje as u32);
    #[doc(alias = "GDK_KEY_Macedonia_kje")]
    pub const Macedonia_kje: Self = Self(ffi::GDK_KEY_Macedonia_kje as u32);
    #[doc(alias = "GDK_KEY_Macro1")]
    pub const Macro1: Self = Self(ffi::GDK_KEY_Macro1 as u32);
    #[doc(alias = "GDK_KEY_Macro10")]
    pub const Macro10: Self = Self(ffi::GDK_KEY_Macro10 as u32);
    #[doc(alias = "GDK_KEY_Macro11")]
    pub const Macro11: Self = Self(ffi::GDK_KEY_Macro11 as u32);
    #[doc(alias = "GDK_KEY_Macro12")]
    pub const Macro12: Self = Self(ffi::GDK_KEY_Macro12 as u32);
    #[doc(alias = "GDK_KEY_Macro13")]
    pub const Macro13: Self = Self(ffi::GDK_KEY_Macro13 as u32);
    #[doc(alias = "GDK_KEY_Macro14")]
    pub const Macro14: Self = Self(ffi::GDK_KEY_Macro14 as u32);
    #[doc(alias = "GDK_KEY_Macro15")]
    pub const Macro15: Self = Self(ffi::GDK_KEY_Macro15 as u32);
    #[doc(alias = "GDK_KEY_Macro16")]
    pub const Macro16: Self = Self(ffi::GDK_KEY_Macro16 as u32);
    #[doc(alias = "GDK_KEY_Macro17")]
    pub const Macro17: Self = Self(ffi::GDK_KEY_Macro17 as u32);
    #[doc(alias = "GDK_KEY_Macro18")]
    pub const Macro18: Self = Self(ffi::GDK_KEY_Macro18 as u32);
    #[doc(alias = "GDK_KEY_Macro19")]
    pub const Macro19: Self = Self(ffi::GDK_KEY_Macro19 as u32);
    #[doc(alias = "GDK_KEY_Macro2")]
    pub const Macro2: Self = Self(ffi::GDK_KEY_Macro2 as u32);
    #[doc(alias = "GDK_KEY_Macro20")]
    pub const Macro20: Self = Self(ffi::GDK_KEY_Macro20 as u32);
    #[doc(alias = "GDK_KEY_Macro21")]
    pub const Macro21: Self = Self(ffi::GDK_KEY_Macro21 as u32);
    #[doc(alias = "GDK_KEY_Macro22")]
    pub const Macro22: Self = Self(ffi::GDK_KEY_Macro22 as u32);
    #[doc(alias = "GDK_KEY_Macro23")]
    pub const Macro23: Self = Self(ffi::GDK_KEY_Macro23 as u32);
    #[doc(alias = "GDK_KEY_Macro24")]
    pub const Macro24: Self = Self(ffi::GDK_KEY_Macro24 as u32);
    #[doc(alias = "GDK_KEY_Macro25")]
    pub const Macro25: Self = Self(ffi::GDK_KEY_Macro25 as u32);
    #[doc(alias = "GDK_KEY_Macro26")]
    pub const Macro26: Self = Self(ffi::GDK_KEY_Macro26 as u32);
    #[doc(alias = "GDK_KEY_Macro27")]
    pub const Macro27: Self = Self(ffi::GDK_KEY_Macro27 as u32);
    #[doc(alias = "GDK_KEY_Macro28")]
    pub const Macro28: Self = Self(ffi::GDK_KEY_Macro28 as u32);
    #[doc(alias = "GDK_KEY_Macro29")]
    pub const Macro29: Self = Self(ffi::GDK_KEY_Macro29 as u32);
    #[doc(alias = "GDK_KEY_Macro3")]
    pub const Macro3: Self = Self(ffi::GDK_KEY_Macro3 as u32);
    #[doc(alias = "GDK_KEY_Macro30")]
    pub const Macro30: Self = Self(ffi::GDK_KEY_Macro30 as u32);
    #[doc(alias = "GDK_KEY_Macro4")]
    pub const Macro4: Self = Self(ffi::GDK_KEY_Macro4 as u32);
    #[doc(alias = "GDK_KEY_Macro5")]
    pub const Macro5: Self = Self(ffi::GDK_KEY_Macro5 as u32);
    #[doc(alias = "GDK_KEY_Macro6")]
    pub const Macro6: Self = Self(ffi::GDK_KEY_Macro6 as u32);
    #[doc(alias = "GDK_KEY_Macro7")]
    pub const Macro7: Self = Self(ffi::GDK_KEY_Macro7 as u32);
    #[doc(alias = "GDK_KEY_Macro8")]
    pub const Macro8: Self = Self(ffi::GDK_KEY_Macro8 as u32);
    #[doc(alias = "GDK_KEY_Macro9")]
    pub const Macro9: Self = Self(ffi::GDK_KEY_Macro9 as u32);
    #[doc(alias = "GDK_KEY_MacroPreset1")]
    pub const MacroPreset1: Self = Self(ffi::GDK_KEY_MacroPreset1 as u32);
    #[doc(alias = "GDK_KEY_MacroPreset2")]
    pub const MacroPreset2: Self = Self(ffi::GDK_KEY_MacroPreset2 as u32);
    #[doc(alias = "GDK_KEY_MacroPreset3")]
    pub const MacroPreset3: Self = Self(ffi::GDK_KEY_MacroPreset3 as u32);
    #[doc(alias = "GDK_KEY_MacroPresetCycle")]
    pub const MacroPresetCycle: Self = Self(ffi::GDK_KEY_MacroPresetCycle as u32);
    #[doc(alias = "GDK_KEY_MacroRecordStart")]
    pub const MacroRecordStart: Self = Self(ffi::GDK_KEY_MacroRecordStart as u32);
    #[doc(alias = "GDK_KEY_MacroRecordStop")]
    pub const MacroRecordStop: Self = Self(ffi::GDK_KEY_MacroRecordStop as u32);
    #[doc(alias = "GDK_KEY_Mae_Koho")]
    pub const Mae_Koho: Self = Self(ffi::GDK_KEY_Mae_Koho as u32);
    #[doc(alias = "GDK_KEY_Mail")]
    pub const Mail: Self = Self(ffi::GDK_KEY_Mail as u32);
    #[doc(alias = "GDK_KEY_MailForward")]
    pub const MailForward: Self = Self(ffi::GDK_KEY_MailForward as u32);
    #[doc(alias = "GDK_KEY_MarkWaypoint")]
    pub const MarkWaypoint: Self = Self(ffi::GDK_KEY_MarkWaypoint as u32);
    #[doc(alias = "GDK_KEY_Market")]
    pub const Market: Self = Self(ffi::GDK_KEY_Market as u32);
    #[doc(alias = "GDK_KEY_Massyo")]
    pub const Massyo: Self = Self(ffi::GDK_KEY_Massyo as u32);
    #[doc(alias = "GDK_KEY_MediaRepeat")]
    pub const MediaRepeat: Self = Self(ffi::GDK_KEY_MediaRepeat as u32);
    #[doc(alias = "GDK_KEY_MediaTopMenu")]
    pub const MediaTopMenu: Self = Self(ffi::GDK_KEY_MediaTopMenu as u32);
    #[doc(alias = "GDK_KEY_Meeting")]
    pub const Meeting: Self = Self(ffi::GDK_KEY_Meeting as u32);
    #[doc(alias = "GDK_KEY_Memo")]
    pub const Memo: Self = Self(ffi::GDK_KEY_Memo as u32);
    #[doc(alias = "GDK_KEY_Menu")]
    pub const Menu: Self = Self(ffi::GDK_KEY_Menu as u32);
    #[doc(alias = "GDK_KEY_MenuKB")]
    pub const MenuKB: Self = Self(ffi::GDK_KEY_MenuKB as u32);
    #[doc(alias = "GDK_KEY_MenuPB")]
    pub const MenuPB: Self = Self(ffi::GDK_KEY_MenuPB as u32);
    #[doc(alias = "GDK_KEY_Messenger")]
    pub const Messenger: Self = Self(ffi::GDK_KEY_Messenger as u32);
    #[doc(alias = "GDK_KEY_Meta_L")]
    pub const Meta_L: Self = Self(ffi::GDK_KEY_Meta_L as u32);
    #[doc(alias = "GDK_KEY_Meta_R")]
    pub const Meta_R: Self = Self(ffi::GDK_KEY_Meta_R as u32);
    #[doc(alias = "GDK_KEY_MillSign")]
    pub const MillSign: Self = Self(ffi::GDK_KEY_MillSign as u32);
    #[doc(alias = "GDK_KEY_ModeLock")]
    pub const ModeLock: Self = Self(ffi::GDK_KEY_ModeLock as u32);
    #[doc(alias = "GDK_KEY_Mode_switch")]
    pub const Mode_switch: Self = Self(ffi::GDK_KEY_Mode_switch as u32);
    #[doc(alias = "GDK_KEY_MonBrightnessCycle")]
    pub const MonBrightnessCycle: Self = Self(ffi::GDK_KEY_MonBrightnessCycle as u32);
    #[doc(alias = "GDK_KEY_MonBrightnessDown")]
    pub const MonBrightnessDown: Self = Self(ffi::GDK_KEY_MonBrightnessDown as u32);
    #[doc(alias = "GDK_KEY_MonBrightnessUp")]
    pub const MonBrightnessUp: Self = Self(ffi::GDK_KEY_MonBrightnessUp as u32);
    #[doc(alias = "GDK_KEY_MouseKeys_Accel_Enable")]
    pub const MouseKeys_Accel_Enable: Self = Self(ffi::GDK_KEY_MouseKeys_Accel_Enable as u32);
    #[doc(alias = "GDK_KEY_MouseKeys_Enable")]
    pub const MouseKeys_Enable: Self = Self(ffi::GDK_KEY_MouseKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_Muhenkan")]
    pub const Muhenkan: Self = Self(ffi::GDK_KEY_Muhenkan as u32);
    #[doc(alias = "GDK_KEY_Multi_key")]
    pub const Multi_key: Self = Self(ffi::GDK_KEY_Multi_key as u32);
    #[doc(alias = "GDK_KEY_MultipleCandidate")]
    pub const MultipleCandidate: Self = Self(ffi::GDK_KEY_MultipleCandidate as u32);
    #[doc(alias = "GDK_KEY_Music")]
    pub const Music: Self = Self(ffi::GDK_KEY_Music as u32);
    #[doc(alias = "GDK_KEY_MyComputer")]
    pub const MyComputer: Self = Self(ffi::GDK_KEY_MyComputer as u32);
    #[doc(alias = "GDK_KEY_MySites")]
    pub const MySites: Self = Self(ffi::GDK_KEY_MySites as u32);
    #[doc(alias = "GDK_KEY_N")]
    pub const N: Self = Self(ffi::GDK_KEY_N as u32);
    #[doc(alias = "GDK_KEY_Nacute")]
    pub const Nacute: Self = Self(ffi::GDK_KEY_Nacute as u32);
    #[doc(alias = "GDK_KEY_NairaSign")]
    pub const NairaSign: Self = Self(ffi::GDK_KEY_NairaSign as u32);
    #[doc(alias = "GDK_KEY_NavChart")]
    pub const NavChart: Self = Self(ffi::GDK_KEY_NavChart as u32);
    #[doc(alias = "GDK_KEY_NavInfo")]
    pub const NavInfo: Self = Self(ffi::GDK_KEY_NavInfo as u32);
    #[doc(alias = "GDK_KEY_Ncaron")]
    pub const Ncaron: Self = Self(ffi::GDK_KEY_Ncaron as u32);
    #[doc(alias = "GDK_KEY_Ncedilla")]
    pub const Ncedilla: Self = Self(ffi::GDK_KEY_Ncedilla as u32);
    #[doc(alias = "GDK_KEY_New")]
    pub const New: Self = Self(ffi::GDK_KEY_New as u32);
    #[doc(alias = "GDK_KEY_NewSheqelSign")]
    pub const NewSheqelSign: Self = Self(ffi::GDK_KEY_NewSheqelSign as u32);
    #[doc(alias = "GDK_KEY_News")]
    pub const News: Self = Self(ffi::GDK_KEY_News as u32);
    #[doc(alias = "GDK_KEY_Next")]
    pub const Next: Self = Self(ffi::GDK_KEY_Next as u32);
    #[doc(alias = "GDK_KEY_NextElement")]
    pub const NextElement: Self = Self(ffi::GDK_KEY_NextElement as u32);
    #[doc(alias = "GDK_KEY_NextFavorite")]
    pub const NextFavorite: Self = Self(ffi::GDK_KEY_NextFavorite as u32);
    #[doc(alias = "GDK_KEY_Next_VMode")]
    pub const Next_VMode: Self = Self(ffi::GDK_KEY_Next_VMode as u32);
    #[doc(alias = "GDK_KEY_Next_Virtual_Screen")]
    pub const Next_Virtual_Screen: Self = Self(ffi::GDK_KEY_Next_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_NotificationCenter")]
    pub const NotificationCenter: Self = Self(ffi::GDK_KEY_NotificationCenter as u32);
    #[doc(alias = "GDK_KEY_Ntilde")]
    pub const Ntilde: Self = Self(ffi::GDK_KEY_Ntilde as u32);
    #[doc(alias = "GDK_KEY_Num_Lock")]
    pub const Num_Lock: Self = Self(ffi::GDK_KEY_Num_Lock as u32);
    #[doc(alias = "GDK_KEY_Numeric0")]
    pub const Numeric0: Self = Self(ffi::GDK_KEY_Numeric0 as u32);
    #[doc(alias = "GDK_KEY_Numeric1")]
    pub const Numeric1: Self = Self(ffi::GDK_KEY_Numeric1 as u32);
    #[doc(alias = "GDK_KEY_Numeric11")]
    pub const Numeric11: Self = Self(ffi::GDK_KEY_Numeric11 as u32);
    #[doc(alias = "GDK_KEY_Numeric12")]
    pub const Numeric12: Self = Self(ffi::GDK_KEY_Numeric12 as u32);
    #[doc(alias = "GDK_KEY_Numeric2")]
    pub const Numeric2: Self = Self(ffi::GDK_KEY_Numeric2 as u32);
    #[doc(alias = "GDK_KEY_Numeric3")]
    pub const Numeric3: Self = Self(ffi::GDK_KEY_Numeric3 as u32);
    #[doc(alias = "GDK_KEY_Numeric4")]
    pub const Numeric4: Self = Self(ffi::GDK_KEY_Numeric4 as u32);
    #[doc(alias = "GDK_KEY_Numeric5")]
    pub const Numeric5: Self = Self(ffi::GDK_KEY_Numeric5 as u32);
    #[doc(alias = "GDK_KEY_Numeric6")]
    pub const Numeric6: Self = Self(ffi::GDK_KEY_Numeric6 as u32);
    #[doc(alias = "GDK_KEY_Numeric7")]
    pub const Numeric7: Self = Self(ffi::GDK_KEY_Numeric7 as u32);
    #[doc(alias = "GDK_KEY_Numeric8")]
    pub const Numeric8: Self = Self(ffi::GDK_KEY_Numeric8 as u32);
    #[doc(alias = "GDK_KEY_Numeric9")]
    pub const Numeric9: Self = Self(ffi::GDK_KEY_Numeric9 as u32);
    #[doc(alias = "GDK_KEY_NumericA")]
    pub const NumericA: Self = Self(ffi::GDK_KEY_NumericA as u32);
    #[doc(alias = "GDK_KEY_NumericB")]
    pub const NumericB: Self = Self(ffi::GDK_KEY_NumericB as u32);
    #[doc(alias = "GDK_KEY_NumericC")]
    pub const NumericC: Self = Self(ffi::GDK_KEY_NumericC as u32);
    #[doc(alias = "GDK_KEY_NumericD")]
    pub const NumericD: Self = Self(ffi::GDK_KEY_NumericD as u32);
    #[doc(alias = "GDK_KEY_NumericPound")]
    pub const NumericPound: Self = Self(ffi::GDK_KEY_NumericPound as u32);
    #[doc(alias = "GDK_KEY_NumericStar")]
    pub const NumericStar: Self = Self(ffi::GDK_KEY_NumericStar as u32);
    #[doc(alias = "GDK_KEY_O")]
    pub const O: Self = Self(ffi::GDK_KEY_O as u32);
    #[doc(alias = "GDK_KEY_OE")]
    pub const OE: Self = Self(ffi::GDK_KEY_OE as u32);
    #[doc(alias = "GDK_KEY_Oacute")]
    pub const Oacute: Self = Self(ffi::GDK_KEY_Oacute as u32);
    #[doc(alias = "GDK_KEY_Obarred")]
    pub const Obarred: Self = Self(ffi::GDK_KEY_Obarred as u32);
    #[doc(alias = "GDK_KEY_Obelowdot")]
    pub const Obelowdot: Self = Self(ffi::GDK_KEY_Obelowdot as u32);
    #[doc(alias = "GDK_KEY_Ocaron")]
    pub const Ocaron: Self = Self(ffi::GDK_KEY_Ocaron as u32);
    #[doc(alias = "GDK_KEY_Ocircumflex")]
    pub const Ocircumflex: Self = Self(ffi::GDK_KEY_Ocircumflex as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexacute")]
    pub const Ocircumflexacute: Self = Self(ffi::GDK_KEY_Ocircumflexacute as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexbelowdot")]
    pub const Ocircumflexbelowdot: Self = Self(ffi::GDK_KEY_Ocircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexgrave")]
    pub const Ocircumflexgrave: Self = Self(ffi::GDK_KEY_Ocircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexhook")]
    pub const Ocircumflexhook: Self = Self(ffi::GDK_KEY_Ocircumflexhook as u32);
    #[doc(alias = "GDK_KEY_Ocircumflextilde")]
    pub const Ocircumflextilde: Self = Self(ffi::GDK_KEY_Ocircumflextilde as u32);
    #[doc(alias = "GDK_KEY_Odiaeresis")]
    pub const Odiaeresis: Self = Self(ffi::GDK_KEY_Odiaeresis as u32);
    #[doc(alias = "GDK_KEY_Odoubleacute")]
    pub const Odoubleacute: Self = Self(ffi::GDK_KEY_Odoubleacute as u32);
    #[doc(alias = "GDK_KEY_OfficeHome")]
    pub const OfficeHome: Self = Self(ffi::GDK_KEY_OfficeHome as u32);
    #[doc(alias = "GDK_KEY_Ograve")]
    pub const Ograve: Self = Self(ffi::GDK_KEY_Ograve as u32);
    #[doc(alias = "GDK_KEY_Ohook")]
    pub const Ohook: Self = Self(ffi::GDK_KEY_Ohook as u32);
    #[doc(alias = "GDK_KEY_Ohorn")]
    pub const Ohorn: Self = Self(ffi::GDK_KEY_Ohorn as u32);
    #[doc(alias = "GDK_KEY_Ohornacute")]
    pub const Ohornacute: Self = Self(ffi::GDK_KEY_Ohornacute as u32);
    #[doc(alias = "GDK_KEY_Ohornbelowdot")]
    pub const Ohornbelowdot: Self = Self(ffi::GDK_KEY_Ohornbelowdot as u32);
    #[doc(alias = "GDK_KEY_Ohorngrave")]
    pub const Ohorngrave: Self = Self(ffi::GDK_KEY_Ohorngrave as u32);
    #[doc(alias = "GDK_KEY_Ohornhook")]
    pub const Ohornhook: Self = Self(ffi::GDK_KEY_Ohornhook as u32);
    #[doc(alias = "GDK_KEY_Ohorntilde")]
    pub const Ohorntilde: Self = Self(ffi::GDK_KEY_Ohorntilde as u32);
    #[doc(alias = "GDK_KEY_Omacron")]
    pub const Omacron: Self = Self(ffi::GDK_KEY_Omacron as u32);
    #[doc(alias = "GDK_KEY_OnScreenKeyboard")]
    pub const OnScreenKeyboard: Self = Self(ffi::GDK_KEY_OnScreenKeyboard as u32);
    #[doc(alias = "GDK_KEY_Ooblique")]
    pub const Ooblique: Self = Self(ffi::GDK_KEY_Ooblique as u32);
    #[doc(alias = "GDK_KEY_Open")]
    pub const Open: Self = Self(ffi::GDK_KEY_Open as u32);
    #[doc(alias = "GDK_KEY_OpenURL")]
    pub const OpenURL: Self = Self(ffi::GDK_KEY_OpenURL as u32);
    #[doc(alias = "GDK_KEY_Option")]
    pub const Option: Self = Self(ffi::GDK_KEY_Option as u32);
    #[doc(alias = "GDK_KEY_Oslash")]
    pub const Oslash: Self = Self(ffi::GDK_KEY_Oslash as u32);
    #[doc(alias = "GDK_KEY_Otilde")]
    pub const Otilde: Self = Self(ffi::GDK_KEY_Otilde as u32);
    #[doc(alias = "GDK_KEY_Overlay1_Enable")]
    pub const Overlay1_Enable: Self = Self(ffi::GDK_KEY_Overlay1_Enable as u32);
    #[doc(alias = "GDK_KEY_Overlay2_Enable")]
    pub const Overlay2_Enable: Self = Self(ffi::GDK_KEY_Overlay2_Enable as u32);
    #[doc(alias = "GDK_KEY_P")]
    pub const P: Self = Self(ffi::GDK_KEY_P as u32);
    #[doc(alias = "GDK_KEY_Pabovedot")]
    pub const Pabovedot: Self = Self(ffi::GDK_KEY_Pabovedot as u32);
    #[doc(alias = "GDK_KEY_Page_Down")]
    pub const Page_Down: Self = Self(ffi::GDK_KEY_Page_Down as u32);
    #[doc(alias = "GDK_KEY_Page_Up")]
    pub const Page_Up: Self = Self(ffi::GDK_KEY_Page_Up as u32);
    #[doc(alias = "GDK_KEY_Paste")]
    pub const Paste: Self = Self(ffi::GDK_KEY_Paste as u32);
    #[doc(alias = "GDK_KEY_Pause")]
    pub const Pause: Self = Self(ffi::GDK_KEY_Pause as u32);
    #[doc(alias = "GDK_KEY_PauseRecord")]
    pub const PauseRecord: Self = Self(ffi::GDK_KEY_PauseRecord as u32);
    #[doc(alias = "GDK_KEY_PesetaSign")]
    pub const PesetaSign: Self = Self(ffi::GDK_KEY_PesetaSign as u32);
    #[doc(alias = "GDK_KEY_Phone")]
    pub const Phone: Self = Self(ffi::GDK_KEY_Phone as u32);
    #[doc(alias = "GDK_KEY_PickupPhone")]
    pub const PickupPhone: Self = Self(ffi::GDK_KEY_PickupPhone as u32);
    #[doc(alias = "GDK_KEY_Pictures")]
    pub const Pictures: Self = Self(ffi::GDK_KEY_Pictures as u32);
    #[doc(alias = "GDK_KEY_Pointer_Accelerate")]
    pub const Pointer_Accelerate: Self = Self(ffi::GDK_KEY_Pointer_Accelerate as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button1")]
    pub const Pointer_Button1: Self = Self(ffi::GDK_KEY_Pointer_Button1 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button2")]
    pub const Pointer_Button2: Self = Self(ffi::GDK_KEY_Pointer_Button2 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button3")]
    pub const Pointer_Button3: Self = Self(ffi::GDK_KEY_Pointer_Button3 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button4")]
    pub const Pointer_Button4: Self = Self(ffi::GDK_KEY_Pointer_Button4 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button5")]
    pub const Pointer_Button5: Self = Self(ffi::GDK_KEY_Pointer_Button5 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button_Dflt")]
    pub const Pointer_Button_Dflt: Self = Self(ffi::GDK_KEY_Pointer_Button_Dflt as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick1")]
    pub const Pointer_DblClick1: Self = Self(ffi::GDK_KEY_Pointer_DblClick1 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick2")]
    pub const Pointer_DblClick2: Self = Self(ffi::GDK_KEY_Pointer_DblClick2 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick3")]
    pub const Pointer_DblClick3: Self = Self(ffi::GDK_KEY_Pointer_DblClick3 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick4")]
    pub const Pointer_DblClick4: Self = Self(ffi::GDK_KEY_Pointer_DblClick4 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick5")]
    pub const Pointer_DblClick5: Self = Self(ffi::GDK_KEY_Pointer_DblClick5 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick_Dflt")]
    pub const Pointer_DblClick_Dflt: Self = Self(ffi::GDK_KEY_Pointer_DblClick_Dflt as u32);
    #[doc(alias = "GDK_KEY_Pointer_DfltBtnNext")]
    pub const Pointer_DfltBtnNext: Self = Self(ffi::GDK_KEY_Pointer_DfltBtnNext as u32);
    #[doc(alias = "GDK_KEY_Pointer_DfltBtnPrev")]
    pub const Pointer_DfltBtnPrev: Self = Self(ffi::GDK_KEY_Pointer_DfltBtnPrev as u32);
    #[doc(alias = "GDK_KEY_Pointer_Down")]
    pub const Pointer_Down: Self = Self(ffi::GDK_KEY_Pointer_Down as u32);
    #[doc(alias = "GDK_KEY_Pointer_DownLeft")]
    pub const Pointer_DownLeft: Self = Self(ffi::GDK_KEY_Pointer_DownLeft as u32);
    #[doc(alias = "GDK_KEY_Pointer_DownRight")]
    pub const Pointer_DownRight: Self = Self(ffi::GDK_KEY_Pointer_DownRight as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag1")]
    pub const Pointer_Drag1: Self = Self(ffi::GDK_KEY_Pointer_Drag1 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag2")]
    pub const Pointer_Drag2: Self = Self(ffi::GDK_KEY_Pointer_Drag2 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag3")]
    pub const Pointer_Drag3: Self = Self(ffi::GDK_KEY_Pointer_Drag3 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag4")]
    pub const Pointer_Drag4: Self = Self(ffi::GDK_KEY_Pointer_Drag4 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag5")]
    pub const Pointer_Drag5: Self = Self(ffi::GDK_KEY_Pointer_Drag5 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag_Dflt")]
    pub const Pointer_Drag_Dflt: Self = Self(ffi::GDK_KEY_Pointer_Drag_Dflt as u32);
    #[doc(alias = "GDK_KEY_Pointer_EnableKeys")]
    pub const Pointer_EnableKeys: Self = Self(ffi::GDK_KEY_Pointer_EnableKeys as u32);
    #[doc(alias = "GDK_KEY_Pointer_Left")]
    pub const Pointer_Left: Self = Self(ffi::GDK_KEY_Pointer_Left as u32);
    #[doc(alias = "GDK_KEY_Pointer_Right")]
    pub const Pointer_Right: Self = Self(ffi::GDK_KEY_Pointer_Right as u32);
    #[doc(alias = "GDK_KEY_Pointer_Up")]
    pub const Pointer_Up: Self = Self(ffi::GDK_KEY_Pointer_Up as u32);
    #[doc(alias = "GDK_KEY_Pointer_UpLeft")]
    pub const Pointer_UpLeft: Self = Self(ffi::GDK_KEY_Pointer_UpLeft as u32);
    #[doc(alias = "GDK_KEY_Pointer_UpRight")]
    pub const Pointer_UpRight: Self = Self(ffi::GDK_KEY_Pointer_UpRight as u32);
    #[doc(alias = "GDK_KEY_PowerDown")]
    pub const PowerDown: Self = Self(ffi::GDK_KEY_PowerDown as u32);
    #[doc(alias = "GDK_KEY_PowerOff")]
    pub const PowerOff: Self = Self(ffi::GDK_KEY_PowerOff as u32);
    #[doc(alias = "GDK_KEY_Presentation")]
    pub const Presentation: Self = Self(ffi::GDK_KEY_Presentation as u32);
    #[doc(alias = "GDK_KEY_Prev_VMode")]
    pub const Prev_VMode: Self = Self(ffi::GDK_KEY_Prev_VMode as u32);
    #[doc(alias = "GDK_KEY_Prev_Virtual_Screen")]
    pub const Prev_Virtual_Screen: Self = Self(ffi::GDK_KEY_Prev_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_PreviousCandidate")]
    pub const PreviousCandidate: Self = Self(ffi::GDK_KEY_PreviousCandidate as u32);
    #[doc(alias = "GDK_KEY_PreviousElement")]
    pub const PreviousElement: Self = Self(ffi::GDK_KEY_PreviousElement as u32);
    #[doc(alias = "GDK_KEY_Print")]
    pub const Print: Self = Self(ffi::GDK_KEY_Print as u32);
    #[doc(alias = "GDK_KEY_Prior")]
    pub const Prior: Self = Self(ffi::GDK_KEY_Prior as u32);
    #[doc(alias = "GDK_KEY_PrivacyScreenToggle")]
    pub const PrivacyScreenToggle: Self = Self(ffi::GDK_KEY_PrivacyScreenToggle as u32);
    #[doc(alias = "GDK_KEY_Q")]
    pub const Q: Self = Self(ffi::GDK_KEY_Q as u32);
    #[doc(alias = "GDK_KEY_R")]
    pub const R: Self = Self(ffi::GDK_KEY_R as u32);
    #[doc(alias = "GDK_KEY_R1")]
    pub const R1: Self = Self(ffi::GDK_KEY_R1 as u32);
    #[doc(alias = "GDK_KEY_R10")]
    pub const R10: Self = Self(ffi::GDK_KEY_R10 as u32);
    #[doc(alias = "GDK_KEY_R11")]
    pub const R11: Self = Self(ffi::GDK_KEY_R11 as u32);
    #[doc(alias = "GDK_KEY_R12")]
    pub const R12: Self = Self(ffi::GDK_KEY_R12 as u32);
    #[doc(alias = "GDK_KEY_R13")]
    pub const R13: Self = Self(ffi::GDK_KEY_R13 as u32);
    #[doc(alias = "GDK_KEY_R14")]
    pub const R14: Self = Self(ffi::GDK_KEY_R14 as u32);
    #[doc(alias = "GDK_KEY_R15")]
    pub const R15: Self = Self(ffi::GDK_KEY_R15 as u32);
    #[doc(alias = "GDK_KEY_R2")]
    pub const R2: Self = Self(ffi::GDK_KEY_R2 as u32);
    #[doc(alias = "GDK_KEY_R3")]
    pub const R3: Self = Self(ffi::GDK_KEY_R3 as u32);
    #[doc(alias = "GDK_KEY_R4")]
    pub const R4: Self = Self(ffi::GDK_KEY_R4 as u32);
    #[doc(alias = "GDK_KEY_R5")]
    pub const R5: Self = Self(ffi::GDK_KEY_R5 as u32);
    #[doc(alias = "GDK_KEY_R6")]
    pub const R6: Self = Self(ffi::GDK_KEY_R6 as u32);
    #[doc(alias = "GDK_KEY_R7")]
    pub const R7: Self = Self(ffi::GDK_KEY_R7 as u32);
    #[doc(alias = "GDK_KEY_R8")]
    pub const R8: Self = Self(ffi::GDK_KEY_R8 as u32);
    #[doc(alias = "GDK_KEY_R9")]
    pub const R9: Self = Self(ffi::GDK_KEY_R9 as u32);
    #[doc(alias = "GDK_KEY_RFKill")]
    pub const RFKill: Self = Self(ffi::GDK_KEY_RFKill as u32);
    #[doc(alias = "GDK_KEY_Racute")]
    pub const Racute: Self = Self(ffi::GDK_KEY_Racute as u32);
    #[doc(alias = "GDK_KEY_RadarOverlay")]
    pub const RadarOverlay: Self = Self(ffi::GDK_KEY_RadarOverlay as u32);
    #[doc(alias = "GDK_KEY_Rcaron")]
    pub const Rcaron: Self = Self(ffi::GDK_KEY_Rcaron as u32);
    #[doc(alias = "GDK_KEY_Rcedilla")]
    pub const Rcedilla: Self = Self(ffi::GDK_KEY_Rcedilla as u32);
    #[doc(alias = "GDK_KEY_Red")]
    pub const Red: Self = Self(ffi::GDK_KEY_Red as u32);
    #[doc(alias = "GDK_KEY_Redo")]
    pub const Redo: Self = Self(ffi::GDK_KEY_Redo as u32);
    #[doc(alias = "GDK_KEY_Refresh")]
    pub const Refresh: Self = Self(ffi::GDK_KEY_Refresh as u32);
    #[doc(alias = "GDK_KEY_RefreshRateToggle")]
    pub const RefreshRateToggle: Self = Self(ffi::GDK_KEY_RefreshRateToggle as u32);
    #[doc(alias = "GDK_KEY_Reload")]
    pub const Reload: Self = Self(ffi::GDK_KEY_Reload as u32);
    #[doc(alias = "GDK_KEY_RepeatKeys_Enable")]
    pub const RepeatKeys_Enable: Self = Self(ffi::GDK_KEY_RepeatKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_Reply")]
    pub const Reply: Self = Self(ffi::GDK_KEY_Reply as u32);
    #[doc(alias = "GDK_KEY_Return")]
    pub const Return: Self = Self(ffi::GDK_KEY_Return as u32);
    #[doc(alias = "GDK_KEY_Right")]
    pub const Right: Self = Self(ffi::GDK_KEY_Right as u32);
    #[doc(alias = "GDK_KEY_RightDown")]
    pub const RightDown: Self = Self(ffi::GDK_KEY_RightDown as u32);
    #[doc(alias = "GDK_KEY_RightUp")]
    pub const RightUp: Self = Self(ffi::GDK_KEY_RightUp as u32);
    #[doc(alias = "GDK_KEY_RockerDown")]
    pub const RockerDown: Self = Self(ffi::GDK_KEY_RockerDown as u32);
    #[doc(alias = "GDK_KEY_RockerEnter")]
    pub const RockerEnter: Self = Self(ffi::GDK_KEY_RockerEnter as u32);
    #[doc(alias = "GDK_KEY_RockerUp")]
    pub const RockerUp: Self = Self(ffi::GDK_KEY_RockerUp as u32);
    #[doc(alias = "GDK_KEY_Romaji")]
    pub const Romaji: Self = Self(ffi::GDK_KEY_Romaji as u32);
    #[doc(alias = "GDK_KEY_RootMenu")]
    pub const RootMenu: Self = Self(ffi::GDK_KEY_RootMenu as u32);
    #[doc(alias = "GDK_KEY_RotateWindows")]
    pub const RotateWindows: Self = Self(ffi::GDK_KEY_RotateWindows as u32);
    #[doc(alias = "GDK_KEY_RotationKB")]
    pub const RotationKB: Self = Self(ffi::GDK_KEY_RotationKB as u32);
    #[doc(alias = "GDK_KEY_RotationLockToggle")]
    pub const RotationLockToggle: Self = Self(ffi::GDK_KEY_RotationLockToggle as u32);
    #[doc(alias = "GDK_KEY_RotationPB")]
    pub const RotationPB: Self = Self(ffi::GDK_KEY_RotationPB as u32);
    #[doc(alias = "GDK_KEY_RupeeSign")]
    pub const RupeeSign: Self = Self(ffi::GDK_KEY_RupeeSign as u32);
    #[doc(alias = "GDK_KEY_S")]
    pub const S: Self = Self(ffi::GDK_KEY_S as u32);
    #[doc(alias = "GDK_KEY_SCHWA")]
    pub const SCHWA: Self = Self(ffi::GDK_KEY_SCHWA as u32);
    #[doc(alias = "GDK_KEY_Sabovedot")]
    pub const Sabovedot: Self = Self(ffi::GDK_KEY_Sabovedot as u32);
    #[doc(alias = "GDK_KEY_Sacute")]
    pub const Sacute: Self = Self(ffi::GDK_KEY_Sacute as u32);
    #[doc(alias = "GDK_KEY_Save")]
    pub const Save: Self = Self(ffi::GDK_KEY_Save as u32);
    #[doc(alias = "GDK_KEY_Scaron")]
    pub const Scaron: Self = Self(ffi::GDK_KEY_Scaron as u32);
    #[doc(alias = "GDK_KEY_Scedilla")]
    pub const Scedilla: Self = Self(ffi::GDK_KEY_Scedilla as u32);
    #[doc(alias = "GDK_KEY_Scircumflex")]
    pub const Scircumflex: Self = Self(ffi::GDK_KEY_Scircumflex as u32);
    #[doc(alias = "GDK_KEY_ScreenSaver")]
    pub const ScreenSaver: Self = Self(ffi::GDK_KEY_ScreenSaver as u32);
    #[doc(alias = "GDK_KEY_Screensaver")]
    pub const Screensaver: Self = Self(ffi::GDK_KEY_Screensaver as u32);
    #[doc(alias = "GDK_KEY_ScrollClick")]
    pub const ScrollClick: Self = Self(ffi::GDK_KEY_ScrollClick as u32);
    #[doc(alias = "GDK_KEY_ScrollDown")]
    pub const ScrollDown: Self = Self(ffi::GDK_KEY_ScrollDown as u32);
    #[doc(alias = "GDK_KEY_ScrollUp")]
    pub const ScrollUp: Self = Self(ffi::GDK_KEY_ScrollUp as u32);
    #[doc(alias = "GDK_KEY_Scroll_Lock")]
    pub const Scroll_Lock: Self = Self(ffi::GDK_KEY_Scroll_Lock as u32);
    #[doc(alias = "GDK_KEY_Search")]
    pub const Search: Self = Self(ffi::GDK_KEY_Search as u32);
    #[doc(alias = "GDK_KEY_Select")]
    pub const Select: Self = Self(ffi::GDK_KEY_Select as u32);
    #[doc(alias = "GDK_KEY_SelectButton")]
    pub const SelectButton: Self = Self(ffi::GDK_KEY_SelectButton as u32);
    #[doc(alias = "GDK_KEY_SelectiveScreenshot")]
    pub const SelectiveScreenshot: Self = Self(ffi::GDK_KEY_SelectiveScreenshot as u32);
    #[doc(alias = "GDK_KEY_Send")]
    pub const Send: Self = Self(ffi::GDK_KEY_Send as u32);
    #[doc(alias = "GDK_KEY_Serbian_DJE")]
    pub const Serbian_DJE: Self = Self(ffi::GDK_KEY_Serbian_DJE as u32);
    #[doc(alias = "GDK_KEY_Serbian_DZE")]
    pub const Serbian_DZE: Self = Self(ffi::GDK_KEY_Serbian_DZE as u32);
    #[doc(alias = "GDK_KEY_Serbian_JE")]
    pub const Serbian_JE: Self = Self(ffi::GDK_KEY_Serbian_JE as u32);
    #[doc(alias = "GDK_KEY_Serbian_LJE")]
    pub const Serbian_LJE: Self = Self(ffi::GDK_KEY_Serbian_LJE as u32);
    #[doc(alias = "GDK_KEY_Serbian_NJE")]
    pub const Serbian_NJE: Self = Self(ffi::GDK_KEY_Serbian_NJE as u32);
    #[doc(alias = "GDK_KEY_Serbian_TSHE")]
    pub const Serbian_TSHE: Self = Self(ffi::GDK_KEY_Serbian_TSHE as u32);
    #[doc(alias = "GDK_KEY_Serbian_dje")]
    pub const Serbian_dje: Self = Self(ffi::GDK_KEY_Serbian_dje as u32);
    #[doc(alias = "GDK_KEY_Serbian_dze")]
    pub const Serbian_dze: Self = Self(ffi::GDK_KEY_Serbian_dze as u32);
    #[doc(alias = "GDK_KEY_Serbian_je")]
    pub const Serbian_je: Self = Self(ffi::GDK_KEY_Serbian_je as u32);
    #[doc(alias = "GDK_KEY_Serbian_lje")]
    pub const Serbian_lje: Self = Self(ffi::GDK_KEY_Serbian_lje as u32);
    #[doc(alias = "GDK_KEY_Serbian_nje")]
    pub const Serbian_nje: Self = Self(ffi::GDK_KEY_Serbian_nje as u32);
    #[doc(alias = "GDK_KEY_Serbian_tshe")]
    pub const Serbian_tshe: Self = Self(ffi::GDK_KEY_Serbian_tshe as u32);
    #[doc(alias = "GDK_KEY_Shift_L")]
    pub const Shift_L: Self = Self(ffi::GDK_KEY_Shift_L as u32);
    #[doc(alias = "GDK_KEY_Shift_Lock")]
    pub const Shift_Lock: Self = Self(ffi::GDK_KEY_Shift_Lock as u32);
    #[doc(alias = "GDK_KEY_Shift_R")]
    pub const Shift_R: Self = Self(ffi::GDK_KEY_Shift_R as u32);
    #[doc(alias = "GDK_KEY_Shop")]
    pub const Shop: Self = Self(ffi::GDK_KEY_Shop as u32);
    #[doc(alias = "GDK_KEY_SidevuSonar")]
    pub const SidevuSonar: Self = Self(ffi::GDK_KEY_SidevuSonar as u32);
    #[doc(alias = "GDK_KEY_SingleCandidate")]
    pub const SingleCandidate: Self = Self(ffi::GDK_KEY_SingleCandidate as u32);
    #[doc(alias = "GDK_KEY_SingleRangeRadar")]
    pub const SingleRangeRadar: Self = Self(ffi::GDK_KEY_SingleRangeRadar as u32);
    #[doc(alias = "GDK_KEY_Sinh_a")]
    pub const Sinh_a: Self = Self(ffi::GDK_KEY_Sinh_a as u32);
    #[doc(alias = "GDK_KEY_Sinh_aa")]
    pub const Sinh_aa: Self = Self(ffi::GDK_KEY_Sinh_aa as u32);
    #[doc(alias = "GDK_KEY_Sinh_aa2")]
    pub const Sinh_aa2: Self = Self(ffi::GDK_KEY_Sinh_aa2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ae")]
    pub const Sinh_ae: Self = Self(ffi::GDK_KEY_Sinh_ae as u32);
    #[doc(alias = "GDK_KEY_Sinh_ae2")]
    pub const Sinh_ae2: Self = Self(ffi::GDK_KEY_Sinh_ae2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_aee")]
    pub const Sinh_aee: Self = Self(ffi::GDK_KEY_Sinh_aee as u32);
    #[doc(alias = "GDK_KEY_Sinh_aee2")]
    pub const Sinh_aee2: Self = Self(ffi::GDK_KEY_Sinh_aee2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ai")]
    pub const Sinh_ai: Self = Self(ffi::GDK_KEY_Sinh_ai as u32);
    #[doc(alias = "GDK_KEY_Sinh_ai2")]
    pub const Sinh_ai2: Self = Self(ffi::GDK_KEY_Sinh_ai2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_al")]
    pub const Sinh_al: Self = Self(ffi::GDK_KEY_Sinh_al as u32);
    #[doc(alias = "GDK_KEY_Sinh_au")]
    pub const Sinh_au: Self = Self(ffi::GDK_KEY_Sinh_au as u32);
    #[doc(alias = "GDK_KEY_Sinh_au2")]
    pub const Sinh_au2: Self = Self(ffi::GDK_KEY_Sinh_au2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ba")]
    pub const Sinh_ba: Self = Self(ffi::GDK_KEY_Sinh_ba as u32);
    #[doc(alias = "GDK_KEY_Sinh_bha")]
    pub const Sinh_bha: Self = Self(ffi::GDK_KEY_Sinh_bha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ca")]
    pub const Sinh_ca: Self = Self(ffi::GDK_KEY_Sinh_ca as u32);
    #[doc(alias = "GDK_KEY_Sinh_cha")]
    pub const Sinh_cha: Self = Self(ffi::GDK_KEY_Sinh_cha as u32);
    #[doc(alias = "GDK_KEY_Sinh_dda")]
    pub const Sinh_dda: Self = Self(ffi::GDK_KEY_Sinh_dda as u32);
    #[doc(alias = "GDK_KEY_Sinh_ddha")]
    pub const Sinh_ddha: Self = Self(ffi::GDK_KEY_Sinh_ddha as u32);
    #[doc(alias = "GDK_KEY_Sinh_dha")]
    pub const Sinh_dha: Self = Self(ffi::GDK_KEY_Sinh_dha as u32);
    #[doc(alias = "GDK_KEY_Sinh_dhha")]
    pub const Sinh_dhha: Self = Self(ffi::GDK_KEY_Sinh_dhha as u32);
    #[doc(alias = "GDK_KEY_Sinh_e")]
    pub const Sinh_e: Self = Self(ffi::GDK_KEY_Sinh_e as u32);
    #[doc(alias = "GDK_KEY_Sinh_e2")]
    pub const Sinh_e2: Self = Self(ffi::GDK_KEY_Sinh_e2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ee")]
    pub const Sinh_ee: Self = Self(ffi::GDK_KEY_Sinh_ee as u32);
    #[doc(alias = "GDK_KEY_Sinh_ee2")]
    pub const Sinh_ee2: Self = Self(ffi::GDK_KEY_Sinh_ee2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_fa")]
    pub const Sinh_fa: Self = Self(ffi::GDK_KEY_Sinh_fa as u32);
    #[doc(alias = "GDK_KEY_Sinh_ga")]
    pub const Sinh_ga: Self = Self(ffi::GDK_KEY_Sinh_ga as u32);
    #[doc(alias = "GDK_KEY_Sinh_gha")]
    pub const Sinh_gha: Self = Self(ffi::GDK_KEY_Sinh_gha as u32);
    #[doc(alias = "GDK_KEY_Sinh_h2")]
    pub const Sinh_h2: Self = Self(ffi::GDK_KEY_Sinh_h2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ha")]
    pub const Sinh_ha: Self = Self(ffi::GDK_KEY_Sinh_ha as u32);
    #[doc(alias = "GDK_KEY_Sinh_i")]
    pub const Sinh_i: Self = Self(ffi::GDK_KEY_Sinh_i as u32);
    #[doc(alias = "GDK_KEY_Sinh_i2")]
    pub const Sinh_i2: Self = Self(ffi::GDK_KEY_Sinh_i2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ii")]
    pub const Sinh_ii: Self = Self(ffi::GDK_KEY_Sinh_ii as u32);
    #[doc(alias = "GDK_KEY_Sinh_ii2")]
    pub const Sinh_ii2: Self = Self(ffi::GDK_KEY_Sinh_ii2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ja")]
    pub const Sinh_ja: Self = Self(ffi::GDK_KEY_Sinh_ja as u32);
    #[doc(alias = "GDK_KEY_Sinh_jha")]
    pub const Sinh_jha: Self = Self(ffi::GDK_KEY_Sinh_jha as u32);
    #[doc(alias = "GDK_KEY_Sinh_jnya")]
    pub const Sinh_jnya: Self = Self(ffi::GDK_KEY_Sinh_jnya as u32);
    #[doc(alias = "GDK_KEY_Sinh_ka")]
    pub const Sinh_ka: Self = Self(ffi::GDK_KEY_Sinh_ka as u32);
    #[doc(alias = "GDK_KEY_Sinh_kha")]
    pub const Sinh_kha: Self = Self(ffi::GDK_KEY_Sinh_kha as u32);
    #[doc(alias = "GDK_KEY_Sinh_kunddaliya")]
    pub const Sinh_kunddaliya: Self = Self(ffi::GDK_KEY_Sinh_kunddaliya as u32);
    #[doc(alias = "GDK_KEY_Sinh_la")]
    pub const Sinh_la: Self = Self(ffi::GDK_KEY_Sinh_la as u32);
    #[doc(alias = "GDK_KEY_Sinh_lla")]
    pub const Sinh_lla: Self = Self(ffi::GDK_KEY_Sinh_lla as u32);
    #[doc(alias = "GDK_KEY_Sinh_lu")]
    pub const Sinh_lu: Self = Self(ffi::GDK_KEY_Sinh_lu as u32);
    #[doc(alias = "GDK_KEY_Sinh_lu2")]
    pub const Sinh_lu2: Self = Self(ffi::GDK_KEY_Sinh_lu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_luu")]
    pub const Sinh_luu: Self = Self(ffi::GDK_KEY_Sinh_luu as u32);
    #[doc(alias = "GDK_KEY_Sinh_luu2")]
    pub const Sinh_luu2: Self = Self(ffi::GDK_KEY_Sinh_luu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ma")]
    pub const Sinh_ma: Self = Self(ffi::GDK_KEY_Sinh_ma as u32);
    #[doc(alias = "GDK_KEY_Sinh_mba")]
    pub const Sinh_mba: Self = Self(ffi::GDK_KEY_Sinh_mba as u32);
    #[doc(alias = "GDK_KEY_Sinh_na")]
    pub const Sinh_na: Self = Self(ffi::GDK_KEY_Sinh_na as u32);
    #[doc(alias = "GDK_KEY_Sinh_ndda")]
    pub const Sinh_ndda: Self = Self(ffi::GDK_KEY_Sinh_ndda as u32);
    #[doc(alias = "GDK_KEY_Sinh_ndha")]
    pub const Sinh_ndha: Self = Self(ffi::GDK_KEY_Sinh_ndha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ng")]
    pub const Sinh_ng: Self = Self(ffi::GDK_KEY_Sinh_ng as u32);
    #[doc(alias = "GDK_KEY_Sinh_ng2")]
    pub const Sinh_ng2: Self = Self(ffi::GDK_KEY_Sinh_ng2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_nga")]
    pub const Sinh_nga: Self = Self(ffi::GDK_KEY_Sinh_nga as u32);
    #[doc(alias = "GDK_KEY_Sinh_nja")]
    pub const Sinh_nja: Self = Self(ffi::GDK_KEY_Sinh_nja as u32);
    #[doc(alias = "GDK_KEY_Sinh_nna")]
    pub const Sinh_nna: Self = Self(ffi::GDK_KEY_Sinh_nna as u32);
    #[doc(alias = "GDK_KEY_Sinh_nya")]
    pub const Sinh_nya: Self = Self(ffi::GDK_KEY_Sinh_nya as u32);
    #[doc(alias = "GDK_KEY_Sinh_o")]
    pub const Sinh_o: Self = Self(ffi::GDK_KEY_Sinh_o as u32);
    #[doc(alias = "GDK_KEY_Sinh_o2")]
    pub const Sinh_o2: Self = Self(ffi::GDK_KEY_Sinh_o2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_oo")]
    pub const Sinh_oo: Self = Self(ffi::GDK_KEY_Sinh_oo as u32);
    #[doc(alias = "GDK_KEY_Sinh_oo2")]
    pub const Sinh_oo2: Self = Self(ffi::GDK_KEY_Sinh_oo2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_pa")]
    pub const Sinh_pa: Self = Self(ffi::GDK_KEY_Sinh_pa as u32);
    #[doc(alias = "GDK_KEY_Sinh_pha")]
    pub const Sinh_pha: Self = Self(ffi::GDK_KEY_Sinh_pha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ra")]
    pub const Sinh_ra: Self = Self(ffi::GDK_KEY_Sinh_ra as u32);
    #[doc(alias = "GDK_KEY_Sinh_ri")]
    pub const Sinh_ri: Self = Self(ffi::GDK_KEY_Sinh_ri as u32);
    #[doc(alias = "GDK_KEY_Sinh_rii")]
    pub const Sinh_rii: Self = Self(ffi::GDK_KEY_Sinh_rii as u32);
    #[doc(alias = "GDK_KEY_Sinh_ru2")]
    pub const Sinh_ru2: Self = Self(ffi::GDK_KEY_Sinh_ru2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ruu2")]
    pub const Sinh_ruu2: Self = Self(ffi::GDK_KEY_Sinh_ruu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_sa")]
    pub const Sinh_sa: Self = Self(ffi::GDK_KEY_Sinh_sa as u32);
    #[doc(alias = "GDK_KEY_Sinh_sha")]
    pub const Sinh_sha: Self = Self(ffi::GDK_KEY_Sinh_sha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ssha")]
    pub const Sinh_ssha: Self = Self(ffi::GDK_KEY_Sinh_ssha as u32);
    #[doc(alias = "GDK_KEY_Sinh_tha")]
    pub const Sinh_tha: Self = Self(ffi::GDK_KEY_Sinh_tha as u32);
    #[doc(alias = "GDK_KEY_Sinh_thha")]
    pub const Sinh_thha: Self = Self(ffi::GDK_KEY_Sinh_thha as u32);
    #[doc(alias = "GDK_KEY_Sinh_tta")]
    pub const Sinh_tta: Self = Self(ffi::GDK_KEY_Sinh_tta as u32);
    #[doc(alias = "GDK_KEY_Sinh_ttha")]
    pub const Sinh_ttha: Self = Self(ffi::GDK_KEY_Sinh_ttha as u32);
    #[doc(alias = "GDK_KEY_Sinh_u")]
    pub const Sinh_u: Self = Self(ffi::GDK_KEY_Sinh_u as u32);
    #[doc(alias = "GDK_KEY_Sinh_u2")]
    pub const Sinh_u2: Self = Self(ffi::GDK_KEY_Sinh_u2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_uu")]
    pub const Sinh_uu: Self = Self(ffi::GDK_KEY_Sinh_uu as u32);
    #[doc(alias = "GDK_KEY_Sinh_uu2")]
    pub const Sinh_uu2: Self = Self(ffi::GDK_KEY_Sinh_uu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_va")]
    pub const Sinh_va: Self = Self(ffi::GDK_KEY_Sinh_va as u32);
    #[doc(alias = "GDK_KEY_Sinh_ya")]
    pub const Sinh_ya: Self = Self(ffi::GDK_KEY_Sinh_ya as u32);
    #[doc(alias = "GDK_KEY_Sleep")]
    pub const Sleep: Self = Self(ffi::GDK_KEY_Sleep as u32);
    #[doc(alias = "GDK_KEY_SlowKeys_Enable")]
    pub const SlowKeys_Enable: Self = Self(ffi::GDK_KEY_SlowKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_SlowReverse")]
    pub const SlowReverse: Self = Self(ffi::GDK_KEY_SlowReverse as u32);
    #[doc(alias = "GDK_KEY_Sos")]
    pub const Sos: Self = Self(ffi::GDK_KEY_Sos as u32);
    #[doc(alias = "GDK_KEY_Spell")]
    pub const Spell: Self = Self(ffi::GDK_KEY_Spell as u32);
    #[doc(alias = "GDK_KEY_SpellCheck")]
    pub const SpellCheck: Self = Self(ffi::GDK_KEY_SpellCheck as u32);
    #[doc(alias = "GDK_KEY_SplitScreen")]
    pub const SplitScreen: Self = Self(ffi::GDK_KEY_SplitScreen as u32);
    #[doc(alias = "GDK_KEY_Standby")]
    pub const Standby: Self = Self(ffi::GDK_KEY_Standby as u32);
    #[doc(alias = "GDK_KEY_Start")]
    pub const Start: Self = Self(ffi::GDK_KEY_Start as u32);
    #[doc(alias = "GDK_KEY_StickyKeys_Enable")]
    pub const StickyKeys_Enable: Self = Self(ffi::GDK_KEY_StickyKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_Stop")]
    pub const Stop: Self = Self(ffi::GDK_KEY_Stop as u32);
    #[doc(alias = "GDK_KEY_StopRecord")]
    pub const StopRecord: Self = Self(ffi::GDK_KEY_StopRecord as u32);
    #[doc(alias = "GDK_KEY_Subtitle")]
    pub const Subtitle: Self = Self(ffi::GDK_KEY_Subtitle as u32);
    #[doc(alias = "GDK_KEY_Super_L")]
    pub const Super_L: Self = Self(ffi::GDK_KEY_Super_L as u32);
    #[doc(alias = "GDK_KEY_Super_R")]
    pub const Super_R: Self = Self(ffi::GDK_KEY_Super_R as u32);
    #[doc(alias = "GDK_KEY_Support")]
    pub const Support: Self = Self(ffi::GDK_KEY_Support as u32);
    #[doc(alias = "GDK_KEY_Suspend")]
    pub const Suspend: Self = Self(ffi::GDK_KEY_Suspend as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_1")]
    pub const Switch_VT_1: Self = Self(ffi::GDK_KEY_Switch_VT_1 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_10")]
    pub const Switch_VT_10: Self = Self(ffi::GDK_KEY_Switch_VT_10 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_11")]
    pub const Switch_VT_11: Self = Self(ffi::GDK_KEY_Switch_VT_11 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_12")]
    pub const Switch_VT_12: Self = Self(ffi::GDK_KEY_Switch_VT_12 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_2")]
    pub const Switch_VT_2: Self = Self(ffi::GDK_KEY_Switch_VT_2 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_3")]
    pub const Switch_VT_3: Self = Self(ffi::GDK_KEY_Switch_VT_3 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_4")]
    pub const Switch_VT_4: Self = Self(ffi::GDK_KEY_Switch_VT_4 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_5")]
    pub const Switch_VT_5: Self = Self(ffi::GDK_KEY_Switch_VT_5 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_6")]
    pub const Switch_VT_6: Self = Self(ffi::GDK_KEY_Switch_VT_6 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_7")]
    pub const Switch_VT_7: Self = Self(ffi::GDK_KEY_Switch_VT_7 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_8")]
    pub const Switch_VT_8: Self = Self(ffi::GDK_KEY_Switch_VT_8 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_9")]
    pub const Switch_VT_9: Self = Self(ffi::GDK_KEY_Switch_VT_9 as u32);
    #[doc(alias = "GDK_KEY_Sys_Req")]
    pub const Sys_Req: Self = Self(ffi::GDK_KEY_Sys_Req as u32);
    #[doc(alias = "GDK_KEY_T")]
    pub const T: Self = Self(ffi::GDK_KEY_T as u32);
    #[doc(alias = "GDK_KEY_THORN")]
    pub const THORN: Self = Self(ffi::GDK_KEY_THORN as u32);
    #[doc(alias = "GDK_KEY_Tab")]
    pub const Tab: Self = Self(ffi::GDK_KEY_Tab as u32);
    #[doc(alias = "GDK_KEY_Tabovedot")]
    pub const Tabovedot: Self = Self(ffi::GDK_KEY_Tabovedot as u32);
    #[doc(alias = "GDK_KEY_TaskPane")]
    pub const TaskPane: Self = Self(ffi::GDK_KEY_TaskPane as u32);
    #[doc(alias = "GDK_KEY_Taskmanager")]
    pub const Taskmanager: Self = Self(ffi::GDK_KEY_Taskmanager as u32);
    #[doc(alias = "GDK_KEY_Tcaron")]
    pub const Tcaron: Self = Self(ffi::GDK_KEY_Tcaron as u32);
    #[doc(alias = "GDK_KEY_Tcedilla")]
    pub const Tcedilla: Self = Self(ffi::GDK_KEY_Tcedilla as u32);
    #[doc(alias = "GDK_KEY_Terminal")]
    pub const Terminal: Self = Self(ffi::GDK_KEY_Terminal as u32);
    #[doc(alias = "GDK_KEY_Terminate_Server")]
    pub const Terminate_Server: Self = Self(ffi::GDK_KEY_Terminate_Server as u32);
    #[doc(alias = "GDK_KEY_Thai_baht")]
    pub const Thai_baht: Self = Self(ffi::GDK_KEY_Thai_baht as u32);
    #[doc(alias = "GDK_KEY_Thai_bobaimai")]
    pub const Thai_bobaimai: Self = Self(ffi::GDK_KEY_Thai_bobaimai as u32);
    #[doc(alias = "GDK_KEY_Thai_chochan")]
    pub const Thai_chochan: Self = Self(ffi::GDK_KEY_Thai_chochan as u32);
    #[doc(alias = "GDK_KEY_Thai_chochang")]
    pub const Thai_chochang: Self = Self(ffi::GDK_KEY_Thai_chochang as u32);
    #[doc(alias = "GDK_KEY_Thai_choching")]
    pub const Thai_choching: Self = Self(ffi::GDK_KEY_Thai_choching as u32);
    #[doc(alias = "GDK_KEY_Thai_chochoe")]
    pub const Thai_chochoe: Self = Self(ffi::GDK_KEY_Thai_chochoe as u32);
    #[doc(alias = "GDK_KEY_Thai_dochada")]
    pub const Thai_dochada: Self = Self(ffi::GDK_KEY_Thai_dochada as u32);
    #[doc(alias = "GDK_KEY_Thai_dodek")]
    pub const Thai_dodek: Self = Self(ffi::GDK_KEY_Thai_dodek as u32);
    #[doc(alias = "GDK_KEY_Thai_fofa")]
    pub const Thai_fofa: Self = Self(ffi::GDK_KEY_Thai_fofa as u32);
    #[doc(alias = "GDK_KEY_Thai_fofan")]
    pub const Thai_fofan: Self = Self(ffi::GDK_KEY_Thai_fofan as u32);
    #[doc(alias = "GDK_KEY_Thai_hohip")]
    pub const Thai_hohip: Self = Self(ffi::GDK_KEY_Thai_hohip as u32);
    #[doc(alias = "GDK_KEY_Thai_honokhuk")]
    pub const Thai_honokhuk: Self = Self(ffi::GDK_KEY_Thai_honokhuk as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhai")]
    pub const Thai_khokhai: Self = Self(ffi::GDK_KEY_Thai_khokhai as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhon")]
    pub const Thai_khokhon: Self = Self(ffi::GDK_KEY_Thai_khokhon as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhuat")]
    pub const Thai_khokhuat: Self = Self(ffi::GDK_KEY_Thai_khokhuat as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhwai")]
    pub const Thai_khokhwai: Self = Self(ffi::GDK_KEY_Thai_khokhwai as u32);
    #[doc(alias = "GDK_KEY_Thai_khorakhang")]
    pub const Thai_khorakhang: Self = Self(ffi::GDK_KEY_Thai_khorakhang as u32);
    #[doc(alias = "GDK_KEY_Thai_kokai")]
    pub const Thai_kokai: Self = Self(ffi::GDK_KEY_Thai_kokai as u32);
    #[doc(alias = "GDK_KEY_Thai_lakkhangyao")]
    pub const Thai_lakkhangyao: Self = Self(ffi::GDK_KEY_Thai_lakkhangyao as u32);
    #[doc(alias = "GDK_KEY_Thai_lekchet")]
    pub const Thai_lekchet: Self = Self(ffi::GDK_KEY_Thai_lekchet as u32);
    #[doc(alias = "GDK_KEY_Thai_lekha")]
    pub const Thai_lekha: Self = Self(ffi::GDK_KEY_Thai_lekha as u32);
    #[doc(alias = "GDK_KEY_Thai_lekhok")]
    pub const Thai_lekhok: Self = Self(ffi::GDK_KEY_Thai_lekhok as u32);
    #[doc(alias = "GDK_KEY_Thai_lekkao")]
    pub const Thai_lekkao: Self = Self(ffi::GDK_KEY_Thai_lekkao as u32);
    #[doc(alias = "GDK_KEY_Thai_leknung")]
    pub const Thai_leknung: Self = Self(ffi::GDK_KEY_Thai_leknung as u32);
    #[doc(alias = "GDK_KEY_Thai_lekpaet")]
    pub const Thai_lekpaet: Self = Self(ffi::GDK_KEY_Thai_lekpaet as u32);
    #[doc(alias = "GDK_KEY_Thai_leksam")]
    pub const Thai_leksam: Self = Self(ffi::GDK_KEY_Thai_leksam as u32);
    #[doc(alias = "GDK_KEY_Thai_leksi")]
    pub const Thai_leksi: Self = Self(ffi::GDK_KEY_Thai_leksi as u32);
    #[doc(alias = "GDK_KEY_Thai_leksong")]
    pub const Thai_leksong: Self = Self(ffi::GDK_KEY_Thai_leksong as u32);
    #[doc(alias = "GDK_KEY_Thai_leksun")]
    pub const Thai_leksun: Self = Self(ffi::GDK_KEY_Thai_leksun as u32);
    #[doc(alias = "GDK_KEY_Thai_lochula")]
    pub const Thai_lochula: Self = Self(ffi::GDK_KEY_Thai_lochula as u32);
    #[doc(alias = "GDK_KEY_Thai_loling")]
    pub const Thai_loling: Self = Self(ffi::GDK_KEY_Thai_loling as u32);
    #[doc(alias = "GDK_KEY_Thai_lu")]
    pub const Thai_lu: Self = Self(ffi::GDK_KEY_Thai_lu as u32);
    #[doc(alias = "GDK_KEY_Thai_maichattawa")]
    pub const Thai_maichattawa: Self = Self(ffi::GDK_KEY_Thai_maichattawa as u32);
    #[doc(alias = "GDK_KEY_Thai_maiek")]
    pub const Thai_maiek: Self = Self(ffi::GDK_KEY_Thai_maiek as u32);
    #[doc(alias = "GDK_KEY_Thai_maihanakat")]
    pub const Thai_maihanakat: Self = Self(ffi::GDK_KEY_Thai_maihanakat as u32);
    #[doc(alias = "GDK_KEY_Thai_maihanakat_maitho")]
    pub const Thai_maihanakat_maitho: Self = Self(ffi::GDK_KEY_Thai_maihanakat_maitho as u32);
    #[doc(alias = "GDK_KEY_Thai_maitaikhu")]
    pub const Thai_maitaikhu: Self = Self(ffi::GDK_KEY_Thai_maitaikhu as u32);
    #[doc(alias = "GDK_KEY_Thai_maitho")]
    pub const Thai_maitho: Self = Self(ffi::GDK_KEY_Thai_maitho as u32);
    #[doc(alias = "GDK_KEY_Thai_maitri")]
    pub const Thai_maitri: Self = Self(ffi::GDK_KEY_Thai_maitri as u32);
    #[doc(alias = "GDK_KEY_Thai_maiyamok")]
    pub const Thai_maiyamok: Self = Self(ffi::GDK_KEY_Thai_maiyamok as u32);
    #[doc(alias = "GDK_KEY_Thai_moma")]
    pub const Thai_moma: Self = Self(ffi::GDK_KEY_Thai_moma as u32);
    #[doc(alias = "GDK_KEY_Thai_ngongu")]
    pub const Thai_ngongu: Self = Self(ffi::GDK_KEY_Thai_ngongu as u32);
    #[doc(alias = "GDK_KEY_Thai_nikhahit")]
    pub const Thai_nikhahit: Self = Self(ffi::GDK_KEY_Thai_nikhahit as u32);
    #[doc(alias = "GDK_KEY_Thai_nonen")]
    pub const Thai_nonen: Self = Self(ffi::GDK_KEY_Thai_nonen as u32);
    #[doc(alias = "GDK_KEY_Thai_nonu")]
    pub const Thai_nonu: Self = Self(ffi::GDK_KEY_Thai_nonu as u32);
    #[doc(alias = "GDK_KEY_Thai_oang")]
    pub const Thai_oang: Self = Self(ffi::GDK_KEY_Thai_oang as u32);
    #[doc(alias = "GDK_KEY_Thai_paiyannoi")]
    pub const Thai_paiyannoi: Self = Self(ffi::GDK_KEY_Thai_paiyannoi as u32);
    #[doc(alias = "GDK_KEY_Thai_phinthu")]
    pub const Thai_phinthu: Self = Self(ffi::GDK_KEY_Thai_phinthu as u32);
    #[doc(alias = "GDK_KEY_Thai_phophan")]
    pub const Thai_phophan: Self = Self(ffi::GDK_KEY_Thai_phophan as u32);
    #[doc(alias = "GDK_KEY_Thai_phophung")]
    pub const Thai_phophung: Self = Self(ffi::GDK_KEY_Thai_phophung as u32);
    #[doc(alias = "GDK_KEY_Thai_phosamphao")]
    pub const Thai_phosamphao: Self = Self(ffi::GDK_KEY_Thai_phosamphao as u32);
    #[doc(alias = "GDK_KEY_Thai_popla")]
    pub const Thai_popla: Self = Self(ffi::GDK_KEY_Thai_popla as u32);
    #[doc(alias = "GDK_KEY_Thai_rorua")]
    pub const Thai_rorua: Self = Self(ffi::GDK_KEY_Thai_rorua as u32);
    #[doc(alias = "GDK_KEY_Thai_ru")]
    pub const Thai_ru: Self = Self(ffi::GDK_KEY_Thai_ru as u32);
    #[doc(alias = "GDK_KEY_Thai_saraa")]
    pub const Thai_saraa: Self = Self(ffi::GDK_KEY_Thai_saraa as u32);
    #[doc(alias = "GDK_KEY_Thai_saraaa")]
    pub const Thai_saraaa: Self = Self(ffi::GDK_KEY_Thai_saraaa as u32);
    #[doc(alias = "GDK_KEY_Thai_saraae")]
    pub const Thai_saraae: Self = Self(ffi::GDK_KEY_Thai_saraae as u32);
    #[doc(alias = "GDK_KEY_Thai_saraaimaimalai")]
    pub const Thai_saraaimaimalai: Self = Self(ffi::GDK_KEY_Thai_saraaimaimalai as u32);
    #[doc(alias = "GDK_KEY_Thai_saraaimaimuan")]
    pub const Thai_saraaimaimuan: Self = Self(ffi::GDK_KEY_Thai_saraaimaimuan as u32);
    #[doc(alias = "GDK_KEY_Thai_saraam")]
    pub const Thai_saraam: Self = Self(ffi::GDK_KEY_Thai_saraam as u32);
    #[doc(alias = "GDK_KEY_Thai_sarae")]
    pub const Thai_sarae: Self = Self(ffi::GDK_KEY_Thai_sarae as u32);
    #[doc(alias = "GDK_KEY_Thai_sarai")]
    pub const Thai_sarai: Self = Self(ffi::GDK_KEY_Thai_sarai as u32);
    #[doc(alias = "GDK_KEY_Thai_saraii")]
    pub const Thai_saraii: Self = Self(ffi::GDK_KEY_Thai_saraii as u32);
    #[doc(alias = "GDK_KEY_Thai_sarao")]
    pub const Thai_sarao: Self = Self(ffi::GDK_KEY_Thai_sarao as u32);
    #[doc(alias = "GDK_KEY_Thai_sarau")]
    pub const Thai_sarau: Self = Self(ffi::GDK_KEY_Thai_sarau as u32);
    #[doc(alias = "GDK_KEY_Thai_saraue")]
    pub const Thai_saraue: Self = Self(ffi::GDK_KEY_Thai_saraue as u32);
    #[doc(alias = "GDK_KEY_Thai_sarauee")]
    pub const Thai_sarauee: Self = Self(ffi::GDK_KEY_Thai_sarauee as u32);
    #[doc(alias = "GDK_KEY_Thai_sarauu")]
    pub const Thai_sarauu: Self = Self(ffi::GDK_KEY_Thai_sarauu as u32);
    #[doc(alias = "GDK_KEY_Thai_sorusi")]
    pub const Thai_sorusi: Self = Self(ffi::GDK_KEY_Thai_sorusi as u32);
    #[doc(alias = "GDK_KEY_Thai_sosala")]
    pub const Thai_sosala: Self = Self(ffi::GDK_KEY_Thai_sosala as u32);
    #[doc(alias = "GDK_KEY_Thai_soso")]
    pub const Thai_soso: Self = Self(ffi::GDK_KEY_Thai_soso as u32);
    #[doc(alias = "GDK_KEY_Thai_sosua")]
    pub const Thai_sosua: Self = Self(ffi::GDK_KEY_Thai_sosua as u32);
    #[doc(alias = "GDK_KEY_Thai_thanthakhat")]
    pub const Thai_thanthakhat: Self = Self(ffi::GDK_KEY_Thai_thanthakhat as u32);
    #[doc(alias = "GDK_KEY_Thai_thonangmontho")]
    pub const Thai_thonangmontho: Self = Self(ffi::GDK_KEY_Thai_thonangmontho as u32);
    #[doc(alias = "GDK_KEY_Thai_thophuthao")]
    pub const Thai_thophuthao: Self = Self(ffi::GDK_KEY_Thai_thophuthao as u32);
    #[doc(alias = "GDK_KEY_Thai_thothahan")]
    pub const Thai_thothahan: Self = Self(ffi::GDK_KEY_Thai_thothahan as u32);
    #[doc(alias = "GDK_KEY_Thai_thothan")]
    pub const Thai_thothan: Self = Self(ffi::GDK_KEY_Thai_thothan as u32);
    #[doc(alias = "GDK_KEY_Thai_thothong")]
    pub const Thai_thothong: Self = Self(ffi::GDK_KEY_Thai_thothong as u32);
    #[doc(alias = "GDK_KEY_Thai_thothung")]
    pub const Thai_thothung: Self = Self(ffi::GDK_KEY_Thai_thothung as u32);
    #[doc(alias = "GDK_KEY_Thai_topatak")]
    pub const Thai_topatak: Self = Self(ffi::GDK_KEY_Thai_topatak as u32);
    #[doc(alias = "GDK_KEY_Thai_totao")]
    pub const Thai_totao: Self = Self(ffi::GDK_KEY_Thai_totao as u32);
    #[doc(alias = "GDK_KEY_Thai_wowaen")]
    pub const Thai_wowaen: Self = Self(ffi::GDK_KEY_Thai_wowaen as u32);
    #[doc(alias = "GDK_KEY_Thai_yoyak")]
    pub const Thai_yoyak: Self = Self(ffi::GDK_KEY_Thai_yoyak as u32);
    #[doc(alias = "GDK_KEY_Thai_yoying")]
    pub const Thai_yoying: Self = Self(ffi::GDK_KEY_Thai_yoying as u32);
    #[doc(alias = "GDK_KEY_Thorn")]
    pub const Thorn: Self = Self(ffi::GDK_KEY_Thorn as u32);
    #[doc(alias = "GDK_KEY_Time")]
    pub const Time: Self = Self(ffi::GDK_KEY_Time as u32);
    #[doc(alias = "GDK_KEY_ToDoList")]
    pub const ToDoList: Self = Self(ffi::GDK_KEY_ToDoList as u32);
    #[doc(alias = "GDK_KEY_Tools")]
    pub const Tools: Self = Self(ffi::GDK_KEY_Tools as u32);
    #[doc(alias = "GDK_KEY_TopMenu")]
    pub const TopMenu: Self = Self(ffi::GDK_KEY_TopMenu as u32);
    #[doc(alias = "GDK_KEY_TouchpadOff")]
    pub const TouchpadOff: Self = Self(ffi::GDK_KEY_TouchpadOff as u32);
    #[doc(alias = "GDK_KEY_TouchpadOn")]
    pub const TouchpadOn: Self = Self(ffi::GDK_KEY_TouchpadOn as u32);
    #[doc(alias = "GDK_KEY_TouchpadToggle")]
    pub const TouchpadToggle: Self = Self(ffi::GDK_KEY_TouchpadToggle as u32);
    #[doc(alias = "GDK_KEY_Touroku")]
    pub const Touroku: Self = Self(ffi::GDK_KEY_Touroku as u32);
    #[doc(alias = "GDK_KEY_TraditionalSonar")]
    pub const TraditionalSonar: Self = Self(ffi::GDK_KEY_TraditionalSonar as u32);
    #[doc(alias = "GDK_KEY_Travel")]
    pub const Travel: Self = Self(ffi::GDK_KEY_Travel as u32);
    #[doc(alias = "GDK_KEY_Tslash")]
    pub const Tslash: Self = Self(ffi::GDK_KEY_Tslash as u32);
    #[doc(alias = "GDK_KEY_U")]
    pub const U: Self = Self(ffi::GDK_KEY_U as u32);
    #[doc(alias = "GDK_KEY_UWB")]
    pub const UWB: Self = Self(ffi::GDK_KEY_UWB as u32);
    #[doc(alias = "GDK_KEY_Uacute")]
    pub const Uacute: Self = Self(ffi::GDK_KEY_Uacute as u32);
    #[doc(alias = "GDK_KEY_Ubelowdot")]
    pub const Ubelowdot: Self = Self(ffi::GDK_KEY_Ubelowdot as u32);
    #[doc(alias = "GDK_KEY_Ubreve")]
    pub const Ubreve: Self = Self(ffi::GDK_KEY_Ubreve as u32);
    #[doc(alias = "GDK_KEY_Ucircumflex")]
    pub const Ucircumflex: Self = Self(ffi::GDK_KEY_Ucircumflex as u32);
    #[doc(alias = "GDK_KEY_Udiaeresis")]
    pub const Udiaeresis: Self = Self(ffi::GDK_KEY_Udiaeresis as u32);
    #[doc(alias = "GDK_KEY_Udoubleacute")]
    pub const Udoubleacute: Self = Self(ffi::GDK_KEY_Udoubleacute as u32);
    #[doc(alias = "GDK_KEY_Ugrave")]
    pub const Ugrave: Self = Self(ffi::GDK_KEY_Ugrave as u32);
    #[doc(alias = "GDK_KEY_Uhook")]
    pub const Uhook: Self = Self(ffi::GDK_KEY_Uhook as u32);
    #[doc(alias = "GDK_KEY_Uhorn")]
    pub const Uhorn: Self = Self(ffi::GDK_KEY_Uhorn as u32);
    #[doc(alias = "GDK_KEY_Uhornacute")]
    pub const Uhornacute: Self = Self(ffi::GDK_KEY_Uhornacute as u32);
    #[doc(alias = "GDK_KEY_Uhornbelowdot")]
    pub const Uhornbelowdot: Self = Self(ffi::GDK_KEY_Uhornbelowdot as u32);
    #[doc(alias = "GDK_KEY_Uhorngrave")]
    pub const Uhorngrave: Self = Self(ffi::GDK_KEY_Uhorngrave as u32);
    #[doc(alias = "GDK_KEY_Uhornhook")]
    pub const Uhornhook: Self = Self(ffi::GDK_KEY_Uhornhook as u32);
    #[doc(alias = "GDK_KEY_Uhorntilde")]
    pub const Uhorntilde: Self = Self(ffi::GDK_KEY_Uhorntilde as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_GHE_WITH_UPTURN")]
    pub const Ukrainian_GHE_WITH_UPTURN: Self = Self(ffi::GDK_KEY_Ukrainian_GHE_WITH_UPTURN as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_I")]
    pub const Ukrainian_I: Self = Self(ffi::GDK_KEY_Ukrainian_I as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_IE")]
    pub const Ukrainian_IE: Self = Self(ffi::GDK_KEY_Ukrainian_IE as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_YI")]
    pub const Ukrainian_YI: Self = Self(ffi::GDK_KEY_Ukrainian_YI as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_ghe_with_upturn")]
    pub const Ukrainian_ghe_with_upturn: Self = Self(ffi::GDK_KEY_Ukrainian_ghe_with_upturn as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_i")]
    pub const Ukrainian_i: Self = Self(ffi::GDK_KEY_Ukrainian_i as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_ie")]
    pub const Ukrainian_ie: Self = Self(ffi::GDK_KEY_Ukrainian_ie as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_yi")]
    pub const Ukrainian_yi: Self = Self(ffi::GDK_KEY_Ukrainian_yi as u32);
    #[doc(alias = "GDK_KEY_Ukranian_I")]
    pub const Ukranian_I: Self = Self(ffi::GDK_KEY_Ukranian_I as u32);
    #[doc(alias = "GDK_KEY_Ukranian_JE")]
    pub const Ukranian_JE: Self = Self(ffi::GDK_KEY_Ukranian_JE as u32);
    #[doc(alias = "GDK_KEY_Ukranian_YI")]
    pub const Ukranian_YI: Self = Self(ffi::GDK_KEY_Ukranian_YI as u32);
    #[doc(alias = "GDK_KEY_Ukranian_i")]
    pub const Ukranian_i: Self = Self(ffi::GDK_KEY_Ukranian_i as u32);
    #[doc(alias = "GDK_KEY_Ukranian_je")]
    pub const Ukranian_je: Self = Self(ffi::GDK_KEY_Ukranian_je as u32);
    #[doc(alias = "GDK_KEY_Ukranian_yi")]
    pub const Ukranian_yi: Self = Self(ffi::GDK_KEY_Ukranian_yi as u32);
    #[doc(alias = "GDK_KEY_Umacron")]
    pub const Umacron: Self = Self(ffi::GDK_KEY_Umacron as u32);
    #[doc(alias = "GDK_KEY_Undo")]
    pub const Undo: Self = Self(ffi::GDK_KEY_Undo as u32);
    #[doc(alias = "GDK_KEY_Ungrab")]
    pub const Ungrab: Self = Self(ffi::GDK_KEY_Ungrab as u32);
    #[doc(alias = "GDK_KEY_Unmute")]
    pub const Unmute: Self = Self(ffi::GDK_KEY_Unmute as u32);
    #[doc(alias = "GDK_KEY_Uogonek")]
    pub const Uogonek: Self = Self(ffi::GDK_KEY_Uogonek as u32);
    #[doc(alias = "GDK_KEY_Up")]
    pub const Up: Self = Self(ffi::GDK_KEY_Up as u32);
    #[doc(alias = "GDK_KEY_Uring")]
    pub const Uring: Self = Self(ffi::GDK_KEY_Uring as u32);
    #[doc(alias = "GDK_KEY_User1KB")]
    pub const User1KB: Self = Self(ffi::GDK_KEY_User1KB as u32);
    #[doc(alias = "GDK_KEY_User2KB")]
    pub const User2KB: Self = Self(ffi::GDK_KEY_User2KB as u32);
    #[doc(alias = "GDK_KEY_UserPB")]
    pub const UserPB: Self = Self(ffi::GDK_KEY_UserPB as u32);
    #[doc(alias = "GDK_KEY_Utilde")]
    pub const Utilde: Self = Self(ffi::GDK_KEY_Utilde as u32);
    #[doc(alias = "GDK_KEY_V")]
    pub const V: Self = Self(ffi::GDK_KEY_V as u32);
    #[doc(alias = "GDK_KEY_VOD")]
    pub const VOD: Self = Self(ffi::GDK_KEY_VOD as u32);
    #[doc(alias = "GDK_KEY_VendorHome")]
    pub const VendorHome: Self = Self(ffi::GDK_KEY_VendorHome as u32);
    #[doc(alias = "GDK_KEY_Video")]
    pub const Video: Self = Self(ffi::GDK_KEY_Video as u32);
    #[doc(alias = "GDK_KEY_VideoPhone")]
    pub const VideoPhone: Self = Self(ffi::GDK_KEY_VideoPhone as u32);
    #[doc(alias = "GDK_KEY_View")]
    pub const View: Self = Self(ffi::GDK_KEY_View as u32);
    #[doc(alias = "GDK_KEY_VoiceCommand")]
    pub const VoiceCommand: Self = Self(ffi::GDK_KEY_VoiceCommand as u32);
    #[doc(alias = "GDK_KEY_Voicemail")]
    pub const Voicemail: Self = Self(ffi::GDK_KEY_Voicemail as u32);
    #[doc(alias = "GDK_KEY_VoidSymbol")]
    pub const VoidSymbol: Self = Self(ffi::GDK_KEY_VoidSymbol as u32);
    #[doc(alias = "GDK_KEY_W")]
    pub const W: Self = Self(ffi::GDK_KEY_W as u32);
    #[doc(alias = "GDK_KEY_WLAN")]
    pub const WLAN: Self = Self(ffi::GDK_KEY_WLAN as u32);
    #[doc(alias = "GDK_KEY_WPSButton")]
    pub const WPSButton: Self = Self(ffi::GDK_KEY_WPSButton as u32);
    #[doc(alias = "GDK_KEY_WWAN")]
    pub const WWAN: Self = Self(ffi::GDK_KEY_WWAN as u32);
    #[doc(alias = "GDK_KEY_WWW")]
    pub const WWW: Self = Self(ffi::GDK_KEY_WWW as u32);
    #[doc(alias = "GDK_KEY_Wacute")]
    pub const Wacute: Self = Self(ffi::GDK_KEY_Wacute as u32);
    #[doc(alias = "GDK_KEY_WakeUp")]
    pub const WakeUp: Self = Self(ffi::GDK_KEY_WakeUp as u32);
    #[doc(alias = "GDK_KEY_Wcircumflex")]
    pub const Wcircumflex: Self = Self(ffi::GDK_KEY_Wcircumflex as u32);
    #[doc(alias = "GDK_KEY_Wdiaeresis")]
    pub const Wdiaeresis: Self = Self(ffi::GDK_KEY_Wdiaeresis as u32);
    #[doc(alias = "GDK_KEY_WebCam")]
    pub const WebCam: Self = Self(ffi::GDK_KEY_WebCam as u32);
    #[doc(alias = "GDK_KEY_Wgrave")]
    pub const Wgrave: Self = Self(ffi::GDK_KEY_Wgrave as u32);
    #[doc(alias = "GDK_KEY_WheelButton")]
    pub const WheelButton: Self = Self(ffi::GDK_KEY_WheelButton as u32);
    #[doc(alias = "GDK_KEY_WindowClear")]
    pub const WindowClear: Self = Self(ffi::GDK_KEY_WindowClear as u32);
    #[doc(alias = "GDK_KEY_WonSign")]
    pub const WonSign: Self = Self(ffi::GDK_KEY_WonSign as u32);
    #[doc(alias = "GDK_KEY_Word")]
    pub const Word: Self = Self(ffi::GDK_KEY_Word as u32);
    #[doc(alias = "GDK_KEY_X")]
    pub const X: Self = Self(ffi::GDK_KEY_X as u32);
    #[doc(alias = "GDK_KEY_Xabovedot")]
    pub const Xabovedot: Self = Self(ffi::GDK_KEY_Xabovedot as u32);
    #[doc(alias = "GDK_KEY_Xfer")]
    pub const Xfer: Self = Self(ffi::GDK_KEY_Xfer as u32);
    #[doc(alias = "GDK_KEY_Y")]
    pub const Y: Self = Self(ffi::GDK_KEY_Y as u32);
    #[doc(alias = "GDK_KEY_Yacute")]
    pub const Yacute: Self = Self(ffi::GDK_KEY_Yacute as u32);
    #[doc(alias = "GDK_KEY_Ybelowdot")]
    pub const Ybelowdot: Self = Self(ffi::GDK_KEY_Ybelowdot as u32);
    #[doc(alias = "GDK_KEY_Ycircumflex")]
    pub const Ycircumflex: Self = Self(ffi::GDK_KEY_Ycircumflex as u32);
    #[doc(alias = "GDK_KEY_Ydiaeresis")]
    pub const Ydiaeresis: Self = Self(ffi::GDK_KEY_Ydiaeresis as u32);
    #[doc(alias = "GDK_KEY_Yellow")]
    pub const Yellow: Self = Self(ffi::GDK_KEY_Yellow as u32);
    #[doc(alias = "GDK_KEY_Ygrave")]
    pub const Ygrave: Self = Self(ffi::GDK_KEY_Ygrave as u32);
    #[doc(alias = "GDK_KEY_Yhook")]
    pub const Yhook: Self = Self(ffi::GDK_KEY_Yhook as u32);
    #[doc(alias = "GDK_KEY_Ytilde")]
    pub const Ytilde: Self = Self(ffi::GDK_KEY_Ytilde as u32);
    #[doc(alias = "GDK_KEY_Z")]
    pub const Z: Self = Self(ffi::GDK_KEY_Z as u32);
    #[doc(alias = "GDK_KEY_Zabovedot")]
    pub const Zabovedot: Self = Self(ffi::GDK_KEY_Zabovedot as u32);
    #[doc(alias = "GDK_KEY_Zacute")]
    pub const Zacute: Self = Self(ffi::GDK_KEY_Zacute as u32);
    #[doc(alias = "GDK_KEY_Zcaron")]
    pub const Zcaron: Self = Self(ffi::GDK_KEY_Zcaron as u32);
    #[doc(alias = "GDK_KEY_Zen_Koho")]
    pub const Zen_Koho: Self = Self(ffi::GDK_KEY_Zen_Koho as u32);
    #[doc(alias = "GDK_KEY_Zenkaku")]
    pub const Zenkaku: Self = Self(ffi::GDK_KEY_Zenkaku as u32);
    #[doc(alias = "GDK_KEY_Zenkaku_Hankaku")]
    pub const Zenkaku_Hankaku: Self = Self(ffi::GDK_KEY_Zenkaku_Hankaku as u32);
    #[doc(alias = "GDK_KEY_ZoomIn")]
    pub const ZoomIn: Self = Self(ffi::GDK_KEY_ZoomIn as u32);
    #[doc(alias = "GDK_KEY_ZoomOut")]
    pub const ZoomOut: Self = Self(ffi::GDK_KEY_ZoomOut as u32);
    #[doc(alias = "GDK_KEY_ZoomReset")]
    pub const ZoomReset: Self = Self(ffi::GDK_KEY_ZoomReset as u32);
    #[doc(alias = "GDK_KEY_Zstroke")]
    pub const Zstroke: Self = Self(ffi::GDK_KEY_Zstroke as u32);
    #[doc(alias = "GDK_KEY_a")]
    pub const a: Self = Self(ffi::GDK_KEY_a as u32);
    #[doc(alias = "GDK_KEY_aacute")]
    pub const aacute: Self = Self(ffi::GDK_KEY_aacute as u32);
    #[doc(alias = "GDK_KEY_abelowdot")]
    pub const abelowdot: Self = Self(ffi::GDK_KEY_abelowdot as u32);
    #[doc(alias = "GDK_KEY_abovedot")]
    pub const abovedot: Self = Self(ffi::GDK_KEY_abovedot as u32);
    #[doc(alias = "GDK_KEY_abreve")]
    pub const abreve: Self = Self(ffi::GDK_KEY_abreve as u32);
    #[doc(alias = "GDK_KEY_abreveacute")]
    pub const abreveacute: Self = Self(ffi::GDK_KEY_abreveacute as u32);
    #[doc(alias = "GDK_KEY_abrevebelowdot")]
    pub const abrevebelowdot: Self = Self(ffi::GDK_KEY_abrevebelowdot as u32);
    #[doc(alias = "GDK_KEY_abrevegrave")]
    pub const abrevegrave: Self = Self(ffi::GDK_KEY_abrevegrave as u32);
    #[doc(alias = "GDK_KEY_abrevehook")]
    pub const abrevehook: Self = Self(ffi::GDK_KEY_abrevehook as u32);
    #[doc(alias = "GDK_KEY_abrevetilde")]
    pub const abrevetilde: Self = Self(ffi::GDK_KEY_abrevetilde as u32);
    #[doc(alias = "GDK_KEY_acircumflex")]
    pub const acircumflex: Self = Self(ffi::GDK_KEY_acircumflex as u32);
    #[doc(alias = "GDK_KEY_acircumflexacute")]
    pub const acircumflexacute: Self = Self(ffi::GDK_KEY_acircumflexacute as u32);
    #[doc(alias = "GDK_KEY_acircumflexbelowdot")]
    pub const acircumflexbelowdot: Self = Self(ffi::GDK_KEY_acircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_acircumflexgrave")]
    pub const acircumflexgrave: Self = Self(ffi::GDK_KEY_acircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_acircumflexhook")]
    pub const acircumflexhook: Self = Self(ffi::GDK_KEY_acircumflexhook as u32);
    #[doc(alias = "GDK_KEY_acircumflextilde")]
    pub const acircumflextilde: Self = Self(ffi::GDK_KEY_acircumflextilde as u32);
    #[doc(alias = "GDK_KEY_acute")]
    pub const acute: Self = Self(ffi::GDK_KEY_acute as u32);
    #[doc(alias = "GDK_KEY_adiaeresis")]
    pub const adiaeresis: Self = Self(ffi::GDK_KEY_adiaeresis as u32);
    #[doc(alias = "GDK_KEY_ae")]
    pub const ae: Self = Self(ffi::GDK_KEY_ae as u32);
    #[doc(alias = "GDK_KEY_agrave")]
    pub const agrave: Self = Self(ffi::GDK_KEY_agrave as u32);
    #[doc(alias = "GDK_KEY_ahook")]
    pub const ahook: Self = Self(ffi::GDK_KEY_ahook as u32);
    #[doc(alias = "GDK_KEY_amacron")]
    pub const amacron: Self = Self(ffi::GDK_KEY_amacron as u32);
    #[doc(alias = "GDK_KEY_ampersand")]
    pub const ampersand: Self = Self(ffi::GDK_KEY_ampersand as u32);
    #[doc(alias = "GDK_KEY_aogonek")]
    pub const aogonek: Self = Self(ffi::GDK_KEY_aogonek as u32);
    #[doc(alias = "GDK_KEY_apostrophe")]
    pub const apostrophe: Self = Self(ffi::GDK_KEY_apostrophe as u32);
    #[doc(alias = "GDK_KEY_approxeq")]
    pub const approxeq: Self = Self(ffi::GDK_KEY_approxeq as u32);
    #[doc(alias = "GDK_KEY_approximate")]
    pub const approximate: Self = Self(ffi::GDK_KEY_approximate as u32);
    #[doc(alias = "GDK_KEY_aring")]
    pub const aring: Self = Self(ffi::GDK_KEY_aring as u32);
    #[doc(alias = "GDK_KEY_asciicircum")]
    pub const asciicircum: Self = Self(ffi::GDK_KEY_asciicircum as u32);
    #[doc(alias = "GDK_KEY_asciitilde")]
    pub const asciitilde: Self = Self(ffi::GDK_KEY_asciitilde as u32);
    #[doc(alias = "GDK_KEY_asterisk")]
    pub const asterisk: Self = Self(ffi::GDK_KEY_asterisk as u32);
    #[doc(alias = "GDK_KEY_at")]
    pub const at: Self = Self(ffi::GDK_KEY_at as u32);
    #[doc(alias = "GDK_KEY_atilde")]
    pub const atilde: Self = Self(ffi::GDK_KEY_atilde as u32);
    #[doc(alias = "GDK_KEY_b")]
    pub const b: Self = Self(ffi::GDK_KEY_b as u32);
    #[doc(alias = "GDK_KEY_babovedot")]
    pub const babovedot: Self = Self(ffi::GDK_KEY_babovedot as u32);
    #[doc(alias = "GDK_KEY_backslash")]
    pub const backslash: Self = Self(ffi::GDK_KEY_backslash as u32);
    #[doc(alias = "GDK_KEY_ballotcross")]
    pub const ballotcross: Self = Self(ffi::GDK_KEY_ballotcross as u32);
    #[doc(alias = "GDK_KEY_bar")]
    pub const bar: Self = Self(ffi::GDK_KEY_bar as u32);
    #[doc(alias = "GDK_KEY_because")]
    pub const because: Self = Self(ffi::GDK_KEY_because as u32);
    #[doc(alias = "GDK_KEY_blank")]
    pub const blank: Self = Self(ffi::GDK_KEY_blank as u32);
    #[doc(alias = "GDK_KEY_botintegral")]
    pub const botintegral: Self = Self(ffi::GDK_KEY_botintegral as u32);
    #[doc(alias = "GDK_KEY_botleftparens")]
    pub const botleftparens: Self = Self(ffi::GDK_KEY_botleftparens as u32);
    #[doc(alias = "GDK_KEY_botleftsqbracket")]
    pub const botleftsqbracket: Self = Self(ffi::GDK_KEY_botleftsqbracket as u32);
    #[doc(alias = "GDK_KEY_botleftsummation")]
    pub const botleftsummation: Self = Self(ffi::GDK_KEY_botleftsummation as u32);
    #[doc(alias = "GDK_KEY_botrightparens")]
    pub const botrightparens: Self = Self(ffi::GDK_KEY_botrightparens as u32);
    #[doc(alias = "GDK_KEY_botrightsqbracket")]
    pub const botrightsqbracket: Self = Self(ffi::GDK_KEY_botrightsqbracket as u32);
    #[doc(alias = "GDK_KEY_botrightsummation")]
    pub const botrightsummation: Self = Self(ffi::GDK_KEY_botrightsummation as u32);
    #[doc(alias = "GDK_KEY_bott")]
    pub const bott: Self = Self(ffi::GDK_KEY_bott as u32);
    #[doc(alias = "GDK_KEY_botvertsummationconnector")]
    pub const botvertsummationconnector: Self = Self(ffi::GDK_KEY_botvertsummationconnector as u32);
    #[doc(alias = "GDK_KEY_braceleft")]
    pub const braceleft: Self = Self(ffi::GDK_KEY_braceleft as u32);
    #[doc(alias = "GDK_KEY_braceright")]
    pub const braceright: Self = Self(ffi::GDK_KEY_braceright as u32);
    #[doc(alias = "GDK_KEY_bracketleft")]
    pub const bracketleft: Self = Self(ffi::GDK_KEY_bracketleft as u32);
    #[doc(alias = "GDK_KEY_bracketright")]
    pub const bracketright: Self = Self(ffi::GDK_KEY_bracketright as u32);
    #[doc(alias = "GDK_KEY_braille_blank")]
    pub const braille_blank: Self = Self(ffi::GDK_KEY_braille_blank as u32);
    #[doc(alias = "GDK_KEY_braille_dot_1")]
    pub const braille_dot_1: Self = Self(ffi::GDK_KEY_braille_dot_1 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_10")]
    pub const braille_dot_10: Self = Self(ffi::GDK_KEY_braille_dot_10 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_2")]
    pub const braille_dot_2: Self = Self(ffi::GDK_KEY_braille_dot_2 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_3")]
    pub const braille_dot_3: Self = Self(ffi::GDK_KEY_braille_dot_3 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_4")]
    pub const braille_dot_4: Self = Self(ffi::GDK_KEY_braille_dot_4 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_5")]
    pub const braille_dot_5: Self = Self(ffi::GDK_KEY_braille_dot_5 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_6")]
    pub const braille_dot_6: Self = Self(ffi::GDK_KEY_braille_dot_6 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_7")]
    pub const braille_dot_7: Self = Self(ffi::GDK_KEY_braille_dot_7 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_8")]
    pub const braille_dot_8: Self = Self(ffi::GDK_KEY_braille_dot_8 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_9")]
    pub const braille_dot_9: Self = Self(ffi::GDK_KEY_braille_dot_9 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1")]
    pub const braille_dots_1: Self = Self(ffi::GDK_KEY_braille_dots_1 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12")]
    pub const braille_dots_12: Self = Self(ffi::GDK_KEY_braille_dots_12 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123")]
    pub const braille_dots_123: Self = Self(ffi::GDK_KEY_braille_dots_123 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234")]
    pub const braille_dots_1234: Self = Self(ffi::GDK_KEY_braille_dots_1234 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12345")]
    pub const braille_dots_12345: Self = Self(ffi::GDK_KEY_braille_dots_12345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123456")]
    pub const braille_dots_123456: Self = Self(ffi::GDK_KEY_braille_dots_123456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234567")]
    pub const braille_dots_1234567: Self = Self(ffi::GDK_KEY_braille_dots_1234567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12345678")]
    pub const braille_dots_12345678: Self = Self(ffi::GDK_KEY_braille_dots_12345678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234568")]
    pub const braille_dots_1234568: Self = Self(ffi::GDK_KEY_braille_dots_1234568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123457")]
    pub const braille_dots_123457: Self = Self(ffi::GDK_KEY_braille_dots_123457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234578")]
    pub const braille_dots_1234578: Self = Self(ffi::GDK_KEY_braille_dots_1234578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123458")]
    pub const braille_dots_123458: Self = Self(ffi::GDK_KEY_braille_dots_123458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12346")]
    pub const braille_dots_12346: Self = Self(ffi::GDK_KEY_braille_dots_12346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123467")]
    pub const braille_dots_123467: Self = Self(ffi::GDK_KEY_braille_dots_123467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234678")]
    pub const braille_dots_1234678: Self = Self(ffi::GDK_KEY_braille_dots_1234678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123468")]
    pub const braille_dots_123468: Self = Self(ffi::GDK_KEY_braille_dots_123468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12347")]
    pub const braille_dots_12347: Self = Self(ffi::GDK_KEY_braille_dots_12347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123478")]
    pub const braille_dots_123478: Self = Self(ffi::GDK_KEY_braille_dots_123478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12348")]
    pub const braille_dots_12348: Self = Self(ffi::GDK_KEY_braille_dots_12348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1235")]
    pub const braille_dots_1235: Self = Self(ffi::GDK_KEY_braille_dots_1235 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12356")]
    pub const braille_dots_12356: Self = Self(ffi::GDK_KEY_braille_dots_12356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123567")]
    pub const braille_dots_123567: Self = Self(ffi::GDK_KEY_braille_dots_123567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1235678")]
    pub const braille_dots_1235678: Self = Self(ffi::GDK_KEY_braille_dots_1235678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123568")]
    pub const braille_dots_123568: Self = Self(ffi::GDK_KEY_braille_dots_123568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12357")]
    pub const braille_dots_12357: Self = Self(ffi::GDK_KEY_braille_dots_12357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123578")]
    pub const braille_dots_123578: Self = Self(ffi::GDK_KEY_braille_dots_123578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12358")]
    pub const braille_dots_12358: Self = Self(ffi::GDK_KEY_braille_dots_12358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1236")]
    pub const braille_dots_1236: Self = Self(ffi::GDK_KEY_braille_dots_1236 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12367")]
    pub const braille_dots_12367: Self = Self(ffi::GDK_KEY_braille_dots_12367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123678")]
    pub const braille_dots_123678: Self = Self(ffi::GDK_KEY_braille_dots_123678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12368")]
    pub const braille_dots_12368: Self = Self(ffi::GDK_KEY_braille_dots_12368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1237")]
    pub const braille_dots_1237: Self = Self(ffi::GDK_KEY_braille_dots_1237 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12378")]
    pub const braille_dots_12378: Self = Self(ffi::GDK_KEY_braille_dots_12378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1238")]
    pub const braille_dots_1238: Self = Self(ffi::GDK_KEY_braille_dots_1238 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124")]
    pub const braille_dots_124: Self = Self(ffi::GDK_KEY_braille_dots_124 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1245")]
    pub const braille_dots_1245: Self = Self(ffi::GDK_KEY_braille_dots_1245 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12456")]
    pub const braille_dots_12456: Self = Self(ffi::GDK_KEY_braille_dots_12456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124567")]
    pub const braille_dots_124567: Self = Self(ffi::GDK_KEY_braille_dots_124567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1245678")]
    pub const braille_dots_1245678: Self = Self(ffi::GDK_KEY_braille_dots_1245678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124568")]
    pub const braille_dots_124568: Self = Self(ffi::GDK_KEY_braille_dots_124568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12457")]
    pub const braille_dots_12457: Self = Self(ffi::GDK_KEY_braille_dots_12457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124578")]
    pub const braille_dots_124578: Self = Self(ffi::GDK_KEY_braille_dots_124578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12458")]
    pub const braille_dots_12458: Self = Self(ffi::GDK_KEY_braille_dots_12458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1246")]
    pub const braille_dots_1246: Self = Self(ffi::GDK_KEY_braille_dots_1246 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12467")]
    pub const braille_dots_12467: Self = Self(ffi::GDK_KEY_braille_dots_12467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124678")]
    pub const braille_dots_124678: Self = Self(ffi::GDK_KEY_braille_dots_124678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12468")]
    pub const braille_dots_12468: Self = Self(ffi::GDK_KEY_braille_dots_12468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1247")]
    pub const braille_dots_1247: Self = Self(ffi::GDK_KEY_braille_dots_1247 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12478")]
    pub const braille_dots_12478: Self = Self(ffi::GDK_KEY_braille_dots_12478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1248")]
    pub const braille_dots_1248: Self = Self(ffi::GDK_KEY_braille_dots_1248 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_125")]
    pub const braille_dots_125: Self = Self(ffi::GDK_KEY_braille_dots_125 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1256")]
    pub const braille_dots_1256: Self = Self(ffi::GDK_KEY_braille_dots_1256 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12567")]
    pub const braille_dots_12567: Self = Self(ffi::GDK_KEY_braille_dots_12567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_125678")]
    pub const braille_dots_125678: Self = Self(ffi::GDK_KEY_braille_dots_125678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12568")]
    pub const braille_dots_12568: Self = Self(ffi::GDK_KEY_braille_dots_12568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1257")]
    pub const braille_dots_1257: Self = Self(ffi::GDK_KEY_braille_dots_1257 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12578")]
    pub const braille_dots_12578: Self = Self(ffi::GDK_KEY_braille_dots_12578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1258")]
    pub const braille_dots_1258: Self = Self(ffi::GDK_KEY_braille_dots_1258 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_126")]
    pub const braille_dots_126: Self = Self(ffi::GDK_KEY_braille_dots_126 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1267")]
    pub const braille_dots_1267: Self = Self(ffi::GDK_KEY_braille_dots_1267 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12678")]
    pub const braille_dots_12678: Self = Self(ffi::GDK_KEY_braille_dots_12678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1268")]
    pub const braille_dots_1268: Self = Self(ffi::GDK_KEY_braille_dots_1268 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_127")]
    pub const braille_dots_127: Self = Self(ffi::GDK_KEY_braille_dots_127 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1278")]
    pub const braille_dots_1278: Self = Self(ffi::GDK_KEY_braille_dots_1278 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_128")]
    pub const braille_dots_128: Self = Self(ffi::GDK_KEY_braille_dots_128 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13")]
    pub const braille_dots_13: Self = Self(ffi::GDK_KEY_braille_dots_13 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134")]
    pub const braille_dots_134: Self = Self(ffi::GDK_KEY_braille_dots_134 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1345")]
    pub const braille_dots_1345: Self = Self(ffi::GDK_KEY_braille_dots_1345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13456")]
    pub const braille_dots_13456: Self = Self(ffi::GDK_KEY_braille_dots_13456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134567")]
    pub const braille_dots_134567: Self = Self(ffi::GDK_KEY_braille_dots_134567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1345678")]
    pub const braille_dots_1345678: Self = Self(ffi::GDK_KEY_braille_dots_1345678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134568")]
    pub const braille_dots_134568: Self = Self(ffi::GDK_KEY_braille_dots_134568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13457")]
    pub const braille_dots_13457: Self = Self(ffi::GDK_KEY_braille_dots_13457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134578")]
    pub const braille_dots_134578: Self = Self(ffi::GDK_KEY_braille_dots_134578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13458")]
    pub const braille_dots_13458: Self = Self(ffi::GDK_KEY_braille_dots_13458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1346")]
    pub const braille_dots_1346: Self = Self(ffi::GDK_KEY_braille_dots_1346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13467")]
    pub const braille_dots_13467: Self = Self(ffi::GDK_KEY_braille_dots_13467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134678")]
    pub const braille_dots_134678: Self = Self(ffi::GDK_KEY_braille_dots_134678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13468")]
    pub const braille_dots_13468: Self = Self(ffi::GDK_KEY_braille_dots_13468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1347")]
    pub const braille_dots_1347: Self = Self(ffi::GDK_KEY_braille_dots_1347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13478")]
    pub const braille_dots_13478: Self = Self(ffi::GDK_KEY_braille_dots_13478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1348")]
    pub const braille_dots_1348: Self = Self(ffi::GDK_KEY_braille_dots_1348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_135")]
    pub const braille_dots_135: Self = Self(ffi::GDK_KEY_braille_dots_135 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1356")]
    pub const braille_dots_1356: Self = Self(ffi::GDK_KEY_braille_dots_1356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13567")]
    pub const braille_dots_13567: Self = Self(ffi::GDK_KEY_braille_dots_13567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_135678")]
    pub const braille_dots_135678: Self = Self(ffi::GDK_KEY_braille_dots_135678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13568")]
    pub const braille_dots_13568: Self = Self(ffi::GDK_KEY_braille_dots_13568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1357")]
    pub const braille_dots_1357: Self = Self(ffi::GDK_KEY_braille_dots_1357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13578")]
    pub const braille_dots_13578: Self = Self(ffi::GDK_KEY_braille_dots_13578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1358")]
    pub const braille_dots_1358: Self = Self(ffi::GDK_KEY_braille_dots_1358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_136")]
    pub const braille_dots_136: Self = Self(ffi::GDK_KEY_braille_dots_136 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1367")]
    pub const braille_dots_1367: Self = Self(ffi::GDK_KEY_braille_dots_1367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13678")]
    pub const braille_dots_13678: Self = Self(ffi::GDK_KEY_braille_dots_13678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1368")]
    pub const braille_dots_1368: Self = Self(ffi::GDK_KEY_braille_dots_1368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_137")]
    pub const braille_dots_137: Self = Self(ffi::GDK_KEY_braille_dots_137 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1378")]
    pub const braille_dots_1378: Self = Self(ffi::GDK_KEY_braille_dots_1378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_138")]
    pub const braille_dots_138: Self = Self(ffi::GDK_KEY_braille_dots_138 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14")]
    pub const braille_dots_14: Self = Self(ffi::GDK_KEY_braille_dots_14 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_145")]
    pub const braille_dots_145: Self = Self(ffi::GDK_KEY_braille_dots_145 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1456")]
    pub const braille_dots_1456: Self = Self(ffi::GDK_KEY_braille_dots_1456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14567")]
    pub const braille_dots_14567: Self = Self(ffi::GDK_KEY_braille_dots_14567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_145678")]
    pub const braille_dots_145678: Self = Self(ffi::GDK_KEY_braille_dots_145678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14568")]
    pub const braille_dots_14568: Self = Self(ffi::GDK_KEY_braille_dots_14568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1457")]
    pub const braille_dots_1457: Self = Self(ffi::GDK_KEY_braille_dots_1457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14578")]
    pub const braille_dots_14578: Self = Self(ffi::GDK_KEY_braille_dots_14578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1458")]
    pub const braille_dots_1458: Self = Self(ffi::GDK_KEY_braille_dots_1458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_146")]
    pub const braille_dots_146: Self = Self(ffi::GDK_KEY_braille_dots_146 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1467")]
    pub const braille_dots_1467: Self = Self(ffi::GDK_KEY_braille_dots_1467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14678")]
    pub const braille_dots_14678: Self = Self(ffi::GDK_KEY_braille_dots_14678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1468")]
    pub const braille_dots_1468: Self = Self(ffi::GDK_KEY_braille_dots_1468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_147")]
    pub const braille_dots_147: Self = Self(ffi::GDK_KEY_braille_dots_147 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1478")]
    pub const braille_dots_1478: Self = Self(ffi::GDK_KEY_braille_dots_1478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_148")]
    pub const braille_dots_148: Self = Self(ffi::GDK_KEY_braille_dots_148 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_15")]
    pub const braille_dots_15: Self = Self(ffi::GDK_KEY_braille_dots_15 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_156")]
    pub const braille_dots_156: Self = Self(ffi::GDK_KEY_braille_dots_156 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1567")]
    pub const braille_dots_1567: Self = Self(ffi::GDK_KEY_braille_dots_1567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_15678")]
    pub const braille_dots_15678: Self = Self(ffi::GDK_KEY_braille_dots_15678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1568")]
    pub const braille_dots_1568: Self = Self(ffi::GDK_KEY_braille_dots_1568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_157")]
    pub const braille_dots_157: Self = Self(ffi::GDK_KEY_braille_dots_157 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1578")]
    pub const braille_dots_1578: Self = Self(ffi::GDK_KEY_braille_dots_1578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_158")]
    pub const braille_dots_158: Self = Self(ffi::GDK_KEY_braille_dots_158 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_16")]
    pub const braille_dots_16: Self = Self(ffi::GDK_KEY_braille_dots_16 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_167")]
    pub const braille_dots_167: Self = Self(ffi::GDK_KEY_braille_dots_167 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1678")]
    pub const braille_dots_1678: Self = Self(ffi::GDK_KEY_braille_dots_1678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_168")]
    pub const braille_dots_168: Self = Self(ffi::GDK_KEY_braille_dots_168 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_17")]
    pub const braille_dots_17: Self = Self(ffi::GDK_KEY_braille_dots_17 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_178")]
    pub const braille_dots_178: Self = Self(ffi::GDK_KEY_braille_dots_178 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_18")]
    pub const braille_dots_18: Self = Self(ffi::GDK_KEY_braille_dots_18 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2")]
    pub const braille_dots_2: Self = Self(ffi::GDK_KEY_braille_dots_2 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23")]
    pub const braille_dots_23: Self = Self(ffi::GDK_KEY_braille_dots_23 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234")]
    pub const braille_dots_234: Self = Self(ffi::GDK_KEY_braille_dots_234 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2345")]
    pub const braille_dots_2345: Self = Self(ffi::GDK_KEY_braille_dots_2345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23456")]
    pub const braille_dots_23456: Self = Self(ffi::GDK_KEY_braille_dots_23456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234567")]
    pub const braille_dots_234567: Self = Self(ffi::GDK_KEY_braille_dots_234567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2345678")]
    pub const braille_dots_2345678: Self = Self(ffi::GDK_KEY_braille_dots_2345678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234568")]
    pub const braille_dots_234568: Self = Self(ffi::GDK_KEY_braille_dots_234568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23457")]
    pub const braille_dots_23457: Self = Self(ffi::GDK_KEY_braille_dots_23457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234578")]
    pub const braille_dots_234578: Self = Self(ffi::GDK_KEY_braille_dots_234578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23458")]
    pub const braille_dots_23458: Self = Self(ffi::GDK_KEY_braille_dots_23458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2346")]
    pub const braille_dots_2346: Self = Self(ffi::GDK_KEY_braille_dots_2346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23467")]
    pub const braille_dots_23467: Self = Self(ffi::GDK_KEY_braille_dots_23467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234678")]
    pub const braille_dots_234678: Self = Self(ffi::GDK_KEY_braille_dots_234678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23468")]
    pub const braille_dots_23468: Self = Self(ffi::GDK_KEY_braille_dots_23468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2347")]
    pub const braille_dots_2347: Self = Self(ffi::GDK_KEY_braille_dots_2347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23478")]
    pub const braille_dots_23478: Self = Self(ffi::GDK_KEY_braille_dots_23478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2348")]
    pub const braille_dots_2348: Self = Self(ffi::GDK_KEY_braille_dots_2348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_235")]
    pub const braille_dots_235: Self = Self(ffi::GDK_KEY_braille_dots_235 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2356")]
    pub const braille_dots_2356: Self = Self(ffi::GDK_KEY_braille_dots_2356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23567")]
    pub const braille_dots_23567: Self = Self(ffi::GDK_KEY_braille_dots_23567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_235678")]
    pub const braille_dots_235678: Self = Self(ffi::GDK_KEY_braille_dots_235678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23568")]
    pub const braille_dots_23568: Self = Self(ffi::GDK_KEY_braille_dots_23568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2357")]
    pub const braille_dots_2357: Self = Self(ffi::GDK_KEY_braille_dots_2357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23578")]
    pub const braille_dots_23578: Self = Self(ffi::GDK_KEY_braille_dots_23578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2358")]
    pub const braille_dots_2358: Self = Self(ffi::GDK_KEY_braille_dots_2358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_236")]
    pub const braille_dots_236: Self = Self(ffi::GDK_KEY_braille_dots_236 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2367")]
    pub const braille_dots_2367: Self = Self(ffi::GDK_KEY_braille_dots_2367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23678")]
    pub const braille_dots_23678: Self = Self(ffi::GDK_KEY_braille_dots_23678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2368")]
    pub const braille_dots_2368: Self = Self(ffi::GDK_KEY_braille_dots_2368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_237")]
    pub const braille_dots_237: Self = Self(ffi::GDK_KEY_braille_dots_237 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2378")]
    pub const braille_dots_2378: Self = Self(ffi::GDK_KEY_braille_dots_2378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_238")]
    pub const braille_dots_238: Self = Self(ffi::GDK_KEY_braille_dots_238 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24")]
    pub const braille_dots_24: Self = Self(ffi::GDK_KEY_braille_dots_24 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_245")]
    pub const braille_dots_245: Self = Self(ffi::GDK_KEY_braille_dots_245 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2456")]
    pub const braille_dots_2456: Self = Self(ffi::GDK_KEY_braille_dots_2456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24567")]
    pub const braille_dots_24567: Self = Self(ffi::GDK_KEY_braille_dots_24567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_245678")]
    pub const braille_dots_245678: Self = Self(ffi::GDK_KEY_braille_dots_245678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24568")]
    pub const braille_dots_24568: Self = Self(ffi::GDK_KEY_braille_dots_24568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2457")]
    pub const braille_dots_2457: Self = Self(ffi::GDK_KEY_braille_dots_2457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24578")]
    pub const braille_dots_24578: Self = Self(ffi::GDK_KEY_braille_dots_24578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2458")]
    pub const braille_dots_2458: Self = Self(ffi::GDK_KEY_braille_dots_2458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_246")]
    pub const braille_dots_246: Self = Self(ffi::GDK_KEY_braille_dots_246 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2467")]
    pub const braille_dots_2467: Self = Self(ffi::GDK_KEY_braille_dots_2467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24678")]
    pub const braille_dots_24678: Self = Self(ffi::GDK_KEY_braille_dots_24678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2468")]
    pub const braille_dots_2468: Self = Self(ffi::GDK_KEY_braille_dots_2468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_247")]
    pub const braille_dots_247: Self = Self(ffi::GDK_KEY_braille_dots_247 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2478")]
    pub const braille_dots_2478: Self = Self(ffi::GDK_KEY_braille_dots_2478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_248")]
    pub const braille_dots_248: Self = Self(ffi::GDK_KEY_braille_dots_248 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_25")]
    pub const braille_dots_25: Self = Self(ffi::GDK_KEY_braille_dots_25 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_256")]
    pub const braille_dots_256: Self = Self(ffi::GDK_KEY_braille_dots_256 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2567")]
    pub const braille_dots_2567: Self = Self(ffi::GDK_KEY_braille_dots_2567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_25678")]
    pub const braille_dots_25678: Self = Self(ffi::GDK_KEY_braille_dots_25678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2568")]
    pub const braille_dots_2568: Self = Self(ffi::GDK_KEY_braille_dots_2568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_257")]
    pub const braille_dots_257: Self = Self(ffi::GDK_KEY_braille_dots_257 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2578")]
    pub const braille_dots_2578: Self = Self(ffi::GDK_KEY_braille_dots_2578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_258")]
    pub const braille_dots_258: Self = Self(ffi::GDK_KEY_braille_dots_258 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_26")]
    pub const braille_dots_26: Self = Self(ffi::GDK_KEY_braille_dots_26 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_267")]
    pub const braille_dots_267: Self = Self(ffi::GDK_KEY_braille_dots_267 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2678")]
    pub const braille_dots_2678: Self = Self(ffi::GDK_KEY_braille_dots_2678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_268")]
    pub const braille_dots_268: Self = Self(ffi::GDK_KEY_braille_dots_268 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_27")]
    pub const braille_dots_27: Self = Self(ffi::GDK_KEY_braille_dots_27 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_278")]
    pub const braille_dots_278: Self = Self(ffi::GDK_KEY_braille_dots_278 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_28")]
    pub const braille_dots_28: Self = Self(ffi::GDK_KEY_braille_dots_28 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3")]
    pub const braille_dots_3: Self = Self(ffi::GDK_KEY_braille_dots_3 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34")]
    pub const braille_dots_34: Self = Self(ffi::GDK_KEY_braille_dots_34 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_345")]
    pub const braille_dots_345: Self = Self(ffi::GDK_KEY_braille_dots_345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3456")]
    pub const braille_dots_3456: Self = Self(ffi::GDK_KEY_braille_dots_3456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34567")]
    pub const braille_dots_34567: Self = Self(ffi::GDK_KEY_braille_dots_34567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_345678")]
    pub const braille_dots_345678: Self = Self(ffi::GDK_KEY_braille_dots_345678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34568")]
    pub const braille_dots_34568: Self = Self(ffi::GDK_KEY_braille_dots_34568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3457")]
    pub const braille_dots_3457: Self = Self(ffi::GDK_KEY_braille_dots_3457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34578")]
    pub const braille_dots_34578: Self = Self(ffi::GDK_KEY_braille_dots_34578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3458")]
    pub const braille_dots_3458: Self = Self(ffi::GDK_KEY_braille_dots_3458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_346")]
    pub const braille_dots_346: Self = Self(ffi::GDK_KEY_braille_dots_346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3467")]
    pub const braille_dots_3467: Self = Self(ffi::GDK_KEY_braille_dots_3467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34678")]
    pub const braille_dots_34678: Self = Self(ffi::GDK_KEY_braille_dots_34678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3468")]
    pub const braille_dots_3468: Self = Self(ffi::GDK_KEY_braille_dots_3468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_347")]
    pub const braille_dots_347: Self = Self(ffi::GDK_KEY_braille_dots_347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3478")]
    pub const braille_dots_3478: Self = Self(ffi::GDK_KEY_braille_dots_3478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_348")]
    pub const braille_dots_348: Self = Self(ffi::GDK_KEY_braille_dots_348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_35")]
    pub const braille_dots_35: Self = Self(ffi::GDK_KEY_braille_dots_35 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_356")]
    pub const braille_dots_356: Self = Self(ffi::GDK_KEY_braille_dots_356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3567")]
    pub const braille_dots_3567: Self = Self(ffi::GDK_KEY_braille_dots_3567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_35678")]
    pub const braille_dots_35678: Self = Self(ffi::GDK_KEY_braille_dots_35678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3568")]
    pub const braille_dots_3568: Self = Self(ffi::GDK_KEY_braille_dots_3568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_357")]
    pub const braille_dots_357: Self = Self(ffi::GDK_KEY_braille_dots_357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3578")]
    pub const braille_dots_3578: Self = Self(ffi::GDK_KEY_braille_dots_3578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_358")]
    pub const braille_dots_358: Self = Self(ffi::GDK_KEY_braille_dots_358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_36")]
    pub const braille_dots_36: Self = Self(ffi::GDK_KEY_braille_dots_36 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_367")]
    pub const braille_dots_367: Self = Self(ffi::GDK_KEY_braille_dots_367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3678")]
    pub const braille_dots_3678: Self = Self(ffi::GDK_KEY_braille_dots_3678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_368")]
    pub const braille_dots_368: Self = Self(ffi::GDK_KEY_braille_dots_368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_37")]
    pub const braille_dots_37: Self = Self(ffi::GDK_KEY_braille_dots_37 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_378")]
    pub const braille_dots_378: Self = Self(ffi::GDK_KEY_braille_dots_378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_38")]
    pub const braille_dots_38: Self = Self(ffi::GDK_KEY_braille_dots_38 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4")]
    pub const braille_dots_4: Self = Self(ffi::GDK_KEY_braille_dots_4 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_45")]
    pub const braille_dots_45: Self = Self(ffi::GDK_KEY_braille_dots_45 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_456")]
    pub const braille_dots_456: Self = Self(ffi::GDK_KEY_braille_dots_456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4567")]
    pub const braille_dots_4567: Self = Self(ffi::GDK_KEY_braille_dots_4567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_45678")]
    pub const braille_dots_45678: Self = Self(ffi::GDK_KEY_braille_dots_45678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4568")]
    pub const braille_dots_4568: Self = Self(ffi::GDK_KEY_braille_dots_4568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_457")]
    pub const braille_dots_457: Self = Self(ffi::GDK_KEY_braille_dots_457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4578")]
    pub const braille_dots_4578: Self = Self(ffi::GDK_KEY_braille_dots_4578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_458")]
    pub const braille_dots_458: Self = Self(ffi::GDK_KEY_braille_dots_458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_46")]
    pub const braille_dots_46: Self = Self(ffi::GDK_KEY_braille_dots_46 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_467")]
    pub const braille_dots_467: Self = Self(ffi::GDK_KEY_braille_dots_467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4678")]
    pub const braille_dots_4678: Self = Self(ffi::GDK_KEY_braille_dots_4678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_468")]
    pub const braille_dots_468: Self = Self(ffi::GDK_KEY_braille_dots_468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_47")]
    pub const braille_dots_47: Self = Self(ffi::GDK_KEY_braille_dots_47 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_478")]
    pub const braille_dots_478: Self = Self(ffi::GDK_KEY_braille_dots_478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_48")]
    pub const braille_dots_48: Self = Self(ffi::GDK_KEY_braille_dots_48 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_5")]
    pub const braille_dots_5: Self = Self(ffi::GDK_KEY_braille_dots_5 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_56")]
    pub const braille_dots_56: Self = Self(ffi::GDK_KEY_braille_dots_56 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_567")]
    pub const braille_dots_567: Self = Self(ffi::GDK_KEY_braille_dots_567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_5678")]
    pub const braille_dots_5678: Self = Self(ffi::GDK_KEY_braille_dots_5678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_568")]
    pub const braille_dots_568: Self = Self(ffi::GDK_KEY_braille_dots_568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_57")]
    pub const braille_dots_57: Self = Self(ffi::GDK_KEY_braille_dots_57 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_578")]
    pub const braille_dots_578: Self = Self(ffi::GDK_KEY_braille_dots_578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_58")]
    pub const braille_dots_58: Self = Self(ffi::GDK_KEY_braille_dots_58 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_6")]
    pub const braille_dots_6: Self = Self(ffi::GDK_KEY_braille_dots_6 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_67")]
    pub const braille_dots_67: Self = Self(ffi::GDK_KEY_braille_dots_67 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_678")]
    pub const braille_dots_678: Self = Self(ffi::GDK_KEY_braille_dots_678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_68")]
    pub const braille_dots_68: Self = Self(ffi::GDK_KEY_braille_dots_68 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_7")]
    pub const braille_dots_7: Self = Self(ffi::GDK_KEY_braille_dots_7 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_78")]
    pub const braille_dots_78: Self = Self(ffi::GDK_KEY_braille_dots_78 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_8")]
    pub const braille_dots_8: Self = Self(ffi::GDK_KEY_braille_dots_8 as u32);
    #[doc(alias = "GDK_KEY_breve")]
    pub const breve: Self = Self(ffi::GDK_KEY_breve as u32);
    #[doc(alias = "GDK_KEY_brokenbar")]
    pub const brokenbar: Self = Self(ffi::GDK_KEY_brokenbar as u32);
    #[doc(alias = "GDK_KEY_c")]
    pub const c: Self = Self(ffi::GDK_KEY_c as u32);
    #[doc(alias = "GDK_KEY_c_h")]
    pub const c_h: Self = Self(ffi::GDK_KEY_c_h as u32);
    #[doc(alias = "GDK_KEY_cabovedot")]
    pub const cabovedot: Self = Self(ffi::GDK_KEY_cabovedot as u32);
    #[doc(alias = "GDK_KEY_cacute")]
    pub const cacute: Self = Self(ffi::GDK_KEY_cacute as u32);
    #[doc(alias = "GDK_KEY_careof")]
    pub const careof: Self = Self(ffi::GDK_KEY_careof as u32);
    #[doc(alias = "GDK_KEY_caret")]
    pub const caret: Self = Self(ffi::GDK_KEY_caret as u32);
    #[doc(alias = "GDK_KEY_caron")]
    pub const caron: Self = Self(ffi::GDK_KEY_caron as u32);
    #[doc(alias = "GDK_KEY_ccaron")]
    pub const ccaron: Self = Self(ffi::GDK_KEY_ccaron as u32);
    #[doc(alias = "GDK_KEY_ccedilla")]
    pub const ccedilla: Self = Self(ffi::GDK_KEY_ccedilla as u32);
    #[doc(alias = "GDK_KEY_ccircumflex")]
    pub const ccircumflex: Self = Self(ffi::GDK_KEY_ccircumflex as u32);
    #[doc(alias = "GDK_KEY_cedilla")]
    pub const cedilla: Self = Self(ffi::GDK_KEY_cedilla as u32);
    #[doc(alias = "GDK_KEY_cent")]
    pub const cent: Self = Self(ffi::GDK_KEY_cent as u32);
    #[doc(alias = "GDK_KEY_ch")]
    pub const ch: Self = Self(ffi::GDK_KEY_ch as u32);
    #[doc(alias = "GDK_KEY_checkerboard")]
    pub const checkerboard: Self = Self(ffi::GDK_KEY_checkerboard as u32);
    #[doc(alias = "GDK_KEY_checkmark")]
    pub const checkmark: Self = Self(ffi::GDK_KEY_checkmark as u32);
    #[doc(alias = "GDK_KEY_circle")]
    pub const circle: Self = Self(ffi::GDK_KEY_circle as u32);
    #[doc(alias = "GDK_KEY_club")]
    pub const club: Self = Self(ffi::GDK_KEY_club as u32);
    #[doc(alias = "GDK_KEY_colon")]
    pub const colon: Self = Self(ffi::GDK_KEY_colon as u32);
    #[doc(alias = "GDK_KEY_combining_acute")]
    pub const combining_acute: Self = Self(ffi::GDK_KEY_combining_acute as u32);
    #[doc(alias = "GDK_KEY_combining_belowdot")]
    pub const combining_belowdot: Self = Self(ffi::GDK_KEY_combining_belowdot as u32);
    #[doc(alias = "GDK_KEY_combining_grave")]
    pub const combining_grave: Self = Self(ffi::GDK_KEY_combining_grave as u32);
    #[doc(alias = "GDK_KEY_combining_hook")]
    pub const combining_hook: Self = Self(ffi::GDK_KEY_combining_hook as u32);
    #[doc(alias = "GDK_KEY_combining_tilde")]
    pub const combining_tilde: Self = Self(ffi::GDK_KEY_combining_tilde as u32);
    #[doc(alias = "GDK_KEY_comma")]
    pub const comma: Self = Self(ffi::GDK_KEY_comma as u32);
    #[doc(alias = "GDK_KEY_containsas")]
    pub const containsas: Self = Self(ffi::GDK_KEY_containsas as u32);
    #[doc(alias = "GDK_KEY_copyright")]
    pub const copyright: Self = Self(ffi::GDK_KEY_copyright as u32);
    #[doc(alias = "GDK_KEY_cr")]
    pub const cr: Self = Self(ffi::GDK_KEY_cr as u32);
    #[doc(alias = "GDK_KEY_crossinglines")]
    pub const crossinglines: Self = Self(ffi::GDK_KEY_crossinglines as u32);
    #[doc(alias = "GDK_KEY_cuberoot")]
    pub const cuberoot: Self = Self(ffi::GDK_KEY_cuberoot as u32);
    #[doc(alias = "GDK_KEY_currency")]
    pub const currency: Self = Self(ffi::GDK_KEY_currency as u32);
    #[doc(alias = "GDK_KEY_cursor")]
    pub const cursor: Self = Self(ffi::GDK_KEY_cursor as u32);
    #[doc(alias = "GDK_KEY_d")]
    pub const d: Self = Self(ffi::GDK_KEY_d as u32);
    #[doc(alias = "GDK_KEY_dabovedot")]
    pub const dabovedot: Self = Self(ffi::GDK_KEY_dabovedot as u32);
    #[doc(alias = "GDK_KEY_dagger")]
    pub const dagger: Self = Self(ffi::GDK_KEY_dagger as u32);
    #[doc(alias = "GDK_KEY_dcaron")]
    pub const dcaron: Self = Self(ffi::GDK_KEY_dcaron as u32);
    #[doc(alias = "GDK_KEY_dead_A")]
    pub const dead_A: Self = Self(ffi::GDK_KEY_dead_A as u32);
    #[doc(alias = "GDK_KEY_dead_E")]
    pub const dead_E: Self = Self(ffi::GDK_KEY_dead_E as u32);
    #[doc(alias = "GDK_KEY_dead_I")]
    pub const dead_I: Self = Self(ffi::GDK_KEY_dead_I as u32);
    #[doc(alias = "GDK_KEY_dead_O")]
    pub const dead_O: Self = Self(ffi::GDK_KEY_dead_O as u32);
    #[doc(alias = "GDK_KEY_dead_SCHWA")]
    pub const dead_SCHWA: Self = Self(ffi::GDK_KEY_dead_SCHWA as u32);
    #[doc(alias = "GDK_KEY_dead_U")]
    pub const dead_U: Self = Self(ffi::GDK_KEY_dead_U as u32);
    #[doc(alias = "GDK_KEY_dead_a")]
    pub const dead_a: Self = Self(ffi::GDK_KEY_dead_a as u32);
    #[doc(alias = "GDK_KEY_dead_abovecomma")]
    pub const dead_abovecomma: Self = Self(ffi::GDK_KEY_dead_abovecomma as u32);
    #[doc(alias = "GDK_KEY_dead_abovedot")]
    pub const dead_abovedot: Self = Self(ffi::GDK_KEY_dead_abovedot as u32);
    #[doc(alias = "GDK_KEY_dead_abovereversedcomma")]
    pub const dead_abovereversedcomma: Self = Self(ffi::GDK_KEY_dead_abovereversedcomma as u32);
    #[doc(alias = "GDK_KEY_dead_abovering")]
    pub const dead_abovering: Self = Self(ffi::GDK_KEY_dead_abovering as u32);
    #[doc(alias = "GDK_KEY_dead_aboveverticalline")]
    pub const dead_aboveverticalline: Self = Self(ffi::GDK_KEY_dead_aboveverticalline as u32);
    #[doc(alias = "GDK_KEY_dead_acute")]
    pub const dead_acute: Self = Self(ffi::GDK_KEY_dead_acute as u32);
    #[doc(alias = "GDK_KEY_dead_belowbreve")]
    pub const dead_belowbreve: Self = Self(ffi::GDK_KEY_dead_belowbreve as u32);
    #[doc(alias = "GDK_KEY_dead_belowcircumflex")]
    pub const dead_belowcircumflex: Self = Self(ffi::GDK_KEY_dead_belowcircumflex as u32);
    #[doc(alias = "GDK_KEY_dead_belowcomma")]
    pub const dead_belowcomma: Self = Self(ffi::GDK_KEY_dead_belowcomma as u32);
    #[doc(alias = "GDK_KEY_dead_belowdiaeresis")]
    pub const dead_belowdiaeresis: Self = Self(ffi::GDK_KEY_dead_belowdiaeresis as u32);
    #[doc(alias = "GDK_KEY_dead_belowdot")]
    pub const dead_belowdot: Self = Self(ffi::GDK_KEY_dead_belowdot as u32);
    #[doc(alias = "GDK_KEY_dead_belowmacron")]
    pub const dead_belowmacron: Self = Self(ffi::GDK_KEY_dead_belowmacron as u32);
    #[doc(alias = "GDK_KEY_dead_belowring")]
    pub const dead_belowring: Self = Self(ffi::GDK_KEY_dead_belowring as u32);
    #[doc(alias = "GDK_KEY_dead_belowtilde")]
    pub const dead_belowtilde: Self = Self(ffi::GDK_KEY_dead_belowtilde as u32);
    #[doc(alias = "GDK_KEY_dead_belowverticalline")]
    pub const dead_belowverticalline: Self = Self(ffi::GDK_KEY_dead_belowverticalline as u32);
    #[doc(alias = "GDK_KEY_dead_breve")]
    pub const dead_breve: Self = Self(ffi::GDK_KEY_dead_breve as u32);
    #[doc(alias = "GDK_KEY_dead_capital_schwa")]
    pub const dead_capital_schwa: Self = Self(ffi::GDK_KEY_dead_capital_schwa as u32);
    #[doc(alias = "GDK_KEY_dead_caron")]
    pub const dead_caron: Self = Self(ffi::GDK_KEY_dead_caron as u32);
    #[doc(alias = "GDK_KEY_dead_cedilla")]
    pub const dead_cedilla: Self = Self(ffi::GDK_KEY_dead_cedilla as u32);
    #[doc(alias = "GDK_KEY_dead_circumflex")]
    pub const dead_circumflex: Self = Self(ffi::GDK_KEY_dead_circumflex as u32);
    #[doc(alias = "GDK_KEY_dead_currency")]
    pub const dead_currency: Self = Self(ffi::GDK_KEY_dead_currency as u32);
    #[doc(alias = "GDK_KEY_dead_dasia")]
    pub const dead_dasia: Self = Self(ffi::GDK_KEY_dead_dasia as u32);
    #[doc(alias = "GDK_KEY_dead_diaeresis")]
    pub const dead_diaeresis: Self = Self(ffi::GDK_KEY_dead_diaeresis as u32);
    #[doc(alias = "GDK_KEY_dead_doubleacute")]
    pub const dead_doubleacute: Self = Self(ffi::GDK_KEY_dead_doubleacute as u32);
    #[doc(alias = "GDK_KEY_dead_doublegrave")]
    pub const dead_doublegrave: Self = Self(ffi::GDK_KEY_dead_doublegrave as u32);
    #[doc(alias = "GDK_KEY_dead_e")]
    pub const dead_e: Self = Self(ffi::GDK_KEY_dead_e as u32);
    #[doc(alias = "GDK_KEY_dead_grave")]
    pub const dead_grave: Self = Self(ffi::GDK_KEY_dead_grave as u32);
    #[doc(alias = "GDK_KEY_dead_greek")]
    pub const dead_greek: Self = Self(ffi::GDK_KEY_dead_greek as u32);
    #[doc(alias = "GDK_KEY_dead_hamza")]
    pub const dead_hamza: Self = Self(ffi::GDK_KEY_dead_hamza as u32);
    #[doc(alias = "GDK_KEY_dead_hook")]
    pub const dead_hook: Self = Self(ffi::GDK_KEY_dead_hook as u32);
    #[doc(alias = "GDK_KEY_dead_horn")]
    pub const dead_horn: Self = Self(ffi::GDK_KEY_dead_horn as u32);
    #[doc(alias = "GDK_KEY_dead_i")]
    pub const dead_i: Self = Self(ffi::GDK_KEY_dead_i as u32);
    #[doc(alias = "GDK_KEY_dead_invertedbreve")]
    pub const dead_invertedbreve: Self = Self(ffi::GDK_KEY_dead_invertedbreve as u32);
    #[doc(alias = "GDK_KEY_dead_iota")]
    pub const dead_iota: Self = Self(ffi::GDK_KEY_dead_iota as u32);
    #[doc(alias = "GDK_KEY_dead_longsolidusoverlay")]
    pub const dead_longsolidusoverlay: Self = Self(ffi::GDK_KEY_dead_longsolidusoverlay as u32);
    #[doc(alias = "GDK_KEY_dead_lowline")]
    pub const dead_lowline: Self = Self(ffi::GDK_KEY_dead_lowline as u32);
    #[doc(alias = "GDK_KEY_dead_macron")]
    pub const dead_macron: Self = Self(ffi::GDK_KEY_dead_macron as u32);
    #[doc(alias = "GDK_KEY_dead_o")]
    pub const dead_o: Self = Self(ffi::GDK_KEY_dead_o as u32);
    #[doc(alias = "GDK_KEY_dead_ogonek")]
    pub const dead_ogonek: Self = Self(ffi::GDK_KEY_dead_ogonek as u32);
    #[doc(alias = "GDK_KEY_dead_perispomeni")]
    pub const dead_perispomeni: Self = Self(ffi::GDK_KEY_dead_perispomeni as u32);
    #[doc(alias = "GDK_KEY_dead_psili")]
    pub const dead_psili: Self = Self(ffi::GDK_KEY_dead_psili as u32);
    #[doc(alias = "GDK_KEY_dead_schwa")]
    pub const dead_schwa: Self = Self(ffi::GDK_KEY_dead_schwa as u32);
    #[doc(alias = "GDK_KEY_dead_semivoiced_sound")]
    pub const dead_semivoiced_sound: Self = Self(ffi::GDK_KEY_dead_semivoiced_sound as u32);
    #[doc(alias = "GDK_KEY_dead_small_schwa")]
    pub const dead_small_schwa: Self = Self(ffi::GDK_KEY_dead_small_schwa as u32);
    #[doc(alias = "GDK_KEY_dead_stroke")]
    pub const dead_stroke: Self = Self(ffi::GDK_KEY_dead_stroke as u32);
    #[doc(alias = "GDK_KEY_dead_tilde")]
    pub const dead_tilde: Self = Self(ffi::GDK_KEY_dead_tilde as u32);
    #[doc(alias = "GDK_KEY_dead_u")]
    pub const dead_u: Self = Self(ffi::GDK_KEY_dead_u as u32);
    #[doc(alias = "GDK_KEY_dead_voiced_sound")]
    pub const dead_voiced_sound: Self = Self(ffi::GDK_KEY_dead_voiced_sound as u32);
    #[doc(alias = "GDK_KEY_decimalpoint")]
    pub const decimalpoint: Self = Self(ffi::GDK_KEY_decimalpoint as u32);
    #[doc(alias = "GDK_KEY_degree")]
    pub const degree: Self = Self(ffi::GDK_KEY_degree as u32);
    #[doc(alias = "GDK_KEY_diaeresis")]
    pub const diaeresis: Self = Self(ffi::GDK_KEY_diaeresis as u32);
    #[doc(alias = "GDK_KEY_diamond")]
    pub const diamond: Self = Self(ffi::GDK_KEY_diamond as u32);
    #[doc(alias = "GDK_KEY_digitspace")]
    pub const digitspace: Self = Self(ffi::GDK_KEY_digitspace as u32);
    #[doc(alias = "GDK_KEY_dintegral")]
    pub const dintegral: Self = Self(ffi::GDK_KEY_dintegral as u32);
    #[doc(alias = "GDK_KEY_division")]
    pub const division: Self = Self(ffi::GDK_KEY_division as u32);
    #[doc(alias = "GDK_KEY_dollar")]
    pub const dollar: Self = Self(ffi::GDK_KEY_dollar as u32);
    #[doc(alias = "GDK_KEY_doubbaselinedot")]
    pub const doubbaselinedot: Self = Self(ffi::GDK_KEY_doubbaselinedot as u32);
    #[doc(alias = "GDK_KEY_doubleacute")]
    pub const doubleacute: Self = Self(ffi::GDK_KEY_doubleacute as u32);
    #[doc(alias = "GDK_KEY_doubledagger")]
    pub const doubledagger: Self = Self(ffi::GDK_KEY_doubledagger as u32);
    #[doc(alias = "GDK_KEY_doublelowquotemark")]
    pub const doublelowquotemark: Self = Self(ffi::GDK_KEY_doublelowquotemark as u32);
    #[doc(alias = "GDK_KEY_downarrow")]
    pub const downarrow: Self = Self(ffi::GDK_KEY_downarrow as u32);
    #[doc(alias = "GDK_KEY_downcaret")]
    pub const downcaret: Self = Self(ffi::GDK_KEY_downcaret as u32);
    #[doc(alias = "GDK_KEY_downshoe")]
    pub const downshoe: Self = Self(ffi::GDK_KEY_downshoe as u32);
    #[doc(alias = "GDK_KEY_downstile")]
    pub const downstile: Self = Self(ffi::GDK_KEY_downstile as u32);
    #[doc(alias = "GDK_KEY_downtack")]
    pub const downtack: Self = Self(ffi::GDK_KEY_downtack as u32);
    #[doc(alias = "GDK_KEY_dstroke")]
    pub const dstroke: Self = Self(ffi::GDK_KEY_dstroke as u32);
    #[doc(alias = "GDK_KEY_e")]
    pub const e: Self = Self(ffi::GDK_KEY_e as u32);
    #[doc(alias = "GDK_KEY_eabovedot")]
    pub const eabovedot: Self = Self(ffi::GDK_KEY_eabovedot as u32);
    #[doc(alias = "GDK_KEY_eacute")]
    pub const eacute: Self = Self(ffi::GDK_KEY_eacute as u32);
    #[doc(alias = "GDK_KEY_ebelowdot")]
    pub const ebelowdot: Self = Self(ffi::GDK_KEY_ebelowdot as u32);
    #[doc(alias = "GDK_KEY_ecaron")]
    pub const ecaron: Self = Self(ffi::GDK_KEY_ecaron as u32);
    #[doc(alias = "GDK_KEY_ecircumflex")]
    pub const ecircumflex: Self = Self(ffi::GDK_KEY_ecircumflex as u32);
    #[doc(alias = "GDK_KEY_ecircumflexacute")]
    pub const ecircumflexacute: Self = Self(ffi::GDK_KEY_ecircumflexacute as u32);
    #[doc(alias = "GDK_KEY_ecircumflexbelowdot")]
    pub const ecircumflexbelowdot: Self = Self(ffi::GDK_KEY_ecircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_ecircumflexgrave")]
    pub const ecircumflexgrave: Self = Self(ffi::GDK_KEY_ecircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_ecircumflexhook")]
    pub const ecircumflexhook: Self = Self(ffi::GDK_KEY_ecircumflexhook as u32);
    #[doc(alias = "GDK_KEY_ecircumflextilde")]
    pub const ecircumflextilde: Self = Self(ffi::GDK_KEY_ecircumflextilde as u32);
    #[doc(alias = "GDK_KEY_ediaeresis")]
    pub const ediaeresis: Self = Self(ffi::GDK_KEY_ediaeresis as u32);
    #[doc(alias = "GDK_KEY_egrave")]
    pub const egrave: Self = Self(ffi::GDK_KEY_egrave as u32);
    #[doc(alias = "GDK_KEY_ehook")]
    pub const ehook: Self = Self(ffi::GDK_KEY_ehook as u32);
    #[doc(alias = "GDK_KEY_eightsubscript")]
    pub const eightsubscript: Self = Self(ffi::GDK_KEY_eightsubscript as u32);
    #[doc(alias = "GDK_KEY_eightsuperior")]
    pub const eightsuperior: Self = Self(ffi::GDK_KEY_eightsuperior as u32);
    #[doc(alias = "GDK_KEY_elementof")]
    pub const elementof: Self = Self(ffi::GDK_KEY_elementof as u32);
    #[doc(alias = "GDK_KEY_ellipsis")]
    pub const ellipsis: Self = Self(ffi::GDK_KEY_ellipsis as u32);
    #[doc(alias = "GDK_KEY_em3space")]
    pub const em3space: Self = Self(ffi::GDK_KEY_em3space as u32);
    #[doc(alias = "GDK_KEY_em4space")]
    pub const em4space: Self = Self(ffi::GDK_KEY_em4space as u32);
    #[doc(alias = "GDK_KEY_emacron")]
    pub const emacron: Self = Self(ffi::GDK_KEY_emacron as u32);
    #[doc(alias = "GDK_KEY_emdash")]
    pub const emdash: Self = Self(ffi::GDK_KEY_emdash as u32);
    #[doc(alias = "GDK_KEY_emfilledcircle")]
    pub const emfilledcircle: Self = Self(ffi::GDK_KEY_emfilledcircle as u32);
    #[doc(alias = "GDK_KEY_emfilledrect")]
    pub const emfilledrect: Self = Self(ffi::GDK_KEY_emfilledrect as u32);
    #[doc(alias = "GDK_KEY_emopencircle")]
    pub const emopencircle: Self = Self(ffi::GDK_KEY_emopencircle as u32);
    #[doc(alias = "GDK_KEY_emopenrectangle")]
    pub const emopenrectangle: Self = Self(ffi::GDK_KEY_emopenrectangle as u32);
    #[doc(alias = "GDK_KEY_emptyset")]
    pub const emptyset: Self = Self(ffi::GDK_KEY_emptyset as u32);
    #[doc(alias = "GDK_KEY_emspace")]
    pub const emspace: Self = Self(ffi::GDK_KEY_emspace as u32);
    #[doc(alias = "GDK_KEY_endash")]
    pub const endash: Self = Self(ffi::GDK_KEY_endash as u32);
    #[doc(alias = "GDK_KEY_enfilledcircbullet")]
    pub const enfilledcircbullet: Self = Self(ffi::GDK_KEY_enfilledcircbullet as u32);
    #[doc(alias = "GDK_KEY_enfilledsqbullet")]
    pub const enfilledsqbullet: Self = Self(ffi::GDK_KEY_enfilledsqbullet as u32);
    #[doc(alias = "GDK_KEY_eng")]
    pub const eng: Self = Self(ffi::GDK_KEY_eng as u32);
    #[doc(alias = "GDK_KEY_enopencircbullet")]
    pub const enopencircbullet: Self = Self(ffi::GDK_KEY_enopencircbullet as u32);
    #[doc(alias = "GDK_KEY_enopensquarebullet")]
    pub const enopensquarebullet: Self = Self(ffi::GDK_KEY_enopensquarebullet as u32);
    #[doc(alias = "GDK_KEY_enspace")]
    pub const enspace: Self = Self(ffi::GDK_KEY_enspace as u32);
    #[doc(alias = "GDK_KEY_eogonek")]
    pub const eogonek: Self = Self(ffi::GDK_KEY_eogonek as u32);
    #[doc(alias = "GDK_KEY_equal")]
    pub const equal: Self = Self(ffi::GDK_KEY_equal as u32);
    #[doc(alias = "GDK_KEY_eth")]
    pub const eth: Self = Self(ffi::GDK_KEY_eth as u32);
    #[doc(alias = "GDK_KEY_etilde")]
    pub const etilde: Self = Self(ffi::GDK_KEY_etilde as u32);
    #[doc(alias = "GDK_KEY_exclam")]
    pub const exclam: Self = Self(ffi::GDK_KEY_exclam as u32);
    #[doc(alias = "GDK_KEY_exclamdown")]
    pub const exclamdown: Self = Self(ffi::GDK_KEY_exclamdown as u32);
    #[doc(alias = "GDK_KEY_ezh")]
    pub const ezh: Self = Self(ffi::GDK_KEY_ezh as u32);
    #[doc(alias = "GDK_KEY_f")]
    pub const f: Self = Self(ffi::GDK_KEY_f as u32);
    #[doc(alias = "GDK_KEY_fabovedot")]
    pub const fabovedot: Self = Self(ffi::GDK_KEY_fabovedot as u32);
    #[doc(alias = "GDK_KEY_femalesymbol")]
    pub const femalesymbol: Self = Self(ffi::GDK_KEY_femalesymbol as u32);
    #[doc(alias = "GDK_KEY_ff")]
    pub const ff: Self = Self(ffi::GDK_KEY_ff as u32);
    #[doc(alias = "GDK_KEY_figdash")]
    pub const figdash: Self = Self(ffi::GDK_KEY_figdash as u32);
    #[doc(alias = "GDK_KEY_filledlefttribullet")]
    pub const filledlefttribullet: Self = Self(ffi::GDK_KEY_filledlefttribullet as u32);
    #[doc(alias = "GDK_KEY_filledrectbullet")]
    pub const filledrectbullet: Self = Self(ffi::GDK_KEY_filledrectbullet as u32);
    #[doc(alias = "GDK_KEY_filledrighttribullet")]
    pub const filledrighttribullet: Self = Self(ffi::GDK_KEY_filledrighttribullet as u32);
    #[doc(alias = "GDK_KEY_filledtribulletdown")]
    pub const filledtribulletdown: Self = Self(ffi::GDK_KEY_filledtribulletdown as u32);
    #[doc(alias = "GDK_KEY_filledtribulletup")]
    pub const filledtribulletup: Self = Self(ffi::GDK_KEY_filledtribulletup as u32);
    #[doc(alias = "GDK_KEY_fiveeighths")]
    pub const fiveeighths: Self = Self(ffi::GDK_KEY_fiveeighths as u32);
    #[doc(alias = "GDK_KEY_fivesixths")]
    pub const fivesixths: Self = Self(ffi::GDK_KEY_fivesixths as u32);
    #[doc(alias = "GDK_KEY_fivesubscript")]
    pub const fivesubscript: Self = Self(ffi::GDK_KEY_fivesubscript as u32);
    #[doc(alias = "GDK_KEY_fivesuperior")]
    pub const fivesuperior: Self = Self(ffi::GDK_KEY_fivesuperior as u32);
    #[doc(alias = "GDK_KEY_fourfifths")]
    pub const fourfifths: Self = Self(ffi::GDK_KEY_fourfifths as u32);
    #[doc(alias = "GDK_KEY_foursubscript")]
    pub const foursubscript: Self = Self(ffi::GDK_KEY_foursubscript as u32);
    #[doc(alias = "GDK_KEY_foursuperior")]
    pub const foursuperior: Self = Self(ffi::GDK_KEY_foursuperior as u32);
    #[doc(alias = "GDK_KEY_fourthroot")]
    pub const fourthroot: Self = Self(ffi::GDK_KEY_fourthroot as u32);
    #[doc(alias = "GDK_KEY_function")]
    pub const function: Self = Self(ffi::GDK_KEY_function as u32);
    #[doc(alias = "GDK_KEY_g")]
    pub const g: Self = Self(ffi::GDK_KEY_g as u32);
    #[doc(alias = "GDK_KEY_gabovedot")]
    pub const gabovedot: Self = Self(ffi::GDK_KEY_gabovedot as u32);
    #[doc(alias = "GDK_KEY_gbreve")]
    pub const gbreve: Self = Self(ffi::GDK_KEY_gbreve as u32);
    #[doc(alias = "GDK_KEY_gcaron")]
    pub const gcaron: Self = Self(ffi::GDK_KEY_gcaron as u32);
    #[doc(alias = "GDK_KEY_gcedilla")]
    pub const gcedilla: Self = Self(ffi::GDK_KEY_gcedilla as u32);
    #[doc(alias = "GDK_KEY_gcircumflex")]
    pub const gcircumflex: Self = Self(ffi::GDK_KEY_gcircumflex as u32);
    #[doc(alias = "GDK_KEY_grave")]
    pub const grave: Self = Self(ffi::GDK_KEY_grave as u32);
    #[doc(alias = "GDK_KEY_greater")]
    pub const greater: Self = Self(ffi::GDK_KEY_greater as u32);
    #[doc(alias = "GDK_KEY_greaterthanequal")]
    pub const greaterthanequal: Self = Self(ffi::GDK_KEY_greaterthanequal as u32);
    #[doc(alias = "GDK_KEY_guillemetleft")]
    pub const guillemetleft: Self = Self(ffi::GDK_KEY_guillemetleft as u32);
    #[doc(alias = "GDK_KEY_guillemetright")]
    pub const guillemetright: Self = Self(ffi::GDK_KEY_guillemetright as u32);
    #[doc(alias = "GDK_KEY_guillemotleft")]
    pub const guillemotleft: Self = Self(ffi::GDK_KEY_guillemotleft as u32);
    #[doc(alias = "GDK_KEY_guillemotright")]
    pub const guillemotright: Self = Self(ffi::GDK_KEY_guillemotright as u32);
    #[doc(alias = "GDK_KEY_h")]
    pub const h: Self = Self(ffi::GDK_KEY_h as u32);
    #[doc(alias = "GDK_KEY_hairspace")]
    pub const hairspace: Self = Self(ffi::GDK_KEY_hairspace as u32);
    #[doc(alias = "GDK_KEY_hcircumflex")]
    pub const hcircumflex: Self = Self(ffi::GDK_KEY_hcircumflex as u32);
    #[doc(alias = "GDK_KEY_heart")]
    pub const heart: Self = Self(ffi::GDK_KEY_heart as u32);
    #[doc(alias = "GDK_KEY_hebrew_aleph")]
    pub const hebrew_aleph: Self = Self(ffi::GDK_KEY_hebrew_aleph as u32);
    #[doc(alias = "GDK_KEY_hebrew_ayin")]
    pub const hebrew_ayin: Self = Self(ffi::GDK_KEY_hebrew_ayin as u32);
    #[doc(alias = "GDK_KEY_hebrew_bet")]
    pub const hebrew_bet: Self = Self(ffi::GDK_KEY_hebrew_bet as u32);
    #[doc(alias = "GDK_KEY_hebrew_beth")]
    pub const hebrew_beth: Self = Self(ffi::GDK_KEY_hebrew_beth as u32);
    #[doc(alias = "GDK_KEY_hebrew_chet")]
    pub const hebrew_chet: Self = Self(ffi::GDK_KEY_hebrew_chet as u32);
    #[doc(alias = "GDK_KEY_hebrew_dalet")]
    pub const hebrew_dalet: Self = Self(ffi::GDK_KEY_hebrew_dalet as u32);
    #[doc(alias = "GDK_KEY_hebrew_daleth")]
    pub const hebrew_daleth: Self = Self(ffi::GDK_KEY_hebrew_daleth as u32);
    #[doc(alias = "GDK_KEY_hebrew_doublelowline")]
    pub const hebrew_doublelowline: Self = Self(ffi::GDK_KEY_hebrew_doublelowline as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalkaph")]
    pub const hebrew_finalkaph: Self = Self(ffi::GDK_KEY_hebrew_finalkaph as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalmem")]
    pub const hebrew_finalmem: Self = Self(ffi::GDK_KEY_hebrew_finalmem as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalnun")]
    pub const hebrew_finalnun: Self = Self(ffi::GDK_KEY_hebrew_finalnun as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalpe")]
    pub const hebrew_finalpe: Self = Self(ffi::GDK_KEY_hebrew_finalpe as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalzade")]
    pub const hebrew_finalzade: Self = Self(ffi::GDK_KEY_hebrew_finalzade as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalzadi")]
    pub const hebrew_finalzadi: Self = Self(ffi::GDK_KEY_hebrew_finalzadi as u32);
    #[doc(alias = "GDK_KEY_hebrew_gimel")]
    pub const hebrew_gimel: Self = Self(ffi::GDK_KEY_hebrew_gimel as u32);
    #[doc(alias = "GDK_KEY_hebrew_gimmel")]
    pub const hebrew_gimmel: Self = Self(ffi::GDK_KEY_hebrew_gimmel as u32);
    #[doc(alias = "GDK_KEY_hebrew_he")]
    pub const hebrew_he: Self = Self(ffi::GDK_KEY_hebrew_he as u32);
    #[doc(alias = "GDK_KEY_hebrew_het")]
    pub const hebrew_het: Self = Self(ffi::GDK_KEY_hebrew_het as u32);
    #[doc(alias = "GDK_KEY_hebrew_kaph")]
    pub const hebrew_kaph: Self = Self(ffi::GDK_KEY_hebrew_kaph as u32);
    #[doc(alias = "GDK_KEY_hebrew_kuf")]
    pub const hebrew_kuf: Self = Self(ffi::GDK_KEY_hebrew_kuf as u32);
    #[doc(alias = "GDK_KEY_hebrew_lamed")]
    pub const hebrew_lamed: Self = Self(ffi::GDK_KEY_hebrew_lamed as u32);
    #[doc(alias = "GDK_KEY_hebrew_mem")]
    pub const hebrew_mem: Self = Self(ffi::GDK_KEY_hebrew_mem as u32);
    #[doc(alias = "GDK_KEY_hebrew_nun")]
    pub const hebrew_nun: Self = Self(ffi::GDK_KEY_hebrew_nun as u32);
    #[doc(alias = "GDK_KEY_hebrew_pe")]
    pub const hebrew_pe: Self = Self(ffi::GDK_KEY_hebrew_pe as u32);
    #[doc(alias = "GDK_KEY_hebrew_qoph")]
    pub const hebrew_qoph: Self = Self(ffi::GDK_KEY_hebrew_qoph as u32);
    #[doc(alias = "GDK_KEY_hebrew_resh")]
    pub const hebrew_resh: Self = Self(ffi::GDK_KEY_hebrew_resh as u32);
    #[doc(alias = "GDK_KEY_hebrew_samech")]
    pub const hebrew_samech: Self = Self(ffi::GDK_KEY_hebrew_samech as u32);
    #[doc(alias = "GDK_KEY_hebrew_samekh")]
    pub const hebrew_samekh: Self = Self(ffi::GDK_KEY_hebrew_samekh as u32);
    #[doc(alias = "GDK_KEY_hebrew_shin")]
    pub const hebrew_shin: Self = Self(ffi::GDK_KEY_hebrew_shin as u32);
    #[doc(alias = "GDK_KEY_hebrew_taf")]
    pub const hebrew_taf: Self = Self(ffi::GDK_KEY_hebrew_taf as u32);
    #[doc(alias = "GDK_KEY_hebrew_taw")]
    pub const hebrew_taw: Self = Self(ffi::GDK_KEY_hebrew_taw as u32);
    #[doc(alias = "GDK_KEY_hebrew_tet")]
    pub const hebrew_tet: Self = Self(ffi::GDK_KEY_hebrew_tet as u32);
    #[doc(alias = "GDK_KEY_hebrew_teth")]
    pub const hebrew_teth: Self = Self(ffi::GDK_KEY_hebrew_teth as u32);
    #[doc(alias = "GDK_KEY_hebrew_waw")]
    pub const hebrew_waw: Self = Self(ffi::GDK_KEY_hebrew_waw as u32);
    #[doc(alias = "GDK_KEY_hebrew_yod")]
    pub const hebrew_yod: Self = Self(ffi::GDK_KEY_hebrew_yod as u32);
    #[doc(alias = "GDK_KEY_hebrew_zade")]
    pub const hebrew_zade: Self = Self(ffi::GDK_KEY_hebrew_zade as u32);
    #[doc(alias = "GDK_KEY_hebrew_zadi")]
    pub const hebrew_zadi: Self = Self(ffi::GDK_KEY_hebrew_zadi as u32);
    #[doc(alias = "GDK_KEY_hebrew_zain")]
    pub const hebrew_zain: Self = Self(ffi::GDK_KEY_hebrew_zain as u32);
    #[doc(alias = "GDK_KEY_hebrew_zayin")]
    pub const hebrew_zayin: Self = Self(ffi::GDK_KEY_hebrew_zayin as u32);
    #[doc(alias = "GDK_KEY_hexagram")]
    pub const hexagram: Self = Self(ffi::GDK_KEY_hexagram as u32);
    #[doc(alias = "GDK_KEY_horizconnector")]
    pub const horizconnector: Self = Self(ffi::GDK_KEY_horizconnector as u32);
    #[doc(alias = "GDK_KEY_horizlinescan1")]
    pub const horizlinescan1: Self = Self(ffi::GDK_KEY_horizlinescan1 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan3")]
    pub const horizlinescan3: Self = Self(ffi::GDK_KEY_horizlinescan3 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan5")]
    pub const horizlinescan5: Self = Self(ffi::GDK_KEY_horizlinescan5 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan7")]
    pub const horizlinescan7: Self = Self(ffi::GDK_KEY_horizlinescan7 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan9")]
    pub const horizlinescan9: Self = Self(ffi::GDK_KEY_horizlinescan9 as u32);
    #[doc(alias = "GDK_KEY_hstroke")]
    pub const hstroke: Self = Self(ffi::GDK_KEY_hstroke as u32);
    #[doc(alias = "GDK_KEY_ht")]
    pub const ht: Self = Self(ffi::GDK_KEY_ht as u32);
    #[doc(alias = "GDK_KEY_hyphen")]
    pub const hyphen: Self = Self(ffi::GDK_KEY_hyphen as u32);
    #[doc(alias = "GDK_KEY_i")]
    pub const i: Self = Self(ffi::GDK_KEY_i as u32);
    #[doc(alias = "GDK_KEY_iTouch")]
    pub const iTouch: Self = Self(ffi::GDK_KEY_iTouch as u32);
    #[doc(alias = "GDK_KEY_iacute")]
    pub const iacute: Self = Self(ffi::GDK_KEY_iacute as u32);
    #[doc(alias = "GDK_KEY_ibelowdot")]
    pub const ibelowdot: Self = Self(ffi::GDK_KEY_ibelowdot as u32);
    #[doc(alias = "GDK_KEY_ibreve")]
    pub const ibreve: Self = Self(ffi::GDK_KEY_ibreve as u32);
    #[doc(alias = "GDK_KEY_icircumflex")]
    pub const icircumflex: Self = Self(ffi::GDK_KEY_icircumflex as u32);
    #[doc(alias = "GDK_KEY_identical")]
    pub const identical: Self = Self(ffi::GDK_KEY_identical as u32);
    #[doc(alias = "GDK_KEY_idiaeresis")]
    pub const idiaeresis: Self = Self(ffi::GDK_KEY_idiaeresis as u32);
    #[doc(alias = "GDK_KEY_idotless")]
    pub const idotless: Self = Self(ffi::GDK_KEY_idotless as u32);
    #[doc(alias = "GDK_KEY_ifonlyif")]
    pub const ifonlyif: Self = Self(ffi::GDK_KEY_ifonlyif as u32);
    #[doc(alias = "GDK_KEY_igrave")]
    pub const igrave: Self = Self(ffi::GDK_KEY_igrave as u32);
    #[doc(alias = "GDK_KEY_ihook")]
    pub const ihook: Self = Self(ffi::GDK_KEY_ihook as u32);
    #[doc(alias = "GDK_KEY_imacron")]
    pub const imacron: Self = Self(ffi::GDK_KEY_imacron as u32);
    #[doc(alias = "GDK_KEY_implies")]
    pub const implies: Self = Self(ffi::GDK_KEY_implies as u32);
    #[doc(alias = "GDK_KEY_includedin")]
    pub const includedin: Self = Self(ffi::GDK_KEY_includedin as u32);
    #[doc(alias = "GDK_KEY_includes")]
    pub const includes: Self = Self(ffi::GDK_KEY_includes as u32);
    #[doc(alias = "GDK_KEY_infinity")]
    pub const infinity: Self = Self(ffi::GDK_KEY_infinity as u32);
    #[doc(alias = "GDK_KEY_integral")]
    pub const integral: Self = Self(ffi::GDK_KEY_integral as u32);
    #[doc(alias = "GDK_KEY_intersection")]
    pub const intersection: Self = Self(ffi::GDK_KEY_intersection as u32);
    #[doc(alias = "GDK_KEY_iogonek")]
    pub const iogonek: Self = Self(ffi::GDK_KEY_iogonek as u32);
    #[doc(alias = "GDK_KEY_itilde")]
    pub const itilde: Self = Self(ffi::GDK_KEY_itilde as u32);
    #[doc(alias = "GDK_KEY_j")]
    pub const j: Self = Self(ffi::GDK_KEY_j as u32);
    #[doc(alias = "GDK_KEY_jcircumflex")]
    pub const jcircumflex: Self = Self(ffi::GDK_KEY_jcircumflex as u32);
    #[doc(alias = "GDK_KEY_jot")]
    pub const jot: Self = Self(ffi::GDK_KEY_jot as u32);
    #[doc(alias = "GDK_KEY_k")]
    pub const k: Self = Self(ffi::GDK_KEY_k as u32);
    #[doc(alias = "GDK_KEY_kana_A")]
    pub const kana_A: Self = Self(ffi::GDK_KEY_kana_A as u32);
    #[doc(alias = "GDK_KEY_kana_CHI")]
    pub const kana_CHI: Self = Self(ffi::GDK_KEY_kana_CHI as u32);
    #[doc(alias = "GDK_KEY_kana_E")]
    pub const kana_E: Self = Self(ffi::GDK_KEY_kana_E as u32);
    #[doc(alias = "GDK_KEY_kana_FU")]
    pub const kana_FU: Self = Self(ffi::GDK_KEY_kana_FU as u32);
    #[doc(alias = "GDK_KEY_kana_HA")]
    pub const kana_HA: Self = Self(ffi::GDK_KEY_kana_HA as u32);
    #[doc(alias = "GDK_KEY_kana_HE")]
    pub const kana_HE: Self = Self(ffi::GDK_KEY_kana_HE as u32);
    #[doc(alias = "GDK_KEY_kana_HI")]
    pub const kana_HI: Self = Self(ffi::GDK_KEY_kana_HI as u32);
    #[doc(alias = "GDK_KEY_kana_HO")]
    pub const kana_HO: Self = Self(ffi::GDK_KEY_kana_HO as u32);
    #[doc(alias = "GDK_KEY_kana_HU")]
    pub const kana_HU: Self = Self(ffi::GDK_KEY_kana_HU as u32);
    #[doc(alias = "GDK_KEY_kana_I")]
    pub const kana_I: Self = Self(ffi::GDK_KEY_kana_I as u32);
    #[doc(alias = "GDK_KEY_kana_KA")]
    pub const kana_KA: Self = Self(ffi::GDK_KEY_kana_KA as u32);
    #[doc(alias = "GDK_KEY_kana_KE")]
    pub const kana_KE: Self = Self(ffi::GDK_KEY_kana_KE as u32);
    #[doc(alias = "GDK_KEY_kana_KI")]
    pub const kana_KI: Self = Self(ffi::GDK_KEY_kana_KI as u32);
    #[doc(alias = "GDK_KEY_kana_KO")]
    pub const kana_KO: Self = Self(ffi::GDK_KEY_kana_KO as u32);
    #[doc(alias = "GDK_KEY_kana_KU")]
    pub const kana_KU: Self = Self(ffi::GDK_KEY_kana_KU as u32);
    #[doc(alias = "GDK_KEY_kana_MA")]
    pub const kana_MA: Self = Self(ffi::GDK_KEY_kana_MA as u32);
    #[doc(alias = "GDK_KEY_kana_ME")]
    pub const kana_ME: Self = Self(ffi::GDK_KEY_kana_ME as u32);
    #[doc(alias = "GDK_KEY_kana_MI")]
    pub const kana_MI: Self = Self(ffi::GDK_KEY_kana_MI as u32);
    #[doc(alias = "GDK_KEY_kana_MO")]
    pub const kana_MO: Self = Self(ffi::GDK_KEY_kana_MO as u32);
    #[doc(alias = "GDK_KEY_kana_MU")]
    pub const kana_MU: Self = Self(ffi::GDK_KEY_kana_MU as u32);
    #[doc(alias = "GDK_KEY_kana_N")]
    pub const kana_N: Self = Self(ffi::GDK_KEY_kana_N as u32);
    #[doc(alias = "GDK_KEY_kana_NA")]
    pub const kana_NA: Self = Self(ffi::GDK_KEY_kana_NA as u32);
    #[doc(alias = "GDK_KEY_kana_NE")]
    pub const kana_NE: Self = Self(ffi::GDK_KEY_kana_NE as u32);
    #[doc(alias = "GDK_KEY_kana_NI")]
    pub const kana_NI: Self = Self(ffi::GDK_KEY_kana_NI as u32);
    #[doc(alias = "GDK_KEY_kana_NO")]
    pub const kana_NO: Self = Self(ffi::GDK_KEY_kana_NO as u32);
    #[doc(alias = "GDK_KEY_kana_NU")]
    pub const kana_NU: Self = Self(ffi::GDK_KEY_kana_NU as u32);
    #[doc(alias = "GDK_KEY_kana_O")]
    pub const kana_O: Self = Self(ffi::GDK_KEY_kana_O as u32);
    #[doc(alias = "GDK_KEY_kana_RA")]
    pub const kana_RA: Self = Self(ffi::GDK_KEY_kana_RA as u32);
    #[doc(alias = "GDK_KEY_kana_RE")]
    pub const kana_RE: Self = Self(ffi::GDK_KEY_kana_RE as u32);
    #[doc(alias = "GDK_KEY_kana_RI")]
    pub const kana_RI: Self = Self(ffi::GDK_KEY_kana_RI as u32);
    #[doc(alias = "GDK_KEY_kana_RO")]
    pub const kana_RO: Self = Self(ffi::GDK_KEY_kana_RO as u32);
    #[doc(alias = "GDK_KEY_kana_RU")]
    pub const kana_RU: Self = Self(ffi::GDK_KEY_kana_RU as u32);
    #[doc(alias = "GDK_KEY_kana_SA")]
    pub const kana_SA: Self = Self(ffi::GDK_KEY_kana_SA as u32);
    #[doc(alias = "GDK_KEY_kana_SE")]
    pub const kana_SE: Self = Self(ffi::GDK_KEY_kana_SE as u32);
    #[doc(alias = "GDK_KEY_kana_SHI")]
    pub const kana_SHI: Self = Self(ffi::GDK_KEY_kana_SHI as u32);
    #[doc(alias = "GDK_KEY_kana_SO")]
    pub const kana_SO: Self = Self(ffi::GDK_KEY_kana_SO as u32);
    #[doc(alias = "GDK_KEY_kana_SU")]
    pub const kana_SU: Self = Self(ffi::GDK_KEY_kana_SU as u32);
    #[doc(alias = "GDK_KEY_kana_TA")]
    pub const kana_TA: Self = Self(ffi::GDK_KEY_kana_TA as u32);
    #[doc(alias = "GDK_KEY_kana_TE")]
    pub const kana_TE: Self = Self(ffi::GDK_KEY_kana_TE as u32);
    #[doc(alias = "GDK_KEY_kana_TI")]
    pub const kana_TI: Self = Self(ffi::GDK_KEY_kana_TI as u32);
    #[doc(alias = "GDK_KEY_kana_TO")]
    pub const kana_TO: Self = Self(ffi::GDK_KEY_kana_TO as u32);
    #[doc(alias = "GDK_KEY_kana_TSU")]
    pub const kana_TSU: Self = Self(ffi::GDK_KEY_kana_TSU as u32);
    #[doc(alias = "GDK_KEY_kana_TU")]
    pub const kana_TU: Self = Self(ffi::GDK_KEY_kana_TU as u32);
    #[doc(alias = "GDK_KEY_kana_U")]
    pub const kana_U: Self = Self(ffi::GDK_KEY_kana_U as u32);
    #[doc(alias = "GDK_KEY_kana_WA")]
    pub const kana_WA: Self = Self(ffi::GDK_KEY_kana_WA as u32);
    #[doc(alias = "GDK_KEY_kana_WO")]
    pub const kana_WO: Self = Self(ffi::GDK_KEY_kana_WO as u32);
    #[doc(alias = "GDK_KEY_kana_YA")]
    pub const kana_YA: Self = Self(ffi::GDK_KEY_kana_YA as u32);
    #[doc(alias = "GDK_KEY_kana_YO")]
    pub const kana_YO: Self = Self(ffi::GDK_KEY_kana_YO as u32);
    #[doc(alias = "GDK_KEY_kana_YU")]
    pub const kana_YU: Self = Self(ffi::GDK_KEY_kana_YU as u32);
    #[doc(alias = "GDK_KEY_kana_a")]
    pub const kana_a: Self = Self(ffi::GDK_KEY_kana_a as u32);
    #[doc(alias = "GDK_KEY_kana_closingbracket")]
    pub const kana_closingbracket: Self = Self(ffi::GDK_KEY_kana_closingbracket as u32);
    #[doc(alias = "GDK_KEY_kana_comma")]
    pub const kana_comma: Self = Self(ffi::GDK_KEY_kana_comma as u32);
    #[doc(alias = "GDK_KEY_kana_conjunctive")]
    pub const kana_conjunctive: Self = Self(ffi::GDK_KEY_kana_conjunctive as u32);
    #[doc(alias = "GDK_KEY_kana_e")]
    pub const kana_e: Self = Self(ffi::GDK_KEY_kana_e as u32);
    #[doc(alias = "GDK_KEY_kana_fullstop")]
    pub const kana_fullstop: Self = Self(ffi::GDK_KEY_kana_fullstop as u32);
    #[doc(alias = "GDK_KEY_kana_i")]
    pub const kana_i: Self = Self(ffi::GDK_KEY_kana_i as u32);
    #[doc(alias = "GDK_KEY_kana_middledot")]
    pub const kana_middledot: Self = Self(ffi::GDK_KEY_kana_middledot as u32);
    #[doc(alias = "GDK_KEY_kana_o")]
    pub const kana_o: Self = Self(ffi::GDK_KEY_kana_o as u32);
    #[doc(alias = "GDK_KEY_kana_openingbracket")]
    pub const kana_openingbracket: Self = Self(ffi::GDK_KEY_kana_openingbracket as u32);
    #[doc(alias = "GDK_KEY_kana_switch")]
    pub const kana_switch: Self = Self(ffi::GDK_KEY_kana_switch as u32);
    #[doc(alias = "GDK_KEY_kana_tsu")]
    pub const kana_tsu: Self = Self(ffi::GDK_KEY_kana_tsu as u32);
    #[doc(alias = "GDK_KEY_kana_tu")]
    pub const kana_tu: Self = Self(ffi::GDK_KEY_kana_tu as u32);
    #[doc(alias = "GDK_KEY_kana_u")]
    pub const kana_u: Self = Self(ffi::GDK_KEY_kana_u as u32);
    #[doc(alias = "GDK_KEY_kana_ya")]
    pub const kana_ya: Self = Self(ffi::GDK_KEY_kana_ya as u32);
    #[doc(alias = "GDK_KEY_kana_yo")]
    pub const kana_yo: Self = Self(ffi::GDK_KEY_kana_yo as u32);
    #[doc(alias = "GDK_KEY_kana_yu")]
    pub const kana_yu: Self = Self(ffi::GDK_KEY_kana_yu as u32);
    #[doc(alias = "GDK_KEY_kappa")]
    pub const kappa: Self = Self(ffi::GDK_KEY_kappa as u32);
    #[doc(alias = "GDK_KEY_kcedilla")]
    pub const kcedilla: Self = Self(ffi::GDK_KEY_kcedilla as u32);
    #[doc(alias = "GDK_KEY_kra")]
    pub const kra: Self = Self(ffi::GDK_KEY_kra as u32);
    #[doc(alias = "GDK_KEY_l")]
    pub const l: Self = Self(ffi::GDK_KEY_l as u32);
    #[doc(alias = "GDK_KEY_lacute")]
    pub const lacute: Self = Self(ffi::GDK_KEY_lacute as u32);
    #[doc(alias = "GDK_KEY_latincross")]
    pub const latincross: Self = Self(ffi::GDK_KEY_latincross as u32);
    #[doc(alias = "GDK_KEY_lbelowdot")]
    pub const lbelowdot: Self = Self(ffi::GDK_KEY_lbelowdot as u32);
    #[doc(alias = "GDK_KEY_lcaron")]
    pub const lcaron: Self = Self(ffi::GDK_KEY_lcaron as u32);
    #[doc(alias = "GDK_KEY_lcedilla")]
    pub const lcedilla: Self = Self(ffi::GDK_KEY_lcedilla as u32);
    #[doc(alias = "GDK_KEY_leftanglebracket")]
    pub const leftanglebracket: Self = Self(ffi::GDK_KEY_leftanglebracket as u32);
    #[doc(alias = "GDK_KEY_leftarrow")]
    pub const leftarrow: Self = Self(ffi::GDK_KEY_leftarrow as u32);
    #[doc(alias = "GDK_KEY_leftcaret")]
    pub const leftcaret: Self = Self(ffi::GDK_KEY_leftcaret as u32);
    #[doc(alias = "GDK_KEY_leftdoublequotemark")]
    pub const leftdoublequotemark: Self = Self(ffi::GDK_KEY_leftdoublequotemark as u32);
    #[doc(alias = "GDK_KEY_leftmiddlecurlybrace")]
    pub const leftmiddlecurlybrace: Self = Self(ffi::GDK_KEY_leftmiddlecurlybrace as u32);
    #[doc(alias = "GDK_KEY_leftopentriangle")]
    pub const leftopentriangle: Self = Self(ffi::GDK_KEY_leftopentriangle as u32);
    #[doc(alias = "GDK_KEY_leftpointer")]
    pub const leftpointer: Self = Self(ffi::GDK_KEY_leftpointer as u32);
    #[doc(alias = "GDK_KEY_leftradical")]
    pub const leftradical: Self = Self(ffi::GDK_KEY_leftradical as u32);
    #[doc(alias = "GDK_KEY_leftshoe")]
    pub const leftshoe: Self = Self(ffi::GDK_KEY_leftshoe as u32);
    #[doc(alias = "GDK_KEY_leftsinglequotemark")]
    pub const leftsinglequotemark: Self = Self(ffi::GDK_KEY_leftsinglequotemark as u32);
    #[doc(alias = "GDK_KEY_leftt")]
    pub const leftt: Self = Self(ffi::GDK_KEY_leftt as u32);
    #[doc(alias = "GDK_KEY_lefttack")]
    pub const lefttack: Self = Self(ffi::GDK_KEY_lefttack as u32);
    #[doc(alias = "GDK_KEY_less")]
    pub const less: Self = Self(ffi::GDK_KEY_less as u32);
    #[doc(alias = "GDK_KEY_lessthanequal")]
    pub const lessthanequal: Self = Self(ffi::GDK_KEY_lessthanequal as u32);
    #[doc(alias = "GDK_KEY_lf")]
    pub const lf: Self = Self(ffi::GDK_KEY_lf as u32);
    #[doc(alias = "GDK_KEY_logicaland")]
    pub const logicaland: Self = Self(ffi::GDK_KEY_logicaland as u32);
    #[doc(alias = "GDK_KEY_logicalor")]
    pub const logicalor: Self = Self(ffi::GDK_KEY_logicalor as u32);
    #[doc(alias = "GDK_KEY_lowleftcorner")]
    pub const lowleftcorner: Self = Self(ffi::GDK_KEY_lowleftcorner as u32);
    #[doc(alias = "GDK_KEY_lowrightcorner")]
    pub const lowrightcorner: Self = Self(ffi::GDK_KEY_lowrightcorner as u32);
    #[doc(alias = "GDK_KEY_lstroke")]
    pub const lstroke: Self = Self(ffi::GDK_KEY_lstroke as u32);
    #[doc(alias = "GDK_KEY_m")]
    pub const m: Self = Self(ffi::GDK_KEY_m as u32);
    #[doc(alias = "GDK_KEY_mabovedot")]
    pub const mabovedot: Self = Self(ffi::GDK_KEY_mabovedot as u32);
    #[doc(alias = "GDK_KEY_macron")]
    pub const macron: Self = Self(ffi::GDK_KEY_macron as u32);
    #[doc(alias = "GDK_KEY_malesymbol")]
    pub const malesymbol: Self = Self(ffi::GDK_KEY_malesymbol as u32);
    #[doc(alias = "GDK_KEY_maltesecross")]
    pub const maltesecross: Self = Self(ffi::GDK_KEY_maltesecross as u32);
    #[doc(alias = "GDK_KEY_marker")]
    pub const marker: Self = Self(ffi::GDK_KEY_marker as u32);
    #[doc(alias = "GDK_KEY_masculine")]
    pub const masculine: Self = Self(ffi::GDK_KEY_masculine as u32);
    #[doc(alias = "GDK_KEY_minus")]
    pub const minus: Self = Self(ffi::GDK_KEY_minus as u32);
    #[doc(alias = "GDK_KEY_minutes")]
    pub const minutes: Self = Self(ffi::GDK_KEY_minutes as u32);
    #[doc(alias = "GDK_KEY_mu")]
    pub const mu: Self = Self(ffi::GDK_KEY_mu as u32);
    #[doc(alias = "GDK_KEY_multiply")]
    pub const multiply: Self = Self(ffi::GDK_KEY_multiply as u32);
    #[doc(alias = "GDK_KEY_musicalflat")]
    pub const musicalflat: Self = Self(ffi::GDK_KEY_musicalflat as u32);
    #[doc(alias = "GDK_KEY_musicalsharp")]
    pub const musicalsharp: Self = Self(ffi::GDK_KEY_musicalsharp as u32);
    #[doc(alias = "GDK_KEY_n")]
    pub const n: Self = Self(ffi::GDK_KEY_n as u32);
    #[doc(alias = "GDK_KEY_nabla")]
    pub const nabla: Self = Self(ffi::GDK_KEY_nabla as u32);
    #[doc(alias = "GDK_KEY_nacute")]
    pub const nacute: Self = Self(ffi::GDK_KEY_nacute as u32);
    #[doc(alias = "GDK_KEY_ncaron")]
    pub const ncaron: Self = Self(ffi::GDK_KEY_ncaron as u32);
    #[doc(alias = "GDK_KEY_ncedilla")]
    pub const ncedilla: Self = Self(ffi::GDK_KEY_ncedilla as u32);
    #[doc(alias = "GDK_KEY_ninesubscript")]
    pub const ninesubscript: Self = Self(ffi::GDK_KEY_ninesubscript as u32);
    #[doc(alias = "GDK_KEY_ninesuperior")]
    pub const ninesuperior: Self = Self(ffi::GDK_KEY_ninesuperior as u32);
    #[doc(alias = "GDK_KEY_nl")]
    pub const nl: Self = Self(ffi::GDK_KEY_nl as u32);
    #[doc(alias = "GDK_KEY_nobreakspace")]
    pub const nobreakspace: Self = Self(ffi::GDK_KEY_nobreakspace as u32);
    #[doc(alias = "GDK_KEY_notapproxeq")]
    pub const notapproxeq: Self = Self(ffi::GDK_KEY_notapproxeq as u32);
    #[doc(alias = "GDK_KEY_notelementof")]
    pub const notelementof: Self = Self(ffi::GDK_KEY_notelementof as u32);
    #[doc(alias = "GDK_KEY_notequal")]
    pub const notequal: Self = Self(ffi::GDK_KEY_notequal as u32);
    #[doc(alias = "GDK_KEY_notidentical")]
    pub const notidentical: Self = Self(ffi::GDK_KEY_notidentical as u32);
    #[doc(alias = "GDK_KEY_notsign")]
    pub const notsign: Self = Self(ffi::GDK_KEY_notsign as u32);
    #[doc(alias = "GDK_KEY_ntilde")]
    pub const ntilde: Self = Self(ffi::GDK_KEY_ntilde as u32);
    #[doc(alias = "GDK_KEY_numbersign")]
    pub const numbersign: Self = Self(ffi::GDK_KEY_numbersign as u32);
    #[doc(alias = "GDK_KEY_numerosign")]
    pub const numerosign: Self = Self(ffi::GDK_KEY_numerosign as u32);
    #[doc(alias = "GDK_KEY_o")]
    pub const o: Self = Self(ffi::GDK_KEY_o as u32);
    #[doc(alias = "GDK_KEY_oacute")]
    pub const oacute: Self = Self(ffi::GDK_KEY_oacute as u32);
    #[doc(alias = "GDK_KEY_obarred")]
    pub const obarred: Self = Self(ffi::GDK_KEY_obarred as u32);
    #[doc(alias = "GDK_KEY_obelowdot")]
    pub const obelowdot: Self = Self(ffi::GDK_KEY_obelowdot as u32);
    #[doc(alias = "GDK_KEY_ocaron")]
    pub const ocaron: Self = Self(ffi::GDK_KEY_ocaron as u32);
    #[doc(alias = "GDK_KEY_ocircumflex")]
    pub const ocircumflex: Self = Self(ffi::GDK_KEY_ocircumflex as u32);
    #[doc(alias = "GDK_KEY_ocircumflexacute")]
    pub const ocircumflexacute: Self = Self(ffi::GDK_KEY_ocircumflexacute as u32);
    #[doc(alias = "GDK_KEY_ocircumflexbelowdot")]
    pub const ocircumflexbelowdot: Self = Self(ffi::GDK_KEY_ocircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_ocircumflexgrave")]
    pub const ocircumflexgrave: Self = Self(ffi::GDK_KEY_ocircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_ocircumflexhook")]
    pub const ocircumflexhook: Self = Self(ffi::GDK_KEY_ocircumflexhook as u32);
    #[doc(alias = "GDK_KEY_ocircumflextilde")]
    pub const ocircumflextilde: Self = Self(ffi::GDK_KEY_ocircumflextilde as u32);
    #[doc(alias = "GDK_KEY_odiaeresis")]
    pub const odiaeresis: Self = Self(ffi::GDK_KEY_odiaeresis as u32);
    #[doc(alias = "GDK_KEY_odoubleacute")]
    pub const odoubleacute: Self = Self(ffi::GDK_KEY_odoubleacute as u32);
    #[doc(alias = "GDK_KEY_oe")]
    pub const oe: Self = Self(ffi::GDK_KEY_oe as u32);
    #[doc(alias = "GDK_KEY_ogonek")]
    pub const ogonek: Self = Self(ffi::GDK_KEY_ogonek as u32);
    #[doc(alias = "GDK_KEY_ograve")]
    pub const ograve: Self = Self(ffi::GDK_KEY_ograve as u32);
    #[doc(alias = "GDK_KEY_ohook")]
    pub const ohook: Self = Self(ffi::GDK_KEY_ohook as u32);
    #[doc(alias = "GDK_KEY_ohorn")]
    pub const ohorn: Self = Self(ffi::GDK_KEY_ohorn as u32);
    #[doc(alias = "GDK_KEY_ohornacute")]
    pub const ohornacute: Self = Self(ffi::GDK_KEY_ohornacute as u32);
    #[doc(alias = "GDK_KEY_ohornbelowdot")]
    pub const ohornbelowdot: Self = Self(ffi::GDK_KEY_ohornbelowdot as u32);
    #[doc(alias = "GDK_KEY_ohorngrave")]
    pub const ohorngrave: Self = Self(ffi::GDK_KEY_ohorngrave as u32);
    #[doc(alias = "GDK_KEY_ohornhook")]
    pub const ohornhook: Self = Self(ffi::GDK_KEY_ohornhook as u32);
    #[doc(alias = "GDK_KEY_ohorntilde")]
    pub const ohorntilde: Self = Self(ffi::GDK_KEY_ohorntilde as u32);
    #[doc(alias = "GDK_KEY_omacron")]
    pub const omacron: Self = Self(ffi::GDK_KEY_omacron as u32);
    #[doc(alias = "GDK_KEY_oneeighth")]
    pub const oneeighth: Self = Self(ffi::GDK_KEY_oneeighth as u32);
    #[doc(alias = "GDK_KEY_onefifth")]
    pub const onefifth: Self = Self(ffi::GDK_KEY_onefifth as u32);
    #[doc(alias = "GDK_KEY_onehalf")]
    pub const onehalf: Self = Self(ffi::GDK_KEY_onehalf as u32);
    #[doc(alias = "GDK_KEY_onequarter")]
    pub const onequarter: Self = Self(ffi::GDK_KEY_onequarter as u32);
    #[doc(alias = "GDK_KEY_onesixth")]
    pub const onesixth: Self = Self(ffi::GDK_KEY_onesixth as u32);
    #[doc(alias = "GDK_KEY_onesubscript")]
    pub const onesubscript: Self = Self(ffi::GDK_KEY_onesubscript as u32);
    #[doc(alias = "GDK_KEY_onesuperior")]
    pub const onesuperior: Self = Self(ffi::GDK_KEY_onesuperior as u32);
    #[doc(alias = "GDK_KEY_onethird")]
    pub const onethird: Self = Self(ffi::GDK_KEY_onethird as u32);
    #[doc(alias = "GDK_KEY_ooblique")]
    pub const ooblique: Self = Self(ffi::GDK_KEY_ooblique as u32);
    #[doc(alias = "GDK_KEY_openrectbullet")]
    pub const openrectbullet: Self = Self(ffi::GDK_KEY_openrectbullet as u32);
    #[doc(alias = "GDK_KEY_openstar")]
    pub const openstar: Self = Self(ffi::GDK_KEY_openstar as u32);
    #[doc(alias = "GDK_KEY_opentribulletdown")]
    pub const opentribulletdown: Self = Self(ffi::GDK_KEY_opentribulletdown as u32);
    #[doc(alias = "GDK_KEY_opentribulletup")]
    pub const opentribulletup: Self = Self(ffi::GDK_KEY_opentribulletup as u32);
    #[doc(alias = "GDK_KEY_ordfeminine")]
    pub const ordfeminine: Self = Self(ffi::GDK_KEY_ordfeminine as u32);
    #[doc(alias = "GDK_KEY_ordmasculine")]
    pub const ordmasculine: Self = Self(ffi::GDK_KEY_ordmasculine as u32);
    #[doc(alias = "GDK_KEY_oslash")]
    pub const oslash: Self = Self(ffi::GDK_KEY_oslash as u32);
    #[doc(alias = "GDK_KEY_otilde")]
    pub const otilde: Self = Self(ffi::GDK_KEY_otilde as u32);
    #[doc(alias = "GDK_KEY_overbar")]
    pub const overbar: Self = Self(ffi::GDK_KEY_overbar as u32);
    #[doc(alias = "GDK_KEY_overline")]
    pub const overline: Self = Self(ffi::GDK_KEY_overline as u32);
    #[doc(alias = "GDK_KEY_p")]
    pub const p: Self = Self(ffi::GDK_KEY_p as u32);
    #[doc(alias = "GDK_KEY_pabovedot")]
    pub const pabovedot: Self = Self(ffi::GDK_KEY_pabovedot as u32);
    #[doc(alias = "GDK_KEY_paragraph")]
    pub const paragraph: Self = Self(ffi::GDK_KEY_paragraph as u32);
    #[doc(alias = "GDK_KEY_parenleft")]
    pub const parenleft: Self = Self(ffi::GDK_KEY_parenleft as u32);
    #[doc(alias = "GDK_KEY_parenright")]
    pub const parenright: Self = Self(ffi::GDK_KEY_parenright as u32);
    #[doc(alias = "GDK_KEY_partdifferential")]
    pub const partdifferential: Self = Self(ffi::GDK_KEY_partdifferential as u32);
    #[doc(alias = "GDK_KEY_partialderivative")]
    pub const partialderivative: Self = Self(ffi::GDK_KEY_partialderivative as u32);
    #[doc(alias = "GDK_KEY_percent")]
    pub const percent: Self = Self(ffi::GDK_KEY_percent as u32);
    #[doc(alias = "GDK_KEY_period")]
    pub const period: Self = Self(ffi::GDK_KEY_period as u32);
    #[doc(alias = "GDK_KEY_periodcentered")]
    pub const periodcentered: Self = Self(ffi::GDK_KEY_periodcentered as u32);
    #[doc(alias = "GDK_KEY_permille")]
    pub const permille: Self = Self(ffi::GDK_KEY_permille as u32);
    #[doc(alias = "GDK_KEY_phonographcopyright")]
    pub const phonographcopyright: Self = Self(ffi::GDK_KEY_phonographcopyright as u32);
    #[doc(alias = "GDK_KEY_plus")]
    pub const plus: Self = Self(ffi::GDK_KEY_plus as u32);
    #[doc(alias = "GDK_KEY_plusminus")]
    pub const plusminus: Self = Self(ffi::GDK_KEY_plusminus as u32);
    #[doc(alias = "GDK_KEY_prescription")]
    pub const prescription: Self = Self(ffi::GDK_KEY_prescription as u32);
    #[doc(alias = "GDK_KEY_prolongedsound")]
    pub const prolongedsound: Self = Self(ffi::GDK_KEY_prolongedsound as u32);
    #[doc(alias = "GDK_KEY_punctspace")]
    pub const punctspace: Self = Self(ffi::GDK_KEY_punctspace as u32);
    #[doc(alias = "GDK_KEY_q")]
    pub const q: Self = Self(ffi::GDK_KEY_q as u32);
    #[doc(alias = "GDK_KEY_quad")]
    pub const quad: Self = Self(ffi::GDK_KEY_quad as u32);
    #[doc(alias = "GDK_KEY_question")]
    pub const question: Self = Self(ffi::GDK_KEY_question as u32);
    #[doc(alias = "GDK_KEY_questiondown")]
    pub const questiondown: Self = Self(ffi::GDK_KEY_questiondown as u32);
    #[doc(alias = "GDK_KEY_quotedbl")]
    pub const quotedbl: Self = Self(ffi::GDK_KEY_quotedbl as u32);
    #[doc(alias = "GDK_KEY_quoteleft")]
    pub const quoteleft: Self = Self(ffi::GDK_KEY_quoteleft as u32);
    #[doc(alias = "GDK_KEY_quoteright")]
    pub const quoteright: Self = Self(ffi::GDK_KEY_quoteright as u32);
    #[doc(alias = "GDK_KEY_r")]
    pub const r: Self = Self(ffi::GDK_KEY_r as u32);
    #[doc(alias = "GDK_KEY_racute")]
    pub const racute: Self = Self(ffi::GDK_KEY_racute as u32);
    #[doc(alias = "GDK_KEY_radical")]
    pub const radical: Self = Self(ffi::GDK_KEY_radical as u32);
    #[doc(alias = "GDK_KEY_rcaron")]
    pub const rcaron: Self = Self(ffi::GDK_KEY_rcaron as u32);
    #[doc(alias = "GDK_KEY_rcedilla")]
    pub const rcedilla: Self = Self(ffi::GDK_KEY_rcedilla as u32);
    #[doc(alias = "GDK_KEY_registered")]
    pub const registered: Self = Self(ffi::GDK_KEY_registered as u32);
    #[doc(alias = "GDK_KEY_rightanglebracket")]
    pub const rightanglebracket: Self = Self(ffi::GDK_KEY_rightanglebracket as u32);
    #[doc(alias = "GDK_KEY_rightarrow")]
    pub const rightarrow: Self = Self(ffi::GDK_KEY_rightarrow as u32);
    #[doc(alias = "GDK_KEY_rightcaret")]
    pub const rightcaret: Self = Self(ffi::GDK_KEY_rightcaret as u32);
    #[doc(alias = "GDK_KEY_rightdoublequotemark")]
    pub const rightdoublequotemark: Self = Self(ffi::GDK_KEY_rightdoublequotemark as u32);
    #[doc(alias = "GDK_KEY_rightmiddlecurlybrace")]
    pub const rightmiddlecurlybrace: Self = Self(ffi::GDK_KEY_rightmiddlecurlybrace as u32);
    #[doc(alias = "GDK_KEY_rightmiddlesummation")]
    pub const rightmiddlesummation: Self = Self(ffi::GDK_KEY_rightmiddlesummation as u32);
    #[doc(alias = "GDK_KEY_rightopentriangle")]
    pub const rightopentriangle: Self = Self(ffi::GDK_KEY_rightopentriangle as u32);
    #[doc(alias = "GDK_KEY_rightpointer")]
    pub const rightpointer: Self = Self(ffi::GDK_KEY_rightpointer as u32);
    #[doc(alias = "GDK_KEY_rightshoe")]
    pub const rightshoe: Self = Self(ffi::GDK_KEY_rightshoe as u32);
    #[doc(alias = "GDK_KEY_rightsinglequotemark")]
    pub const rightsinglequotemark: Self = Self(ffi::GDK_KEY_rightsinglequotemark as u32);
    #[doc(alias = "GDK_KEY_rightt")]
    pub const rightt: Self = Self(ffi::GDK_KEY_rightt as u32);
    #[doc(alias = "GDK_KEY_righttack")]
    pub const righttack: Self = Self(ffi::GDK_KEY_righttack as u32);
    #[doc(alias = "GDK_KEY_s")]
    pub const s: Self = Self(ffi::GDK_KEY_s as u32);
    #[doc(alias = "GDK_KEY_sabovedot")]
    pub const sabovedot: Self = Self(ffi::GDK_KEY_sabovedot as u32);
    #[doc(alias = "GDK_KEY_sacute")]
    pub const sacute: Self = Self(ffi::GDK_KEY_sacute as u32);
    #[doc(alias = "GDK_KEY_scaron")]
    pub const scaron: Self = Self(ffi::GDK_KEY_scaron as u32);
    #[doc(alias = "GDK_KEY_scedilla")]
    pub const scedilla: Self = Self(ffi::GDK_KEY_scedilla as u32);
    #[doc(alias = "GDK_KEY_schwa")]
    pub const schwa: Self = Self(ffi::GDK_KEY_schwa as u32);
    #[doc(alias = "GDK_KEY_scircumflex")]
    pub const scircumflex: Self = Self(ffi::GDK_KEY_scircumflex as u32);
    #[doc(alias = "GDK_KEY_script_switch")]
    pub const script_switch: Self = Self(ffi::GDK_KEY_script_switch as u32);
    #[doc(alias = "GDK_KEY_seconds")]
    pub const seconds: Self = Self(ffi::GDK_KEY_seconds as u32);
    #[doc(alias = "GDK_KEY_section")]
    pub const section: Self = Self(ffi::GDK_KEY_section as u32);
    #[doc(alias = "GDK_KEY_semicolon")]
    pub const semicolon: Self = Self(ffi::GDK_KEY_semicolon as u32);
    #[doc(alias = "GDK_KEY_semivoicedsound")]
    pub const semivoicedsound: Self = Self(ffi::GDK_KEY_semivoicedsound as u32);
    #[doc(alias = "GDK_KEY_seveneighths")]
    pub const seveneighths: Self = Self(ffi::GDK_KEY_seveneighths as u32);
    #[doc(alias = "GDK_KEY_sevensubscript")]
    pub const sevensubscript: Self = Self(ffi::GDK_KEY_sevensubscript as u32);
    #[doc(alias = "GDK_KEY_sevensuperior")]
    pub const sevensuperior: Self = Self(ffi::GDK_KEY_sevensuperior as u32);
    #[doc(alias = "GDK_KEY_signaturemark")]
    pub const signaturemark: Self = Self(ffi::GDK_KEY_signaturemark as u32);
    #[doc(alias = "GDK_KEY_signifblank")]
    pub const signifblank: Self = Self(ffi::GDK_KEY_signifblank as u32);
    #[doc(alias = "GDK_KEY_similarequal")]
    pub const similarequal: Self = Self(ffi::GDK_KEY_similarequal as u32);
    #[doc(alias = "GDK_KEY_singlelowquotemark")]
    pub const singlelowquotemark: Self = Self(ffi::GDK_KEY_singlelowquotemark as u32);
    #[doc(alias = "GDK_KEY_sixsubscript")]
    pub const sixsubscript: Self = Self(ffi::GDK_KEY_sixsubscript as u32);
    #[doc(alias = "GDK_KEY_sixsuperior")]
    pub const sixsuperior: Self = Self(ffi::GDK_KEY_sixsuperior as u32);
    #[doc(alias = "GDK_KEY_slash")]
    pub const slash: Self = Self(ffi::GDK_KEY_slash as u32);
    #[doc(alias = "GDK_KEY_soliddiamond")]
    pub const soliddiamond: Self = Self(ffi::GDK_KEY_soliddiamond as u32);
    #[doc(alias = "GDK_KEY_space")]
    pub const space: Self = Self(ffi::GDK_KEY_space as u32);
    #[doc(alias = "GDK_KEY_squareroot")]
    pub const squareroot: Self = Self(ffi::GDK_KEY_squareroot as u32);
    #[doc(alias = "GDK_KEY_ssharp")]
    pub const ssharp: Self = Self(ffi::GDK_KEY_ssharp as u32);
    #[doc(alias = "GDK_KEY_sterling")]
    pub const sterling: Self = Self(ffi::GDK_KEY_sterling as u32);
    #[doc(alias = "GDK_KEY_stricteq")]
    pub const stricteq: Self = Self(ffi::GDK_KEY_stricteq as u32);
    #[doc(alias = "GDK_KEY_t")]
    pub const t: Self = Self(ffi::GDK_KEY_t as u32);
    #[doc(alias = "GDK_KEY_tabovedot")]
    pub const tabovedot: Self = Self(ffi::GDK_KEY_tabovedot as u32);
    #[doc(alias = "GDK_KEY_tcaron")]
    pub const tcaron: Self = Self(ffi::GDK_KEY_tcaron as u32);
    #[doc(alias = "GDK_KEY_tcedilla")]
    pub const tcedilla: Self = Self(ffi::GDK_KEY_tcedilla as u32);
    #[doc(alias = "GDK_KEY_telephone")]
    pub const telephone: Self = Self(ffi::GDK_KEY_telephone as u32);
    #[doc(alias = "GDK_KEY_telephonerecorder")]
    pub const telephonerecorder: Self = Self(ffi::GDK_KEY_telephonerecorder as u32);
    #[doc(alias = "GDK_KEY_therefore")]
    pub const therefore: Self = Self(ffi::GDK_KEY_therefore as u32);
    #[doc(alias = "GDK_KEY_thinspace")]
    pub const thinspace: Self = Self(ffi::GDK_KEY_thinspace as u32);
    #[doc(alias = "GDK_KEY_thorn")]
    pub const thorn: Self = Self(ffi::GDK_KEY_thorn as u32);
    #[doc(alias = "GDK_KEY_threeeighths")]
    pub const threeeighths: Self = Self(ffi::GDK_KEY_threeeighths as u32);
    #[doc(alias = "GDK_KEY_threefifths")]
    pub const threefifths: Self = Self(ffi::GDK_KEY_threefifths as u32);
    #[doc(alias = "GDK_KEY_threequarters")]
    pub const threequarters: Self = Self(ffi::GDK_KEY_threequarters as u32);
    #[doc(alias = "GDK_KEY_threesubscript")]
    pub const threesubscript: Self = Self(ffi::GDK_KEY_threesubscript as u32);
    #[doc(alias = "GDK_KEY_threesuperior")]
    pub const threesuperior: Self = Self(ffi::GDK_KEY_threesuperior as u32);
    #[doc(alias = "GDK_KEY_tintegral")]
    pub const tintegral: Self = Self(ffi::GDK_KEY_tintegral as u32);
    #[doc(alias = "GDK_KEY_topintegral")]
    pub const topintegral: Self = Self(ffi::GDK_KEY_topintegral as u32);
    #[doc(alias = "GDK_KEY_topleftparens")]
    pub const topleftparens: Self = Self(ffi::GDK_KEY_topleftparens as u32);
    #[doc(alias = "GDK_KEY_topleftradical")]
    pub const topleftradical: Self = Self(ffi::GDK_KEY_topleftradical as u32);
    #[doc(alias = "GDK_KEY_topleftsqbracket")]
    pub const topleftsqbracket: Self = Self(ffi::GDK_KEY_topleftsqbracket as u32);
    #[doc(alias = "GDK_KEY_topleftsummation")]
    pub const topleftsummation: Self = Self(ffi::GDK_KEY_topleftsummation as u32);
    #[doc(alias = "GDK_KEY_toprightparens")]
    pub const toprightparens: Self = Self(ffi::GDK_KEY_toprightparens as u32);
    #[doc(alias = "GDK_KEY_toprightsqbracket")]
    pub const toprightsqbracket: Self = Self(ffi::GDK_KEY_toprightsqbracket as u32);
    #[doc(alias = "GDK_KEY_toprightsummation")]
    pub const toprightsummation: Self = Self(ffi::GDK_KEY_toprightsummation as u32);
    #[doc(alias = "GDK_KEY_topt")]
    pub const topt: Self = Self(ffi::GDK_KEY_topt as u32);
    #[doc(alias = "GDK_KEY_topvertsummationconnector")]
    pub const topvertsummationconnector: Self = Self(ffi::GDK_KEY_topvertsummationconnector as u32);
    #[doc(alias = "GDK_KEY_trademark")]
    pub const trademark: Self = Self(ffi::GDK_KEY_trademark as u32);
    #[doc(alias = "GDK_KEY_trademarkincircle")]
    pub const trademarkincircle: Self = Self(ffi::GDK_KEY_trademarkincircle as u32);
    #[doc(alias = "GDK_KEY_tslash")]
    pub const tslash: Self = Self(ffi::GDK_KEY_tslash as u32);
    #[doc(alias = "GDK_KEY_twofifths")]
    pub const twofifths: Self = Self(ffi::GDK_KEY_twofifths as u32);
    #[doc(alias = "GDK_KEY_twosubscript")]
    pub const twosubscript: Self = Self(ffi::GDK_KEY_twosubscript as u32);
    #[doc(alias = "GDK_KEY_twosuperior")]
    pub const twosuperior: Self = Self(ffi::GDK_KEY_twosuperior as u32);
    #[doc(alias = "GDK_KEY_twothirds")]
    pub const twothirds: Self = Self(ffi::GDK_KEY_twothirds as u32);
    #[doc(alias = "GDK_KEY_u")]
    pub const u: Self = Self(ffi::GDK_KEY_u as u32);
    #[doc(alias = "GDK_KEY_uacute")]
    pub const uacute: Self = Self(ffi::GDK_KEY_uacute as u32);
    #[doc(alias = "GDK_KEY_ubelowdot")]
    pub const ubelowdot: Self = Self(ffi::GDK_KEY_ubelowdot as u32);
    #[doc(alias = "GDK_KEY_ubreve")]
    pub const ubreve: Self = Self(ffi::GDK_KEY_ubreve as u32);
    #[doc(alias = "GDK_KEY_ucircumflex")]
    pub const ucircumflex: Self = Self(ffi::GDK_KEY_ucircumflex as u32);
    #[doc(alias = "GDK_KEY_udiaeresis")]
    pub const udiaeresis: Self = Self(ffi::GDK_KEY_udiaeresis as u32);
    #[doc(alias = "GDK_KEY_udoubleacute")]
    pub const udoubleacute: Self = Self(ffi::GDK_KEY_udoubleacute as u32);
    #[doc(alias = "GDK_KEY_ugrave")]
    pub const ugrave: Self = Self(ffi::GDK_KEY_ugrave as u32);
    #[doc(alias = "GDK_KEY_uhook")]
    pub const uhook: Self = Self(ffi::GDK_KEY_uhook as u32);
    #[doc(alias = "GDK_KEY_uhorn")]
    pub const uhorn: Self = Self(ffi::GDK_KEY_uhorn as u32);
    #[doc(alias = "GDK_KEY_uhornacute")]
    pub const uhornacute: Self = Self(ffi::GDK_KEY_uhornacute as u32);
    #[doc(alias = "GDK_KEY_uhornbelowdot")]
    pub const uhornbelowdot: Self = Self(ffi::GDK_KEY_uhornbelowdot as u32);
    #[doc(alias = "GDK_KEY_uhorngrave")]
    pub const uhorngrave: Self = Self(ffi::GDK_KEY_uhorngrave as u32);
    #[doc(alias = "GDK_KEY_uhornhook")]
    pub const uhornhook: Self = Self(ffi::GDK_KEY_uhornhook as u32);
    #[doc(alias = "GDK_KEY_uhorntilde")]
    pub const uhorntilde: Self = Self(ffi::GDK_KEY_uhorntilde as u32);
    #[doc(alias = "GDK_KEY_umacron")]
    pub const umacron: Self = Self(ffi::GDK_KEY_umacron as u32);
    #[doc(alias = "GDK_KEY_underbar")]
    pub const underbar: Self = Self(ffi::GDK_KEY_underbar as u32);
    #[doc(alias = "GDK_KEY_underscore")]
    pub const underscore: Self = Self(ffi::GDK_KEY_underscore as u32);
    #[doc(alias = "GDK_KEY_union")]
    pub const union: Self = Self(ffi::GDK_KEY_union as u32);
    #[doc(alias = "GDK_KEY_uogonek")]
    pub const uogonek: Self = Self(ffi::GDK_KEY_uogonek as u32);
    #[doc(alias = "GDK_KEY_uparrow")]
    pub const uparrow: Self = Self(ffi::GDK_KEY_uparrow as u32);
    #[doc(alias = "GDK_KEY_upcaret")]
    pub const upcaret: Self = Self(ffi::GDK_KEY_upcaret as u32);
    #[doc(alias = "GDK_KEY_upleftcorner")]
    pub const upleftcorner: Self = Self(ffi::GDK_KEY_upleftcorner as u32);
    #[doc(alias = "GDK_KEY_uprightcorner")]
    pub const uprightcorner: Self = Self(ffi::GDK_KEY_uprightcorner as u32);
    #[doc(alias = "GDK_KEY_upshoe")]
    pub const upshoe: Self = Self(ffi::GDK_KEY_upshoe as u32);
    #[doc(alias = "GDK_KEY_upstile")]
    pub const upstile: Self = Self(ffi::GDK_KEY_upstile as u32);
    #[doc(alias = "GDK_KEY_uptack")]
    pub const uptack: Self = Self(ffi::GDK_KEY_uptack as u32);
    #[doc(alias = "GDK_KEY_uring")]
    pub const uring: Self = Self(ffi::GDK_KEY_uring as u32);
    #[doc(alias = "GDK_KEY_utilde")]
    pub const utilde: Self = Self(ffi::GDK_KEY_utilde as u32);
    #[doc(alias = "GDK_KEY_v")]
    pub const v: Self = Self(ffi::GDK_KEY_v as u32);
    #[doc(alias = "GDK_KEY_variation")]
    pub const variation: Self = Self(ffi::GDK_KEY_variation as u32);
    #[doc(alias = "GDK_KEY_vertbar")]
    pub const vertbar: Self = Self(ffi::GDK_KEY_vertbar as u32);
    #[doc(alias = "GDK_KEY_vertconnector")]
    pub const vertconnector: Self = Self(ffi::GDK_KEY_vertconnector as u32);
    #[doc(alias = "GDK_KEY_voicedsound")]
    pub const voicedsound: Self = Self(ffi::GDK_KEY_voicedsound as u32);
    #[doc(alias = "GDK_KEY_vt")]
    pub const vt: Self = Self(ffi::GDK_KEY_vt as u32);
    #[doc(alias = "GDK_KEY_w")]
    pub const w: Self = Self(ffi::GDK_KEY_w as u32);
    #[doc(alias = "GDK_KEY_wacute")]
    pub const wacute: Self = Self(ffi::GDK_KEY_wacute as u32);
    #[doc(alias = "GDK_KEY_wcircumflex")]
    pub const wcircumflex: Self = Self(ffi::GDK_KEY_wcircumflex as u32);
    #[doc(alias = "GDK_KEY_wdiaeresis")]
    pub const wdiaeresis: Self = Self(ffi::GDK_KEY_wdiaeresis as u32);
    #[doc(alias = "GDK_KEY_wgrave")]
    pub const wgrave: Self = Self(ffi::GDK_KEY_wgrave as u32);
    #[doc(alias = "GDK_KEY_x")]
    pub const x: Self = Self(ffi::GDK_KEY_x as u32);
    #[doc(alias = "GDK_KEY_xabovedot")]
    pub const xabovedot: Self = Self(ffi::GDK_KEY_xabovedot as u32);
    #[doc(alias = "GDK_KEY_y")]
    pub const y: Self = Self(ffi::GDK_KEY_y as u32);
    #[doc(alias = "GDK_KEY_yacute")]
    pub const yacute: Self = Self(ffi::GDK_KEY_yacute as u32);
    #[doc(alias = "GDK_KEY_ybelowdot")]
    pub const ybelowdot: Self = Self(ffi::GDK_KEY_ybelowdot as u32);
    #[doc(alias = "GDK_KEY_ycircumflex")]
    pub const ycircumflex: Self = Self(ffi::GDK_KEY_ycircumflex as u32);
    #[doc(alias = "GDK_KEY_ydiaeresis")]
    pub const ydiaeresis: Self = Self(ffi::GDK_KEY_ydiaeresis as u32);
    #[doc(alias = "GDK_KEY_yen")]
    pub const yen: Self = Self(ffi::GDK_KEY_yen as u32);
    #[doc(alias = "GDK_KEY_ygrave")]
    pub const ygrave: Self = Self(ffi::GDK_KEY_ygrave as u32);
    #[doc(alias = "GDK_KEY_yhook")]
    pub const yhook: Self = Self(ffi::GDK_KEY_yhook as u32);
    #[doc(alias = "GDK_KEY_ytilde")]
    pub const ytilde: Self = Self(ffi::GDK_KEY_ytilde as u32);
    #[doc(alias = "GDK_KEY_z")]
    pub const z: Self = Self(ffi::GDK_KEY_z as u32);
    #[doc(alias = "GDK_KEY_zabovedot")]
    pub const zabovedot: Self = Self(ffi::GDK_KEY_zabovedot as u32);
    #[doc(alias = "GDK_KEY_zacute")]
    pub const zacute: Self = Self(ffi::GDK_KEY_zacute as u32);
    #[doc(alias = "GDK_KEY_zcaron")]
    pub const zcaron: Self = Self(ffi::GDK_KEY_zcaron as u32);
    #[doc(alias = "GDK_KEY_zerosubscript")]
    pub const zerosubscript: Self = Self(ffi::GDK_KEY_zerosubscript as u32);
    #[doc(alias = "GDK_KEY_zerosuperior")]
    pub const zerosuperior: Self = Self(ffi::GDK_KEY_zerosuperior as u32);
    #[doc(alias = "GDK_KEY_zstroke")]
    pub const zstroke: Self = Self(ffi::GDK_KEY_zstroke as u32);
}

impl ::std::fmt::Display for Key {
    fn fmt(&self, f: &mut ::std::fmt::Formatter<'_>) -> ::std::fmt::Result {
        f.debug_struct("Key")
            .field("name", &self.name())
            .field("is_lower", &self.is_lower())
            .field("is_upper", &self.is_upper())
            .finish()
    }
}

impl FromGlib<u32> for Key {
    #[inline]
    unsafe fn from_glib(value: u32) -> Self {
        skip_assert_initialized!();
        Self(value)
    }
}

impl IntoGlib for Key {
    type GlibType = u32;

    #[inline]
    fn into_glib(self) -> u32 {
        self.0
    }
}

impl ValueType for Key {
    type Type = u32;
}

unsafe impl<'a> FromValue<'a> for Key {
    type Checker = glib::value::GenericValueTypeChecker<Key>;

    #[inline]
    unsafe fn from_value(value: &'a Value) -> Self {
        let res: u32 = glib::gobject_ffi::g_value_get_uint(value.to_glib_none().0);
        // As most of gdk_keyval_ apis don't really do any check for the input value
        // (the key number) other than gdk_keyval_from_name, it is safe to not
        // do any checks and assume people will not mis-use it
        Key::from_glib(res)
    }
}

impl ToValue for Key {
    #[inline]
    fn to_value(&self) -> Value {
        let value = Value::for_value_type::<Self>();
        unsafe {
            glib::gobject_ffi::g_value_set_uint(value.to_glib_none().0 as *mut _, self.into_glib());
        }
        value
    }

    #[inline]
    fn value_type(&self) -> Type {
        Type::U32
    }
}

impl From<Key> for Value {
    #[inline]
    fn from(k: Key) -> Self {
        skip_assert_initialized!();
        k.to_value()
    }
}

// TODO: Drop once https://github.com/gtk-rs/gtk-rs-core/issues/612
impl StaticType for Key {
    #[inline]
    fn static_type() -> Type {
        Type::U32
    }
}

#[cfg(test)]
mod test {
    use super::*;

    #[std::prelude::v1::test]
    fn test_key_value() {
        let key = Key::KP_Enter;
        let value = key.to_value();

        assert_eq!(value.get::<u32>(), Ok(Key::KP_Enter.into_glib()));
        assert_eq!(unsafe { Key::from_value(&value) }, key);
        assert_eq!(unsafe { Key::from_glib(value.get::<u32>().unwrap()) }, key);
    }
}
