# Copyright 2025 Felix Fontein <felix@fontein.de>
# GNU General Public License v3.0+ (see LICENSES/GPL-3.0-or-later.txt or https://www.gnu.org/licenses/gpl-3.0.txt)
# SPDX-License-Identifier: GPL-3.0-or-later

from __future__ import annotations

import typing as t

import pytest

from ansible_collections.community.docker.plugins.modules.docker_container_copy_into import (
    parse_modern,
    parse_octal_string_only,
)


@pytest.mark.parametrize(
    "value, expected",
    [
        ("0777", 0o777),
        ("777", 0o777),
        ("0o777", 0o777),
        ("0755", 0o755),
        ("755", 0o755),
        ("0o755", 0o755),
        ("0644", 0o644),
        ("644", 0o644),
        ("0o644", 0o644),
        (" 0644 ", 0o644),
        (" 644 ", 0o644),
        (" 0o644 ", 0o644),
        ("-1", -1),
    ],
)
def test_parse_string(value: str, expected: int) -> None:
    assert parse_modern(value) == expected
    assert parse_octal_string_only(value) == expected


@pytest.mark.parametrize(
    "value",
    [
        0o777,
        0o755,
        0o644,
        12345,
        123456789012345678901234567890123456789012345678901234567890,
    ],
)
def test_parse_int(value: int) -> None:
    assert parse_modern(value) == value
    with pytest.raises(TypeError, match=f"^must be an octal string, got {value}L?$"):
        parse_octal_string_only(value)  # type: ignore


@pytest.mark.parametrize(
    "value",
    [
        1.0,
        755.5,
        [],
        {},
    ],
)
def test_parse_bad_type(value: t.Any) -> None:
    with pytest.raises(TypeError, match="^must be an octal string or an integer, got "):
        parse_modern(value)
    with pytest.raises(TypeError, match="^must be an octal string, got "):
        parse_octal_string_only(value)


@pytest.mark.parametrize(
    "value",
    [
        "foo",
        "8",
        "9",
    ],
)
def test_parse_bad_value(value: str) -> None:
    with pytest.raises(ValueError):
        parse_modern(value)
    with pytest.raises(ValueError):
        parse_octal_string_only(value)
