/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include <Application.h>
#include <Core.h>
#include <HotPlugExtensionManager.h>
#include <ExtensionManager.h>
#include <MainWindow.h>
#include <Log.h>

using namespace camitk;

#include <iostream>
#include <QDateTime>
#include <QJsonDocument>

#include "CommandLineOptions.hxx"
#include "CamiTKVersionInformation.h"

// description of the application. Please update the manpage-prologue.1.in also if you modify this string.
const char* description = "camitk-config is the configuration utility for CamiTK,\n"
                          "it can report various information about the current configuration.\n\n"
                          "Please visit https://camitk.imag.fr for more information.\n"
                          "(c) Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, France";

// usage for this application
void usage(std::string msg = "") {
    if (msg != "") {
        std::cerr << msg << std::endl << std::endl;
    }
    std::cerr << "Usage: camitk-config [options]" << std::endl << std::endl;
    std::cerr << description << std::endl;
    std::cerr << std::endl;
    std::cerr << "Version: " << Core::version() << std::endl;
    std::cerr << std::endl;
    std::cerr << "Options:" << std::endl;
    options::print_usage(std::cerr);
}

int main(int argc, char* argv[]) {
    try {
        int end; // End of options.
        options o(argc, argv, end);

        // if specific help or no options provided
        if (o.help()) {
            usage(argv[0]);
            return EXIT_SUCCESS;
        }
        else {
            // print all types of versions (just using the char* defined in CamiTKVersion.h at configuration time)
            if (o.version()) {
                std::cout << argv[0] << " build using " << Core::version() << std::endl;
                if (compiledFromPackageSource()) {
                    std::cout << "Compiled using package source." << std::endl;
                }
                else {
                    std::cout << "Compiled using git";
                    if (hasMoreGitInformation()) { // Check if CAMITK_GIT_FOUND can be set to "1" otherwise we never get into this 'if'
                        std::cout << " Hash: " << gitHash() << ", Date: " << gitDate();
                    }
                    std::cout << "." << std::endl;
                }
                return EXIT_SUCCESS;
            }
            else if (o.complete_version()) {
                std::cout << Core::version() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (o.short_version()) {
                std::cout << Core::shortVersion() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (o.time_stamp()) {
                QDateTime now = QDateTime::currentDateTime();
                std::cout << now.toString("yyyy-MM-dd").toStdString() << "T" << now.toString("hh:mm:ss").toStdString() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (o.print_paths()) {
                // init the camitk application context without auto-loading all extensions, log level is NONE
                Log::getLogger()->setLogLevel(InterfaceLogger::NONE);
                Application a("camitk-config", argc, argv, false, false);
                std::cout << Core::getPaths().toStdString() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (o.camitk_dir()) {
                // init the camitk application context without auto-loading all extensions, log level is NONE
                Log::getLogger()->setLogLevel(InterfaceLogger::NONE);
                Application a("camitk-config", argc, argv, false, false);
                std::cout << Core::getGlobalInstallDir().toStdString() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (o.config()) {
                // init the camitk application context (auto-load of all extensions), log level is WARNING
                Log::getLogger()->setLogLevel(InterfaceLogger::WARNING);
                Application a("camitk-config", argc, argv, true, false);
                // avoid all redirection to console
                MainWindow* defaultMainWindow = dynamic_cast<MainWindow*>(a.getMainWindow());
                defaultMainWindow->redirectToConsole(false);
                // print the configuration on the std::cout
                std::cout << Core::getConfig().toStdString() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (o.config_json()) {
                // init the camitk application context (auto-load of all extensions), log level is WARNING
                Log::getLogger()->setLogLevel(InterfaceLogger::WARNING);
                Application a("camitk-config", argc, argv, true, false);
                // avoid all redirection to console
                MainWindow* defaultMainWindow = dynamic_cast<MainWindow*>(a.getMainWindow());
                defaultMainWindow->redirectToConsole(false);
                // print the JSON configuration on the std::cout
                QJsonObject config = Core::getConfigAsJson();
                std::cout << QJsonDocument(config).toJson(QJsonDocument::Indented).toStdString() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (o.bug_report_info()) {
                // init the camitk application context (auto-load of all extensions), log level is INFO
                Log::getLogger()->setLogLevel(InterfaceLogger::WARNING);
                Application a("camitk-config", argc, argv, true, false);
                // avoid all redirection to console
                MainWindow* defaultMainWindow = dynamic_cast<MainWindow*>(a.getMainWindow());
                defaultMainWindow->redirectToConsole(false);
                // print the configuration on the std::cout
                std::cout << Core::getBugReport().toStdString() << std::endl;
                return EXIT_SUCCESS;
            }
            else if (!o.check_hotplug_extension().empty()) {
                // init the camitk application context (auto-load of all extensions), log level is WARNING
                Log::getLogger()->setLogLevel(InterfaceLogger::WARNING);
                Log::getLogger()->setMessageBoxLevel(InterfaceLogger::NONE);
                Application a("camitk-config", argc, argv, true, false);

                // force back log level in case the is a different configuration setup in the .ini file
                // for camitk-config
                Log::getLogger()->setLogLevel(InterfaceLogger::WARNING);

                // load the python extension
                if (HotPlugExtensionManager::load(QString::fromLocal8Bit(o.check_hotplug_extension().c_str()))) {
                    Log::getLogger()->setLogLevel(InterfaceLogger::NONE); // so that the application exit message is not the last one to appear on the console, but the successful message
                    std::cout << "Loading " << o.check_hotplug_extension() << " hotplug extension successful." << std::endl;
                    return EXIT_SUCCESS;
                }
                else {
                    Log::getLogger()->setLogLevel(InterfaceLogger::NONE); // so that the application exit message is not the last one to appear on the console, but the failure message
                    std::cout << "Loading " << o.check_hotplug_extension() << " hotplug extension failed." << std::endl;
                    return EXIT_FAILURE;
                }
            }
            else {
                usage(argv[0]);
                return EXIT_SUCCESS;
            }
        }
    }
    catch (const cli::exception& e) {
        std::cerr << e << std::endl << std::endl;
        usage();
        return EXIT_FAILURE;
    }

}
