/*
 *	recite - english text speech synthesizer
 *	Copyright (C) 1993 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: interface definition for recite/file.c
 */

#include <stdio.h>

#include <arglex.h>
#include <error.h>
#include <file.h>
#include <frame.h>
#include <help.h>
#include <mem.h>
#include <trace.h>
#include <version.h>
#include <../english/english.h>
#include <../phonemes/phonemes.h>
#include <../klatt/klatt.h>
#include <../audio/audio.h>

enum
{
	arglex_token_trace,
	arglex_token_volume,
	arglex_token_input,
	arglex_token_output,

	/* group these together, order is important */
	arglex_token_read_english,
	arglex_token_read_phonemes,
	arglex_token_read_klatt,
	arglex_token_read_ulaw,

	/* group these together, order is important */
	arglex_token_write_phonemes,
	arglex_token_write_klatt,
	arglex_token_write_ulaw,
	arglex_token_write_speaker,
	arglex_token_write_jack,
};

static arglex_table_t argtab[] =
{
	{ "-Input",		arglex_token_input,		},
	{ "-Output",		arglex_token_output,		},
	{ "-TRace",		arglex_token_trace,		},
	{ "-Volume",		arglex_token_volume,		},
	{ "-Read_English",	arglex_token_read_english,	},
	{ "-Read_Phonemes",	arglex_token_read_phonemes,	},
	{ "-Read_Klatt",	arglex_token_read_klatt,	},
	{ "-Read_Ulaw",		arglex_token_read_ulaw,		},
	{ "-Write_Phonemes",	arglex_token_write_phonemes,	},
	{ "-Write_Klatt",	arglex_token_write_klatt,	},
	{ "-Write_Ulaw",	arglex_token_write_ulaw,	},
	{ "-Write_Speaker",	arglex_token_write_speaker,	},
	{ "-Write_Jack",	arglex_token_write_jack,	},
	{ 0, 0, }, /* end marker */
};


static void usage _((void));

static void
usage()
{
	fprintf
	(
		stderr,
		"usage: %s [ <option>... ][ <infile> [ <outfile> ]]\n",
		progname
	);
	fprintf(stderr, "       %s -Help\n", progname);
	fprintf(stderr, "       %s -VERSion\n", progname);
	exit(1);
}


static void main_help _((void));

static void
main_help()
{
	static char *text[] =
	{
"NAME",
"	%s - english text speech synthesizer",
"",
"SYNOPSIS",
"	%s [ <option>... ][ <infile> [ <outfile> ]]",
"	%s -Help",
"	%s -VERSion",
"",
"DESCRIPTION",
"	The %s program is used to read english text and speak",
"	the words from the workstation's speaker.",
"",
"	By default, the standard input is read and the english",
"	text is spoken from the speaker.  If a file name is given",
"	on the command line, the text is taken from the file",
"	rather than the standard input.  If a second file is",
"	named, a ulaw encoded file is written to the named file,",
"	rather then emitting it from the speaker.  This file will",
"	be compatible with Sun's audiotool(1).",
"",
"OPTIONS",
"	The following options are understood:",
"",
"	-Write_Phonemes",
"		The option may be used to cause the translation",
"		process to cease after the text has been",
"		translated into phonemes.  The phonemes will be",
"		emitted on the standard output, or to the",
"		appropriate file if an output file has been",
"		specified.",
"",
"	-Help",
"		Provide some help with using the %s program.",
"",
"	-VERSion",
"		Print the version of the %s program being",
"		executed.",
"",
"	All other options will produce a diagnostic error.",
"	Options may be abbreviated, the minimum abbreviation is",
"	shown in upper-case.  Options are case insensitive.",
"	Options and file names may be mixed arbitrarily on the",
"	command line.",
"",
"EXIT STATUS",
"	The %s program will exit with a status of 1 on any",
"	error.  The %s program will only exit with a status",
"	of 0 if there are no errors.",
"",
"COPYRIGHT",
"	%C",
"",
"AUTHOR",
"	%A",
	};

	help(text, SIZEOF(text), usage);
}


int main _((int, char **));

int
main(argc, argv)
	int	argc;
	char	**argv;
{
	char	*infile;
	char	*outfile;
	int	itype;
	int	otype;
	int	volume;
	char	*data;
	long	datalen;

	arglex_init(argc, argv, argtab);
	switch (arglex())
	{
	case arglex_token_help:
		main_help();
		exit(0);

	case arglex_token_version:
		version();
		exit(0);
	
	default:
		break;
	}
	infile = 0;
	outfile = 0;
	itype = -1;
	otype = -1;
	volume = 0;
	data = 0;
	datalen = 0;
	while (arglex_token != arglex_token_eoln)
	{
		switch (arglex_token)
		{
		default:
			error("misplaced \"%s\" command line argument", arglex_value.alv_string);
			usage();

		case arglex_token_input:
			if (infile)
			{
				duplicate:
				error
				(
					"duplicate \"%s\" option",
					arglex_value.alv_string
				);
				usage();
			}
			if (arglex() != arglex_token_string)
			{
				error("-Input must be followed by a file name");
				usage();
			}
			infile = arglex_value.alv_string;
			break;

		case arglex_token_output:
			if (outfile)
				goto duplicate;
			if (arglex() != arglex_token_string)
			{
				error("-Output must be followed by a file name");
				usage();
			}
			outfile = arglex_value.alv_string;
			break;

		case arglex_token_string:
			{
				char	*cp;
				size_t	cplen;

				cp = arglex_value.alv_string;
				cplen = strlen(cp);
				if (!data)
				{
					datalen = cplen;
					data = mem_alloc(datalen);
					memcpy(data, cp, cplen);
				}
				else
				{
					size_t	newlen;

					newlen = datalen + 1 + cplen;
					mem_change_size(&data, newlen);
					data[datalen] = ' ';
					memcpy(data + datalen + 1, cp, cplen);
					datalen = newlen;
				}
			}
			break;

		case arglex_token_read_english:
		case arglex_token_read_phonemes:
		case arglex_token_read_klatt:
		case arglex_token_read_ulaw:
			if (itype >= 0)
			{
				error("multiple input types specified");
				usage();
			}
			itype = arglex_token;
			break;
			
		case arglex_token_write_phonemes:
		case arglex_token_write_klatt:
		case arglex_token_write_ulaw:
		case arglex_token_write_speaker:
		case arglex_token_write_jack:
			if (otype >= 0)
			{
				error("multiple output types specified");
				usage();
			}
			otype = arglex_token;
			break;

		case arglex_token_volume:
			if (volume)
				goto duplicate;
			if (arglex() != arglex_token_number)
			{
				error("-Volume must be followed by a number");
				usage();
			}
			volume = arglex_value.alv_number;
			if (volume < 1 || volume > 100)
				fatal("volume %d is out of range", volume);
			break;

		case arglex_token_trace:
			if (arglex() != arglex_token_string)
			{
				error("-TRace must be followed by file names");
				usage();
			}
			for (;;)
			{
				trace_enable(arglex_value.alv_string);
				if (arglex() != arglex_token_string)
					break;
			}
			continue;
		}
		arglex();
	}

	/*
	 * make sure the arguments are consistent
	 * and default those not specified
	 */
	if
	(
		(
			otype == arglex_token_write_speaker
		||
			otype == arglex_token_write_jack
		)
	&&
		outfile
	)
		fatal("may not specify an output file with that output type");
	if (itype < 0)
		itype = arglex_token_read_english;
	if (data && infile)
		error("may not specify both a string and an input file");
	if (otype < 0)
	{
		if (outfile)
			otype = arglex_token_write_ulaw;
		else
			otype = arglex_token_write_speaker;
	}
	if (infile && !*infile)
		infile = 0;
	if (outfile && !*outfile)
		outfile = 0;
	if (volume)
	{
		if
		(
			otype != arglex_token_write_speaker
		&&
			otype != arglex_token_write_jack
		)
			fatal("may not specify a volume with that output type");
	}
	if
	(
		(itype - arglex_token_read_english)
	>
		(otype - arglex_token_write_phonemes)
	)
		fatal("input type and output type are incompatible");

	/*
	 * read the input data
	 */
	switch (itype)
	{
	case arglex_token_read_klatt:
		trace(("frame read\n"));
		if (data)
			fatal("may not specify a string with klatt read");
		frame_read(infile, &data, &datalen);
		break;

	case arglex_token_read_ulaw:
		trace(("ulaw read\n"));
		if (data)
			fatal("may not specify a string with ulaw read");
		ulaw_read(infile, &data, &datalen);
		break;

	default:
		trace(("file read\n"));
		if (!data)
			file_read(infile, &data, &datalen);
		break;
	}

	/*
	 * convert english to phonemes
	 */
	if (itype == arglex_token_read_english)
	{
		char	*tmp;
		long	tmplen;

		trace(("english to phonemes\n"));
		english_to_phonemes(data, datalen, &tmp, &tmplen);
		mem_free(data);
		data = tmp;
		datalen = tmplen;
	}

	/*
	 * convert phonemes to klatt
	 */
	if
	(
		itype <= arglex_token_read_phonemes
	&&
		otype > arglex_token_write_phonemes
	)
	{
		char	*tmp;
		long	tmplen;

		trace(("phonemes to klatt\n"));
		phonemes_to_klatt(data, datalen, &tmp, &tmplen);
		mem_free(data);
		data = tmp;
		datalen = tmplen;
	}

	/*
	 * convert klatt to ulaw
	 */
	if
	(
		itype <= arglex_token_read_klatt
	&&
		otype > arglex_token_write_klatt
	)
	{
		char	*tmp;
		long	tmplen;

		trace(("klatt to ulaw\n"));
		klatt_to_ulaw(data, datalen, &tmp, &tmplen);
		mem_free(data);
		data = tmp;
		datalen = tmplen;
	}

	/*
	 * write the data to the output file or device
	 */
	switch (otype)
	{
	case arglex_token_write_klatt:
		trace(("frame write\n"));
		frame_write(outfile, data, datalen);
		break;

	case arglex_token_write_ulaw:
		trace(("ulaw write\n"));
		ulaw_write(outfile, data, datalen);
		break;

	case arglex_token_write_speaker:
		trace(("speaker write\n"));
		if (volume)
			ulaw_volume(volume / 100.);
		ulaw_play(0, data, datalen);
		break;

	case arglex_token_write_jack:
		trace(("jack write\n"));
		if (volume)
			ulaw_volume(volume / 100.);
		ulaw_play(1, data, datalen);
		break;

	default:
		trace(("file write\n"));
		file_write(outfile, data, datalen);
	}

	exit(0);
	return 0;
}
