/*  RipOff - Plugin based graphical CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <fcntl.h>
#include <unistd.h>
#include <cdio/cdio.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include "lib/RipOff.h"
#include "RipOffPreferences.h"
#include "RipOffCore.h"

/* idling function used to monitor the status of the CD drive. Polls every second
   and updates the GtkModel representing the CD if the CD is ejected and/or changed*/
gboolean monitor_cd(gpointer ptr);
void initial_cd_status(RipOffPreferences prefs, RipOffCore core);
gboolean save_settings(gpointer ptr);
gboolean ripoff_drive_not_ready = FALSE;

gint main( gint   argc,
          gchar *argv[])
{
	RipOffCore core;
	g_thread_init(NULL);
        gdk_threads_init();
	gtk_init (&argc, &argv);
	extern GtkWidget *status_bar;
	RipOffPreferences prefs;
	gchar *config_file_dir;
	gchar *home_dir = getenv("HOME");
	
	if(home_dir != NULL)
	{
		config_file_dir = g_strconcat(home_dir, "/.ripoff", NULL);
		mkdir(config_file_dir, 0777);

		free(config_file_dir);

		config_file_dir = g_strconcat(home_dir, "/.ripoff/plugins", NULL);
		mkdir(config_file_dir, 0777);

		free(config_file_dir);
	}
	
	prefs = ripoff_preferences_new();
	status_bar = gtk_statusbar_new();
	core = ripoff_core_new(prefs);
	initial_cd_status(prefs, core);
	monitor_cd(core);
	g_timeout_add(1000, (GSourceFunc) monitor_cd, core);
	g_timeout_add(10000, (GSourceFunc) save_settings, prefs);


	gtk_main();


	return 0;
}

void initial_cd_status(RipOffPreferences prefs, RipOffCore core)
{
	gint drive;
	gint drive2;
	const gchar *device_path;
	gchar *error_message;
	extern GtkWidget *status_bar;

	device_path = ripoff_preferences_get_cd_device_path(prefs);
	drive = open(device_path, O_RDONLY | O_NONBLOCK, S_IRWXO);

	if(drive == -1)
	{
		error_message = g_malloc(strlen(device_path) + strlen(" is not a valid device.")
					 +1);
		strcpy(error_message, device_path);
		strcat(error_message, " is not a valid device.");
		gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 1);
		gtk_statusbar_push(GTK_STATUSBAR(status_bar), 1, error_message);
		g_free(error_message);
	}

	else
	{
	
		drive2 = ioctl(drive, CDROM_DRIVE_STATUS);

		if(drive2 != CDS_DISC_OK)
		{
			ripoff_drive_not_ready = TRUE;
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 1);
			gtk_statusbar_push(GTK_STATUSBAR(status_bar), 1, "No CD inserted");	
		}

		else if(drive2 == CDS_DISC_OK)
		{
			ripoff_core_set_album(core, "Untitled Album");
			ripoff_core_set_artist(core, "Unknown Artist");
			ripoff_core_set_genre(core, "Miscellaneous");
			ripoff_drive_not_ready = FALSE;
			ripoff_core_update_model(core);
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 1);
			gtk_statusbar_push(GTK_STATUSBAR(status_bar), 1, "CD successfully read");
		}

		close(drive);
	}
		
}

gboolean monitor_cd(gpointer ptr)
{
	gint drive;
	gint drive2;
	extern GtkWidget *status_bar;
	const gchar *device_path;
	gchar *error_message;
	RipOffCore core = (RipOffCore) ptr;
	
	device_path = ripoff_preferences_get_cd_device_path(core->prefs);
	drive = open(device_path, O_RDONLY | O_NONBLOCK, S_IRWXO);

	if(drive == -1)
	{
		error_message = g_malloc(strlen(device_path) + strlen(" is not a valid device.")
					 +1);
		strcpy(error_message, device_path);
		strcat(error_message, " is not a valid device.");
		gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 1);
		gtk_statusbar_push(GTK_STATUSBAR(status_bar), 1, error_message);
		g_free(error_message);
		ripoff_core_clear_model((RipOffCore)ptr);
		ripoff_drive_not_ready = TRUE;
	}

	else
	{
		drive2 = ioctl(drive, CDROM_DRIVE_STATUS);

		if(drive2 != CDS_DISC_OK && !ripoff_drive_not_ready)
		{
			ripoff_core_clear_model((RipOffCore)ptr);
			ripoff_core_set_album(core, "");
			ripoff_core_set_artist(core, "");
			ripoff_core_set_year(core, "");
			ripoff_core_set_genre(core, "");
			ripoff_drive_not_ready = TRUE;
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 1);
			gtk_statusbar_push(GTK_STATUSBAR(status_bar), 1, "No CD inserted");	
		}

		else if(drive2 == CDS_DISC_OK && ripoff_drive_not_ready)
		{
			ripoff_core_set_album(core, "Untitled Album");
			ripoff_core_set_artist(core, "Unknown Artist");
			ripoff_core_set_genre(core, "Miscellaneous");
			ripoff_core_update_model((RipOffCore)ptr);
			ripoff_drive_not_ready = FALSE;
			gtk_statusbar_pop(GTK_STATUSBAR(status_bar), 1);
			gtk_statusbar_push(GTK_STATUSBAR(status_bar), 1, "CD successfully read");
		}
	
		
		close(drive);
	}

	return TRUE;	
}

gboolean save_settings(gpointer ptr)
{
	ripoff_preferences_save((RipOffPreferences)ptr);
	return TRUE;
}
