/************************************************************************
 *
 * Copyright (C) 2020-2024 IRCAD France
 * Copyright (C) 2020 IHU Strasbourg
 *
 * This file is part of Sight.
 *
 * Sight is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sight. If not, see <https://www.gnu.org/licenses/>.
 *
 ***********************************************************************/

#pragma once

#include <service/controller.hpp>

#include <QString>

#include <filesystem>

namespace sight::module::ui::qt
{

/**
 * @brief selector of theme/style.
 * Theme should be stored in the rc/ folder of this module.
 *
 * When a new theme is applied on an application, it will be saved in preference file with "THEME" key.
 * style_selector will first load & apply the theme saved in preference file.
 *
 * Theme names are generated by parsing the *.rcc (from *.qrc files) and then name are uppercased (ex: flatdark.rcc
 * becames FLATDARK).
 *
 * To change a theme while application is running please use a preferences_configuration like :
 * @code{.xml}
    <service uid=".." type="sight::module::ui::qt::preferences_configuration">
        <preference>
            <type>combobox</type>
            <name>Color Theme</name>
            <key>THEME</key>
            <values>DEFAULT,FLATDARK</values>
            <default_value>DEFAULT</default_value>
        </preference>
    </service>
 * @endcode
 *
 * Next connect the "parameters_modified" signal to the "update_from_preferences" slot of this service.
 *
 * @section Slots Slots
 * - \b updateFromPrefs(): called when changing the preference key "THEME", apply the selected theme
 * by reading preference file.
 *
 * @section XML XML Configuration
 *
 * @code{.xml}
        <service uid="..." type="sight::module::ui::qt::style_selector" />
   @endcode
 */
class style_selector : public service::controller
{
public:

    SIGHT_DECLARE_SERVICE(style_selector, sight::service::controller);

    /// Constructor, initializes slots.
    style_selector() noexcept;

    /// Destructor, clears the position map.
    ~style_selector() noexcept override;

protected:

    /** @name Service methods ( override from service::base )
     * @{
     */

    /// Does nothing.
    void configuring() override;

    /**
     * @brief Starts and setups the service by parsing rc folder to find availble themes.
     */
    void starting() override;

    /**
     * @brief Does nothing
     */
    void stopping() override;

    /**
     * @brief Does nothing.
     */
    void updating() override;

private:

    /// Applies the style _stylename.
    void change_style(const std::string& _style_name);

    ///Slot: check preference key THEME, and update theme accordingly.
    void update_from_prefs();

    /// Map to stores theme names and their path.
    std::map<std::string, std::filesystem::path> m_style_map;
};

} //namespace sight::module::ui::qt
