use parser;
use core:io;

// Parse a string into JSON.
JsonValue parseJson(Str text) {
	parseJson(text, text.begin);
}

JsonValue parseJson(Str text, Str:Iter start) {
	var result = jsonParser(text, start);
	if (r = result.value) {
		if (result.end != text.end)
			throw JsonParseError("The provided string contains more than one JSON element: ${result.end}");
		return r;
	}
	if (error = result.error)
		throw JsonParseError(error.message);
	throw InternalError("Should not be possible.");
}

// Parse a buffer containing UTF-8 (or ASCII) encoded JSON data.
JsonValue parseJson(Buffer text) {
	parseJson(text, 0);
}

JsonValue parseJson(Buffer text, Nat start) {
	var result = jsonParser(text, start);
	if (r = result.value) {
		if (result.end != text.filled) {
			throw JsonParseError("The provided buffer contains more than one JSON element, starting at offset ${result.end}");
		}
		return r;
	}
	if (error = result.error)
		throw JsonParseError(error.message);
	throw InternalError("Should not be possible.");
}

private:

// The parser itself. Allows parsing both text and binary blobs.
jsonParser : parser(recursive descent, text and binary) {
	start = SRoot;
	delimiter = Delim;

	void Delim();
	Delim : "[ \n\r\t]*";

	JsonValue SRoot();
	SRoot => v : Delim - SValue v - Delim;

	JsonValue SValue();
	SValue => JsonValue() : "null";
	SValue => JsonValue(true) : "true";
	SValue => JsonValue(false) : "false";
	SValue => toNumber(matched) : (SIntNum - SFracExp)@ matched;
	SValue => toString(decoder) : SString decoder;
	SValue => x : "\[", SArray x;
	SValue => x : "{", SObject x;

	void SIntNum();
	SIntNum : "[\-+]?[0-9]+" s;

	void SFracExp();
	SFracExp : "\.[0-9]+" - ("[eE][\-+]?[0-9]+")?;
	SFracExp : "[eE][\-+]?[0-9]+";
	SFracExp : ;

	StringDecoder SString();
	SString => StringDecoder() : "\"" - SStringBody(me) - "\"";

	void SStringBody(StringDecoder decoder);
	SStringBody => decoder : "[^\"\\]+" -> push - SStringBody(decoder);
	SStringBody : "\\" - SStringEscape(decoder) - SStringBody(decoder);
	SStringBody : ;

	void SStringEscape(StringDecoder decoder);
	SStringEscape => decoder : "[\"\\/]" -> push;
	SStringEscape => decoder : "[bfnrt]" -> pushWellKnown;
	SStringEscape => decoder : "u" - "[0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f]" -> pushUtf;

	JsonValue SArray();
	SArray => JsonValue.emptyArray() : "\]";
	SArray => JsonValue.emptyArray() : SValue -> push, (",", SValue -> push,)* - "\]";

	JsonValue SObject();
	SObject => JsonValue.emptyObject() : "}";
	SObject => JsonValue.emptyObject() : SKeyval(me), (",", SKeyval(me),)* - "}";

	void SKeyval(JsonValue to);
	SKeyval => put(to, key, val) : SString key, ":", SValue val;
}

private JsonValue toNumber(Str captured) {
	if (whole = captured.long)
		JsonValue(whole);
	else
		JsonValue(captured.toDouble);
}

private JsonValue toNumber(Buffer captured) {
	toNumber(captured.fromUtf8());
}

private class StringDecoder {
	StrBuf out;

	// First surrogate pair, if any.
	Nat firstSurrogate;

	// Initialize.
	init() {}

	// Add a string.
	void push(Str s) {
		// If we have a partial surrogate pair, then output a ? to indicate that something is wrong.
		if (firstSurrogate != 0) {
			out << "?";
			firstSurrogate = 0;
		}

		out << s;
	}
	void push(Buffer b) {
		push(b.fromUtf8);
	}

	// Add a well-known escaped char.
	void pushWellKnown(Str v) {
		if (v == "b")
			push("\b");
		else if (v == "f")
			push("\f");
		else if (v == "n")
			push("\n");
		else if (v == "r")
			push("\r");
		else if (v == "t")
			push("\t");
	}
	void pushWellKnown(Buffer b) {
		Byte ch = b[0];
		if (ch == 0x62)
			push("\b");
		else if (ch == 0x66)
			push("\f");
		else if (ch == 0x6E)
			push("\n");
		else if (ch == 0x72)
			push("\r");
		else if (ch == 0x74)
			push("\t");
	}

	// Add an UTF-16 codepoint.
	void pushUtf(Str hex) {
		pushUtf(hex.hexToNat());
	}
	void pushUtf(Buffer hex) {
		Nat out = 0;
		for (Nat i = 0; i < hex.filled; i++) {
			Byte ch = hex[i];
			out <<= 4;
			if (ch >= 0x30 & ch <= 0x39)
				out += ch - 0x30;
			else if (ch >= 0x41 & ch <= 0x46)
				out += ch - 0x41 + 0xA;
			else if (ch >= 0x61 & ch <= 0x66)
				out += ch - 0x61 + 0xA;
		}
		pushUtf(out);
	}
	void pushUtf(Nat val) {
		// Try to assemble:
		var assembled = Char:utf16Assemble(firstSurrogate, val);
		if (assembled.codepoint == 0 & val != 0) {
			if (firstSurrogate != 0)
				out << "?";
			firstSurrogate = val;
		} else {
			firstSurrogate = 0;
			out << assembled;
		}
	}

	// Convert to a Json string.
	JsonValue toString() {
		JsonValue(toS);
	}

	// Finalize the string.
	Str toS() {
		out.toS;
	}
}

private void put(JsonValue to, StringDecoder key, JsonValue value) {
	to.put(key.toS, value);
}
